package ksc

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
)

type SrvView service

const (
	apiMethodSrvViewResetIterator   apiMethod = "/api/v1.0/SrvView.ResetIterator"
	apiMethodSrvViewGetRecordCount  apiMethod = "/api/v1.0/SrvView.GetRecordCount"
	apiMethodSrvViewGetRecordRange  apiMethod = "/api/v1.0/SrvView.GetRecordRange"
	apiMethodSrvViewReleaseIterator apiMethod = "/api/v1.0/SrvView.ReleaseIterator"
)

const (
	GlobalUsersListSrvViewName View = "GlobalUsersListSrvViewName"
	HWInvPCSrvViewName         View = "HWInvPCSrvViewName"
	KLPolProfileHST            View = "KLPOL_PROFILE_HST"
)

type View string

type Attributer interface {
	Attribute() string
}

type ViewParams interface {
	viewParams() viewParams
}

type viewParams interface{}

type srvViewIterator struct {
	ID    string   `json:"wstrIteratorId"`
	Error kscError `json:"PxgError"`
}

func (sv *SrvView) ResetIterator(ctx context.Context, viewName View, filter string, fields []Attributer, lifeTime int, params ViewParams) (string, error) {
	id, err := sv.resetIterator(ctx, viewName, filter, fields, lifeTime, params.viewParams())
	if err != nil {
		return id, fmt.Errorf("SrvView::ResetIterator(): %w", err)
	}

	return id, nil
}

func (sv *SrvView) resetIterator(ctx context.Context, viewName View, filter string, fields []Attributer, lifeTime int, params viewParams) (string, error) {
	req := sv.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSrvViewResetIterator.String()

	resp := srvViewIterator{}
	var attributes []string
	for _, field := range fields {
		attributes = append(attributes, field.Attribute())
	}

	reqData := struct {
		ViewName       View       `json:"wstrViewName"`
		Filter         string     `json:"wstrFilter"`
		FieldsToReturn []string   `json:"vecFieldsToReturn"`
		Params         viewParams `json:"pParams"`
		MaxLifeTime    int        `json:"lifetimeSec"`
	}{
		ViewName:       viewName,
		Filter:         filter,
		FieldsToReturn: attributes,
		Params:         params,
		MaxLifeTime:    lifeTime,
	}

	req.Body = reqData

	err := sv.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SrvView::resetIterator(): %w", err)
		return resp.ID, err
	}

	if resp.Error.Code != 0 {
		return resp.ID, fmt.Errorf("SrvView::resetIterator(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.ID, nil
}

func (sv *SrvView) GetRecordCount(ctx context.Context, id string) (int, error) {
	count, err := sv.getRecordCount(ctx, id)
	if err != nil {
		return count, fmt.Errorf("SrvView::GetRecordCount(): %w", err)
	}

	return count, nil
}

func (sv *SrvView) getRecordCount(ctx context.Context, id string) (int, error) {
	req := sv.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSrvViewGetRecordCount.String()
	req.SetHeader("Transfer-Encoding", "chunked")

	resp := struct {
		Count int      `json:"PxgRetVal"`
		Error kscError `json:"PxgError"`
	}{}

	reqData := struct {
		ID string `json:"wstrIteratorId"`
	}{
		ID: id,
	}

	req.Body = reqData

	err := sv.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SrvView::getRecordCount(): %w", err)
		return resp.Count, err
	}

	if resp.Error.Code != 0 {
		return resp.Count, fmt.Errorf("SrvView::getRecordCount(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.Count, nil
}

func (sv *SrvView) GetRecordRange(ctx context.Context, id string, rangeBegin int, rangeEnd int, out interface{}) error {
	err := sv.getRecordRange(ctx, id, rangeBegin, rangeEnd, out)
	if err != nil {
		return fmt.Errorf("SrvView::GetRecordRange(): %w", err)
	}

	return nil
}

func (sv *SrvView) getRecordRange(ctx context.Context, id string, rangeBegin int, rangeEnd int, out interface{}) error {
	req := sv.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSrvViewGetRecordRange.String()
	req.SetHeader("Transfer-Encoding", "chunked")

	reqData := struct {
		IteratorID string `json:"wstrIteratorId"`
		RangeBegin int    `json:"nStart"`
		RangeEnd   int    `json:"nEnd"`
	}{
		IteratorID: id,
		RangeBegin: rangeBegin,
		RangeEnd:   rangeEnd,
	}

	resp := struct {
		Chunk struct {
			Array []struct {
				Type  string          `json:"type"`
				Value json.RawMessage `json:"value"`
			} `json:"KLCSP_ITERATOR_ARRAY"`
		} `json:"pRecords"`
		Error kscError `json:"PxgError"`
	}{}

	req.Body = reqData

	err := sv.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SrvView::getRecordRange(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("SrvView::getRecordRange(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	switch outType := out.(type) {
	case *[]GlobalUsersListView:
		var view globalUsersListView
		for _, item := range resp.Chunk.Array {
			err = json.Unmarshal(item.Value, &view)
			if err != nil {
				return fmt.Errorf("SrvView::getRecordRange(): unmarshal data: %w", err)
			}
			userInfo, err := view.toGlobalUsersListView()
			if err != nil {
				return fmt.Errorf("SrvView::getRecordRange(): cast data: %w", err)
			}
			*outType = append(*outType, userInfo)
		}
	case *[]HwInvPCView:
		var view hwInvPCView
		for _, item := range resp.Chunk.Array {
			err = json.Unmarshal(item.Value, &view)
			if err != nil {
				return fmt.Errorf("SrvView::getRecordRange(): unmarshal data: %w", err)
			}
			hwInfo, err := view.toHwInvPCView()
			if err != nil {
				return fmt.Errorf("SrvView::getRecordRange(): cast data: %w", err)
			}
			*outType = append(*outType, hwInfo)
		}
	case *[]KLPolProfileHSTView:
		var view klPolProfileHSTView
		for _, item := range resp.Chunk.Array {
			err = json.Unmarshal(item.Value, &view)
			if err != nil {
				return fmt.Errorf("SrvView::getRecordRange(): unmarshal data: %w", err)
			}
			policyInfo, err := view.toKLPolProfileHSTView()
			if err != nil {
				return fmt.Errorf("SrvView::getRecordRange(): cast data: %w", err)
			}
			*outType = append(*outType, policyInfo)
		}
	default:
		return fmt.Errorf("SrvView::getRecordRange(): unsupported type %T", outType)
	}

	return nil
}

func (sv *SrvView) ReleaseIterator(ctx context.Context, id string) error {
	err := sv.releaseIterator(ctx, id)
	if err != nil {
		return fmt.Errorf("SrvView::ReleaseIterator(): %w", err)
	}

	return nil
}

func (sv *SrvView) releaseIterator(ctx context.Context, id string) error {
	req := sv.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSrvViewReleaseIterator.String()

	reqData := struct {
		ID string `json:"wstrIteratorId"`
	}{id}

	req.Body = reqData

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}

	err := sv.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SrvView::releaseIterator(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("SrvView::releaseIterator(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}
