package ksc

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
)

type SsContents service

const (
	apiMethodSsContentsSsRead     apiMethod = "/api/v1.0/SsContents.Ss_Read"
	apiMethodSsContentsSsRelease  apiMethod = "/api/v1.0/SsContents.Ss_Release"
	apiMethodSsContentsSsGetNames apiMethod = "/api/v1.0/SsContents.SS_GetNames"
)

// Products
// product name string, string, not longer than 31 character, and cannot contain characters /\:*?"<>
const (
	KLProductEmpty = "" // empty string means that list of products is needed
	KLProductKES   = "KES"
	KLProductKSC   = "1093"
)

// Versions
// version string, string, not longer than 31 character, and cannot contain characters /\:*?"<>
const (
	KLProductVersionEmpty = "" // empty string means that list of versions is needed
	KLProductVersion11    = "11.0.0.0"
)

func (ss *SsContents) SsGetNames(ctx context.Context, id string, product string, version string) ([]string, error) {
	params, err := ss.ssGetNames(ctx, id, product, version)
	if err != nil {
		err = fmt.Errorf("SsContents::SsGetNames(): %w", err)
	}
	return params, err
}

func (ss *SsContents) ssGetNames(ctx context.Context, id string, product string, version string) ([]string, error) {
	req := ss.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSsContentsSsGetNames.String()

	reqData := struct {
		ID      string `json:"wstrID"`
		Product string `json:"wstrProduct"`
		Version string `json:"wstrVersion"`
	}{
		ID:      id,
		Product: product,
		Version: version,
	}

	resp := struct {
		Params []string `json:"PxgRetVal"`
		Error  kscError `json:"PxgError"`
	}{}

	req.Body = reqData

	err := ss.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SsContents::ssGetNames(): %w", err)
		return resp.Params, err
	}

	if resp.Error.Code != 0 {
		return resp.Params, fmt.Errorf("SsContents::ssGetNames(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return resp.Params, nil
}

func (ss *SsContents) SsRead(ctx context.Context, id string, product string, version string, sectionName string, out interface{}) error {
	raw, err := ss.ssRead(ctx, id, product, version, sectionName)
	if err != nil {
		err = fmt.Errorf("SsContents::SsRead(): %w", err)
		return err
	}

	switch outType := out.(type) {
	case *KesPolicySectionExclusions:
		var section kesPolicySectionExclusions
		err = json.Unmarshal(raw, &section)
		if err != nil {
			return fmt.Errorf("SsContents::ReadContents(): unmarshal data: %w", err)
		}
		*outType, err = section.toKesPolicySectionExclusions()
		if err != nil {
			return fmt.Errorf("SsContents::ReadContents(): cast data: %w", err)
		}
	default:
		err = fmt.Errorf("SsContents::ReadContents(): unsupported type %T", outType)
	}

	return err
}

func (ss *SsContents) ssRead(ctx context.Context, id string, product string, version string, sectionName string) (raw json.RawMessage, err error) {
	req := ss.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSsContentsSsRead.String()

	reqData := struct {
		ID      string `json:"wstrID"`
		Product string `json:"wstrProduct"`
		Version string `json:"wstrVersion"`
		Section string `json:"wstrSection"`
	}{
		ID:      id,
		Product: product,
		Version: version,
		Section: sectionName,
	}

	req.Body = reqData

	resp := struct {
		Section json.RawMessage `json:"PxgRetVal"`
		Error   kscError        `json:"PxgError"`
	}{}

	err = ss.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SsContents::ssRead(): %w", err)
		return
	}

	if resp.Error.Code != 0 {
		err = fmt.Errorf("SsContents::ssRead(): code %d: %s", resp.Error.Code, resp.Error.Message)
		return
	}

	return resp.Section, nil
}

func (ss *SsContents) SsRelease(ctx context.Context, id string) error {
	err := ss.ssRelease(ctx, id)
	if err != nil {
		err = fmt.Errorf("SsContents::SsRelease(): %w", err)
	}
	return err
}

func (ss *SsContents) ssRelease(ctx context.Context, id string) error {
	req := ss.client.client.R()
	req.Method = http.MethodPost
	req.URL = apiMethodSsContentsSsRelease.String()

	reqData := struct {
		ID string `json:"wstrID"`
	}{
		ID: id,
	}

	resp := struct {
		Error kscError `json:"PxgError"`
	}{}

	req.Body = reqData

	err := ss.client.do(ctx, req, &resp)
	if err != nil {
		err = fmt.Errorf("SsContents::ssRelease(): %w", err)
		return err
	}

	if resp.Error.Code != 0 {
		return fmt.Errorf("SsContents::ssRelease(): code %d: %s", resp.Error.Code, resp.Error.Message)
	}

	return nil
}
