namespace java ru.yandex.iss.thrift
namespace py iss_thrift3

/* Just a description useful for write-operations */
struct AuthorSignature {
    /* Author identifier (free form) */
    10: required string author,
    /* Change description (free form) */
    20: required string message = ""
}

/* Configuration Identifier in <pre>family#randomid</pre> format */
typedef string ConfigurationId

/* Instance identifier (unique only in Configuration scope) also known as Slot. Format:  <pre>service_id@host.fqdn.tld</pre> */
typedef string Slot

/* Full instance identifier: Slot + Configuration identifier */
struct InstanceId {
    /* slot definition. Format is <pre>service_id@host.fqdn.tld</pre> */
    10: required Slot slot,
    /* Configuration id as String with format family#contenthash*/
    20: required ConfigurationId configuration
}

/* Configuration meta information */
struct ConfigurationDescription {
    /* Configuration id as String with format family#contenthash*/
    10: required ConfigurationId id,
    /* Parent configuration id. May be not defined when configuration has no parent */
    20: optional ConfigurationId parent,
    /* Configuration creation timestamp (Cacher time) */
    30: required i64 timestamp,
    /* Configuration author signature */
    40: required AuthorSignature signature,

    50: required bool isApplicable;
}

/* Everything required for target state change: query to select instances from some configurations and target state wanted to be applied */
struct TargetStateTransition {
    /* List of instance filters defined as string representations of queries.
     * Required grammar is new one.
     *  Query example: <pre>((configuration.id == "validation#f1bdbb2e") && (!instance[test-prop]))</pre>
     * */
    10: required list<string> filters,

    /* New target state for selected instances. May be REMOVED, PREPARED or ACTIVE */
    20: required string state,
}

/* List of instances and target state wanted to be applied */
struct InstanceListTargetStateTransition {
    /* List of instance identifiers to apply new state */
    10: required list<InstanceId> instances,

    /* New target state for selected instances. May be REMOVED, PREPARED or ACTIVE */
    20: required string state,
}

/* Description of some target state change previously made
* TODO: not used after https://jira.yandex-team.ru/browse/ISS-782 , should be used again in ISS-524
*
*/
struct DetailedTargetState {
    /* Transition descriptio */
    10: required TargetStateTransition transition,
    /* Target state application timestamp (Cacher time) */
    20: required i64 timestamp,
    /* Transition author signature */
    30: required AuthorSignature signature
}

/* Agent status report */
struct DetailedCurrentState {
    /* Report acceptance timestamp (Cacher time) */
    10: required i64 timestamp
    /* Current state reported by agent (free-form) */
    20: required string state,
    /* Some properties related to report (free-form)*/
    30: required map<string, string> properties,
}

/* Representation of unbound instance (not related to any configuration) */
struct Instance {
    /* slot definition. Format is <pre>service_id@host.fqdn.tld</pre> */
    10: required Slot slot,
    /* Instance properties (free form) */
    20: required map<string, string> properties
}


/* Full instance state: Slot, Configuration Identifier, target state and current state */
struct InstanceState {
    /* Instance identifier */
    10: required InstanceId id,
    /* Target state of instance. May be REMOVED, PREPARED or ACTIVE */
    20: required string targetState,
    /* Current state of instance. Free-form. */
    30: required string feedbackState
}

/* Extended full instance state: Slot, Configuration Identifier, target state and current state and metadata */
struct DetailedInstanceState {
    /* Instance identifier */
    10: required InstanceId id,
    /* Target state value */
    20: required string target,
    /* Agent status report */
    30: required DetailedCurrentState feedback,
}

/* Representation of bound instance (Instance of concrete Configuration) */
struct InstanceProperties {
    /* Instance identifier */
    10: required InstanceId id,
    30: required map<string, string> properties
}

struct HostConfigurationInstance {
    10: required InstanceId id,
    20: required string targetState,
    30: required map<string, string> properties,
    40: optional string stateOperationId,

}

struct Version {
    /** Version from artifactory */
    10: required string version,
    /** Compilation timestamp */
    20: required string timestamp
}

/* Identifier of Cacher instance request was handled by */
struct ServiceSignature {
    10: required string hostFQDN
}

/* Offset+limit for requests returning long lists of results */
struct Portion {
    10: i32 offset = 0,
    20: i32 limit = 100
}

/* Just a timeframe with begin and end in unix timestamp format */
struct TimeFrame {
    10: i64 beginTimestamp = -1,
    20: i64 endTimestamp = -1,
}

/* This exception will be thrown when incorrect parameters was passed to some method */
exception InvalidRequestException {
    10: required string message,
    20: string stacktrace,
    /* Identifier of cacher exception happend on */
    30: required ServiceSignature signature
}

/* Exception to be thrown when some entity was not found by Cacher (configuration, bracnh, etc) */
exception NotFoundException {
    10: required string message,
    20: string stacktrace,
    /* Identifier of cacher exception happend on */
    30: required ServiceSignature signature
}

/* Error code for SystemException */
enum ErrorCode {
    /* Not implemented yet */
    NOT_IMPLEMENTED = 1,

    /** Cassandra DB is not accessible */
    CASSANDRA_CONNECTION_BROKEN = 2,

    /** NPE and other unpredictable behavior causes this error code */
    INTERNAL_ERROR = 4,
}

/* This exception will be thrown by any failed request without specific handler */
exception SystemException {
    10: required string message,
    20: required ErrorCode errorCode,
    /* Identifier of cacher exception happend on */
    30: required ServiceSignature signature,
    40: string stacktrace
}

/* Metadata of new configuration and Cacher identifier */
struct CreateConfigurationResponse {
    /* Metadata of new configuration */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Metadata of new configuration and Cacher identifier */
struct CopyAndModifyConfigurationFromExternalResponse {
    /* Metadata of new configuration */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Metadata of new configuration and Cacher identifier */
struct CopyAndModifyConfigurationResponse {
    /* Metadata of new configuration */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Url of exported configuration and Cacher identifier */
struct ExportConfigurationResponse {
    /* URL containing exported configuration */
    10: required string url,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* TODO */
struct ValidationMessage {
   /* */
   1: required string type,
   /* */
   2: required string ruleName,
   /* */
   3: required string description,
   /* */
   4: required i64 timestamp
}

/* TODO */
struct ValidationReport {
   /* */
   1: required list<ValidationMessage> messages,
   /* */
   2: required list<string> rules,
   /* */
   3: required map<string,i64> rulesExecutionDuration
}

/* Validation result and Cacher identifier */
struct ValidateResponse {
    10: required ValidationReport report,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Configuration metadata and Cacher identifier */
struct GetConfigurationDescriptionResponse {
    /* Configuration metadata */
    10: required ConfigurationDescription configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct SetBranchHeadResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct RemoveBranchHeadResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Configuration identifier of requested branch head and Cacher identifier */
struct GetBranchHeadResponse {
    /* Configuration id as String with format family#contenthash*/
    10: required ConfigurationId configuration,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Set of configuration identifiers of requested branch heads and Cacher identifier */
struct GetBranchHeadsResponse {
    /* Set of configuration identifiers as Strings with format family#contenthash*/
    10: required set<ConfigurationId> configurations,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Not used for now. Remove it?*/
/*
struct TransitionError {
    10: optional Slot slot,
    20: optional string filter,
    30: optional string configurationId,
    40: optional string error,
    50: optional ErrorCode errorCode,
    60: optional string stacktrace
}
*/

/* Per-configuration sets of instances affected by target state change operation */
struct ApplyResponse {
    /* Keys of this map are identifiers of configurations affected by request. Values are sets of slots in this configuration
    * which state was changed*/
    10: required map<ConfigurationId, set<Slot>> affected,
    /*Abstract configaratuions that cannot be applied*/
    20: required list<ConfigurationId> nonApplicable,
    /* Identifier of cacher request was processed on */
    30: required ServiceSignature signature
}

/* Instance state (target+current) and Cacher identifier */
struct GetInstanceStateResponse {
    /*  Instance state (target and current)*/
    10: required InstanceState instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Instance states (target+current) and Cacher identifier */
struct GetStateOfHostsResponse {
    /*  Instance states (target and current)*/
    10: required list<InstanceState> instances,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Extended instance state (target+current+metatada) and Cacher identifier */
struct GetDetailedInstanceStateResponse {
    /* Extended instance state (target and current and metadata)*/
    10: required DetailedInstanceState instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Extended instance states (target+current+metatada) and Cacher identifier */
struct GetDetailedStateOfHostsResponse {
    /* Extended instance states (target and current and metadata)*/
    10: required list<DetailedInstanceState> instances,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Instance properties (as string-to-string map)*/
struct GetInstancePropertiesResponse {
    /* Instance properties */
    10: required InstanceProperties instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Result of filtering instances in some configuration with some query */
struct QueryResult {
    /* Configuration which instances was filtered */
    10: required ConfigurationId configuration,
    /* List of slots of instances which fullfills query criterias */
    20: required list<Slot> instances
}

/* List of per-configuration results of some instances query */
struct QueryResponse {
    /* Per-configuration query results */
    10: required list<QueryResult> result,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}


/* Full configuration representation. */
struct GetConfigurationResponse {
    /* Instances of this configuration. This list may be paged by GetConfigurationResponse method (see method description for details) */
    10: required list<Instance> instances,
    /* List of mutations which was used to get this configuration. Empty for root configuration (without parent) */
    20: required list<string> mutations,
    /* Configuration Identifier in <pre>family#randomid</pre> format */
    30: required ConfigurationId id,
    /* Parent Configuration Identifier in <pre>family#randomid</pre> format. May be undefined. */
    40: optional ConfigurationId parentId,
    /* Identifier of cacher request was processed on */
    50: required ServiceSignature signature,
}

/* Cacher version */
struct VersionResponse {
    /* Cacher version as string */
    10: required Version version,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}

/* Configuration of any concrete hosts. */
struct GetHostConfigurationResponse {
    /* List of all the bound instances (slot + configuration identifier) related to requested host which are in ACTIVE or PREPARED state */
    10: required list<HostConfigurationInstance> instances,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}

/* Configuration of any concrete hosts. */
struct GetSlotConfigurationResponse {
    /* List of all the bound instances (slot + configuration identifier) related to requested host which are in ACTIVE or PREPARED state */
    10: required HostConfigurationInstance instance,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}

/* Configuration of any concrete hosts. */
struct GetSlotsConfigurationResponse {
    /* List of all the bound instances (slot + configuration identifier) related to requested host which are in ACTIVE or PREPARED state */
    10: required list<HostConfigurationInstance> instances,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature,
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct SaveFeedbackResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

/* Rendered MVEL template and Cacher identifier*/
struct RenderTemplateResponse {
    /* MVEL template rendered in Cacher context */
    10: required string result,
    /* Identifier of cacher request was processed on */
    20: required ServiceSignature signature
}


/* Offset+limit for requests returning long lists of results */
struct Metric {
    10: optional i64 longValue,
    20: optional string stringValue,
    30: optional list<string> stringListValue,
}

/* Contains only Cacher identifier. Obtaining instance of this class means request success. */
struct MetricResponse {
    /* Identifier of cacher request was processed on */
    10: required ServiceSignature signature
}

struct SecurityToken {
    10: required string user,
    // timestamp of request
    20: required i64 timestamp,
    // request hash with digital signature TODO: wiki
    30: required binary digest
}

enum ConsistencyLevel {
    /* rely on keyspace/colmn family defaults */
    DEFAULT = 0

    ONE = 10,
    QUORUM = 20,
    LOCAL_QUORUM = 30,
    EACH_QUORUM = 40,
    ALL = 50,
    ANY = 60,
    TWO = 70,
    THREE = 80,
}

service IssService {
    /* create new configuration from external dump */
    CreateConfigurationResponse
        createConfigurationFromExternal(10: required string family,
                                        20: required string dumpURL,
                                        30: required string dumpFormat,
                                        40: required AuthorSignature signature,
                                        50: required bool isApplicable)
        throws (10:InvalidRequestException ire,
                20:SystemException se),

    /* copy and modify configuration from external dump */
    CopyAndModifyConfigurationFromExternalResponse
        copyAndModifyConfigurationFromExternal(10: required ConfigurationId parent,
                                               20: required string modifierURL,
                                               30: required string modifierFormat,
                                               40: required AuthorSignature signature,
                                               50: required bool isApplicable)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* create configuration from given instances */
    CreateConfigurationResponse
        createConfiguration(10: required string family
                            20: required list<Instance> instances,
                            30: required AuthorSignature signature,
                            40: required bool isApplicable)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* create configuration from existing by applying mutation rules */
    CopyAndModifyConfigurationResponse
        copyAndModifyConfiguration(10: required ConfigurationId parent,
                                   20: required string mutation,
                                   30: required AuthorSignature signature,
                                   40: required bool isApplicable)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Set given configuration as head of given branch (in configuration's family) */
    SetBranchHeadResponse
        setBranchHead(10: required ConfigurationId configuration,
                      20: required string branch,
                      40: required AuthorSignature signature)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Remove configuration (if any) set as head of given branch (in configuration's family) */
    RemoveBranchHeadResponse
        removeBranchHead(10: required string family,
                         20: required string branch,
                         30: required AuthorSignature signature)
        throws (10:InvalidRequestException ire,
                        20:NotFoundException nfe,
                        40:SystemException se),

    /* Apply new target state to set of instances defined by given query */
    ApplyResponse
        apply(10: required TargetStateTransition transition,
              20: required AuthorSignature signature,
              30: optional ConsistencyLevel consistencyLevel = ConsistencyLevel.DEFAULT,
              40: optional SecurityToken token)
        throws (10:InvalidRequestException ire,
                40:SystemException se),

    /* Apply new target state to set of instances defined by given query */
    ApplyResponse
        applyToInstances(10: required InstanceListTargetStateTransition transition,
              20: required AuthorSignature signature,
              30: optional ConsistencyLevel consistencyLevel = ConsistencyLevel.DEFAULT,
              40: optional SecurityToken token)
        throws (10:InvalidRequestException ire,
                40:SystemException se),

    /* Save current state report. This method inteded to be used by Agent. */
    SaveFeedbackResponse
        saveFeedback(10: required InstanceId instance,
                     20: required DetailedCurrentState feedbackMessage)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Get all the data related to given hostname. Such data must include any instances in PREPARED or ACTIVE state with host part of slot equal to given one.
     This method inteded to be used by Agent. */
    GetHostConfigurationResponse
        getHostConfiguration(10: required string hostFQDN)
        throws (10:InvalidRequestException ire,
                40:SystemException se),

    /* TODO */
    ValidateResponse
        validateFromExternal(10: required list<string> ruleURLs,
                             20: map<string, string> bindings)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),
    /* TODO */
    ValidateResponse
            validate(10: required list<string> rules,
                     20: map<string, string> bindings)
            throws (10:InvalidRequestException ire,
                    20:NotFoundException nfe,
                    40:SystemException se),

    /* Renders any MVEL template in context of Cacher API */
    RenderTemplateResponse
        renderTemplate(10: required string template,
                       20: required InstanceId instanceId)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se)

    /* Load configuration metatada (parent identifier, creation timestamp, etc) */
    GetConfigurationDescriptionResponse
        getConfigurationDescription(10: required ConfigurationId configuration)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Load all the configuration content. <b>>WARNING</b>: result may contain large amount of data (really large), so method
     is paged and should not be used frequently */
    GetConfigurationResponse
            getConfiguration(10: required ConfigurationId configuration
                             20: required Portion portion)
            throws (10:InvalidRequestException ire,
                    20:NotFoundException nfe,
                    40:SystemException se),

    /* Get configuration identifier set as head of given branch in given family */
    GetBranchHeadResponse
        getBranchHead(10: required string family
                      20: required string branch)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Get all the heads of given branches in all the configuration families */
    GetBranchHeadsResponse
        getBranchHeads(10: required string branch)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Perform instances query. Returns per-configuration lists of instances. Result is paged. */
    QueryResponse
        query(10: required string filter,
              20: required Portion portion)
        throws (10:InvalidRequestException ire,
                40:SystemException se),

    /* Get state (targe+current) of given instance in given configuration */
    GetInstanceStateResponse
        getInstanceState(10: required InstanceId instanceId)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Get state (target+current) of instances on given hosts */
    GetStateOfHostsResponse
        getStateOfHosts(10: required list<string> hosts)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /*Get state of given slot */
    GetSlotConfigurationResponse
        getSlotConfiguration(10: required Slot slot)
        throws (10:InvalidRequestException ire,
                40:SystemException se),

    /* Get state of given slots */
    GetSlotsConfigurationResponse
        getSlotsConfiguration(10: required list<Slot> slots)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Get extended state (targe+current+medatata) of given instance in given configuration */
    GetDetailedInstanceStateResponse
        getDetailedInstanceState(10: required InstanceId instanceId)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Get extended state (targe+current+medatata) of given instances */
    GetDetailedStateOfHostsResponse
        getDetailedStateOfHosts(10: required list<string> hosts)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Get properties of given instance in given configuration */
    GetInstancePropertiesResponse
        getInstanceProperties(10: required InstanceId instanceId)
        throws (10:InvalidRequestException ire,
                20:NotFoundException nfe,
                40:SystemException se),

    /* Cacher version */
    VersionResponse
        getVersion()
        throws (10:SystemException se),

    /* Cacher version */
    MetricResponse
        recordMetric(10: required string metricName,
                     20: required string metricType,
                     30: required Metric metric)
        throws (10:SystemException se),
}

