package controller

import (
	"a.yandex-team.ru/infra/alert_controller/internal/config"
	"a.yandex-team.ru/infra/alert_controller/internal/service"
	"a.yandex-team.ru/infra/alert_controller/internal/solomon"
	"a.yandex-team.ru/infra/alert_controller/internal/unistat"
	"a.yandex-team.ru/infra/alert_controller/internal/util"
	"a.yandex-team.ru/infra/alert_controller/internal/yp"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"context"
	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"
	"time"
)

type AppConfig struct {
	IterationInterval     int    `yaml:"iteration_interval_s"`
	AlertsTestProject     string `yaml:"alerts_test_project"`
	YPCluster             string `yaml:"yp_cluster"`
	UsePublishedTemplates bool   `yaml:"use_published_templates"`
	ForceReCreateAlerts   bool   `yaml:"force_re_create_alerts"`
}

func NewApp(logger *zap.Logger, config AppConfig, metrics *unistat.Stats) (*App, error) {
	// Init YP client
	ypClient, err := yp.NewYpClient(config.YPCluster)
	if err != nil {
		panic(err)
	}
	// Init solomon API client
	solomonClient, err := solomon.NewSolomonClient(logger)
	if err != nil {
		panic(err)
	}

	return &App{
		logger:         logger,
		config:         config,
		iterationIndex: 0,
		metrics:        metrics,
		ypClient:       ypClient,
		solomonClient:  solomonClient,
	}, nil
}

func NewGenericApp(logger *zap.Logger, configPtr interface{}, metrics *unistat.Stats) (util.App, error) {
	var appConfig AppConfig
	data, err := yaml.Marshal(configPtr)
	if err != nil {
		panic(err)
	}
	err = yaml.Unmarshal(data, &appConfig)
	if err != nil {
		panic(err)
	}
	return NewApp(logger, appConfig, metrics)
}

type App struct {
	logger         *zap.Logger
	config         AppConfig
	iterationIndex int
	syncedServices map[string]service.Service
	metrics        *unistat.Stats
	ypClient       *yp.Client
	solomonClient  *solomon.Client
}

func (a *App) Run(ctx context.Context) error {
	for {
		err := a.runAlertSync(ctx)
		if err == nil {
			a.logger.Info("iteration successful")
		} else {
			a.logger.Error("iteration finished with error", log.Error(err))
			a.metrics.SyncErrors.Update(1)
		}

		err = a.updateCustomMetric(ctx)
		if err != nil {
			a.logger.Error("test alerts stats are not updated", log.Error(err))
			a.metrics.SyncErrors.Update(1)
		}

		sleep := make(chan struct{}, 1)
		go func() {
			time.Sleep(time.Duration(a.config.IterationInterval) * time.Second)
			sleep <- struct{}{}
		}()

		select {
		case <-sleep:
			continue
		case <-ctx.Done():
			a.logger.Info("exiting app")
			return nil
		}
	}
}

func (a *App) runAlertSync(ctx context.Context) error {
	defer a.metrics.SyncCount.Update(1)

	a.logger.Info("start new iteration")

	projects := a.ypClient.FetchAllProjects(ctx)
	a.logger.Infof("total projects: %d", len(projects))

	stages := a.ypClient.FetchAllStages(ctx)
	a.logger.Infof("total stages: %d", len(stages))

	serviceManager := service.NewManager(a.solomonClient, a.logger, a.metrics)
	currentServices := serviceManager.ListDeployService(ctx, projects, stages)
	a.logger.Infof("total services: %d", len(currentServices))

	publishedTemplates, err := a.solomonClient.SelectTemplate(ctx)
	var templates []solomon.TemplateInfo
	if err != nil {
		templates = solomon.LoadSavedTemplates()
		a.logger.With(log.Error(err)).Error("error during published templates load, use fallback")
	} else {
		templates = solomon.ConvertFromSolomonTemplates(publishedTemplates)
	}

	if !a.config.UsePublishedTemplates {
		templates = solomon.LoadSavedTemplates()
		a.logger.Info("use local templates")
	}

	serviceManager.SyncAlerts(ctx, a.syncedServices, currentServices, templates, a.config.ForceReCreateAlerts)

	a.syncedServices = currentServices

	return nil
}

func (a *App) updateCustomMetric(ctx context.Context) error {
	resp, _, err := a.solomonClient.AlertingApi.GetProjectStatsUsingGET(ctx, a.config.AlertsTestProject)
	if err != nil {
		return err
	}
	a.metrics.TestAlertsWarning.Update(float64(resp.EvaluationSummary.Alarm))
	a.metrics.TestAlertsAlarm.Update(float64(resp.EvaluationSummary.Warn))
	a.metrics.TestAlertsTotal.Update(float64(resp.AlertsCount))
	return nil
}

func StartController() *cobra.Command {
	return &cobra.Command{
		Use:   "controller",
		Short: "Start alert controller",
		Long:  `Start alert controller`,
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			logger := zap.Must(zap.NewProductionDeployConfig())

			fullConfig, err := config.ReadConfig(args[0])
			if err != nil {
				logger.Fatal("failed to parse config", log.Error(err))
			}

			bootstrap, err := util.NewBootstrap(logger, fullConfig, NewGenericApp)
			if err != nil {
				logger.Fatal("failed to create bootstrap", log.Error(err))
			}

			ctx, cancel := context.WithCancel(context.Background())

			go util.HandleSignals(logger, cancel)

			err = bootstrap.Run(ctx)
			logger.Fatal("bootstrap exited", log.Error(err))
		},
	}
}
