package export

import (
	"a.yandex-team.ru/infra/alert_controller/internal/abc"
	"a.yandex-team.ru/infra/alert_controller/internal/config"
	"a.yandex-team.ru/infra/alert_controller/internal/unistat"
	"a.yandex-team.ru/infra/alert_controller/internal/util"
	"a.yandex-team.ru/infra/alert_controller/internal/yp"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"context"
	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"
	"os"
	"strconv"
	"time"
)

const (
	resourceType    = "deploy_unit"
	abcPrefix       = "abc:service:"
	serviceProvider = "deploy"
)

type AppConfig struct {
	IterationInterval int    `yaml:"iteration_interval_s"`
	YPCluster         string `yaml:"yp_cluster"`
	YTCluster         string `yaml:"yt_cluster"`
	TablePath         string `yaml:"table_path"`
}

func NewApp(logger *zap.Logger, config AppConfig, metrics *unistat.Stats) (*App, error) {
	abcToken, found := os.LookupEnv("ABC_TOKEN")
	if !found {
		panic("no abc token")
	}

	abcClientConfig := &abc.ClientConfig{
		OauthToken: abcToken,
	}

	ypc, err := yp.NewYpClient(config.YPCluster)
	if err != nil {
		panic(err)
	}

	clientConfig := util.YTClientConfig{
		Proxy: config.YTCluster,
	}

	ytToken, found := os.LookupEnv("YT_EXPORT_TOKEN")
	if !found {
		panic("no yt token")
	}

	ytc, err := util.NewYTClient(logger, clientConfig, ytToken)
	if err != nil {
		panic(err)
	}

	logger.Info("app has been created")

	return &App{
		logger:         logger,
		config:         config,
		iterationIndex: 0,
		cluster:        config.YPCluster,
		metrics:        metrics,
		abcClient:      abc.NewClient(abcClientConfig),
		ytClient:       ytc,
		ypClient:       ypc,
	}, nil
}

func NewGenericApp(logger *zap.Logger, configPtr interface{}, metrics *unistat.Stats) (util.App, error) {
	var appConfig AppConfig
	data, err := yaml.Marshal(configPtr)
	if err != nil {
		panic(err)
	}
	err = yaml.Unmarshal(data, &appConfig)
	if err != nil {
		panic(err)
	}
	return NewApp(logger, appConfig, metrics)
}

type App struct {
	logger         *zap.Logger
	config         AppConfig
	cluster        string
	iterationIndex int
	metrics        *unistat.Stats
	abcClient      *abc.Client
	ypClient       *yp.Client
	ytClient       yt.Client
}

type ServiceInfo struct {
	ServiceProviderID string            `yson:"service_provider_id"`
	ResourceID        map[string]string `yson:"resource_id"`
	ResourceType      string            `yson:"resource_type"`
	Environment       string            `yson:"environment"`
	AbcID             int64             `yson:"abc_id,omitempty"`
	AbcSlug           string            `yson:"abc_slug,omitempty"`
	Responsible       string            `yson:"responsible"`
}

func convertAbcID(abc string) int64 {
	if abc == "tmp" {
		return 0
	}

	if len(abc) > len(abcPrefix) {
		num, err := strconv.ParseInt(abc[len(abcPrefix):], 10, 64)
		if err != nil {
			panic(err)
		}

		return num
	}

	return 0
}

func convertEnvironment(environment ypapi.TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment) string {
	switch environment {
	case ypapi.TStageSpec_TDeployUnitSettings_UNKNOWN, ypapi.TStageSpec_TDeployUnitSettings_STABLE: // TODO: fix after YP release
		return "PRODUCTION"
	case ypapi.TStageSpec_TDeployUnitSettings_TESTING:
		return "TESTING"
	case ypapi.TStageSpec_TDeployUnitSettings_PRESTABLE:
		return "PRESTABLE"
	}
	return "UNKNOWN"
}

func getOutputTableSchema() schema.Schema {
	tableSchema, err := schema.Infer(&ServiceInfo{})
	if err != nil {
		panic(err)
	}
	return tableSchema
}

func (a *App) dumpToYT(ctx context.Context, outputData []ServiceInfo) {
	tableSchema := getOutputTableSchema()
	tablePath := ypath.Path(a.config.TablePath)
	_, err := yt.CreateTable(ctx, a.ytClient, tablePath, yt.WithSchema(tableSchema), yt.WithForce())
	if err != nil {
		panic(err)
	}

	writer, err := a.ytClient.WriteTable(ctx, tablePath, nil)
	if err != nil {
		panic(err)
	}

	for _, outputData := range outputData {
		if err = writer.Write(outputData); err != nil {
			panic(err)
		}
	}

	if err = writer.Commit(); err != nil {
		panic(err)
	}

	a.logger.Infof("inserted rows: %d", len(outputData))
}

func (a *App) syncAbcResources(ctx context.Context) error {
	projects := a.ypClient.FetchAllProjects(ctx)
	stages := a.ypClient.FetchAllStages(ctx)

	projectsMap := make(map[string]*ypapi.TProject)
	for _, project := range projects {
		projectsMap[project.GetMeta().GetId()] = project
	}

	a.logger.Infof("discovered %d stages and %d projects", len(stages), len(projectsMap))

	resources := make([]ServiceInfo, 0, len(stages))
	for _, stage := range stages {
		project := projectsMap[stage.GetMeta().GetProjectId()]
		for duName, du := range stage.GetSpec().DeployUnits {
			rawAccount := stage.GetMeta().GetAccountId()
			if rawAccount == "" {
				rawAccount = project.GetSpec().GetAccountId()
			}
			abcID := convertAbcID(rawAccount)
			abcSlug, _ := a.abcClient.GetServiceSlugByID(strconv.FormatInt(abcID, 10))
			itype := yp.SelectItype(du)
			env := convertEnvironment(stage.GetSpec().GetDeployUnitSettings()[duName].GetEnvironment())
			responsible := project.GetMeta().GetOwnerId()
			r := ServiceInfo{
				ServiceProviderID: serviceProvider,
				ResourceID: map[string]string{
					"stage":      stage.GetMeta().GetId(),
					"deployUnit": duName,
					"itype":      itype,
				},
				ResourceType: resourceType,
				Environment:  env,
				AbcID:        abcID,
				AbcSlug:      abcSlug,
				Responsible:  responsible,
			}
			resources = append(resources, r)
		}
	}
	a.logger.Infof("discovered %d services", len(resources))

	a.dumpToYT(ctx, resources)

	return nil
}

func (a *App) Run(ctx context.Context) error {
	for {
		a.logger.Info("start new iteration")
		err := a.syncAbcResources(ctx)
		if err == nil {
			a.logger.Info("iteration successful")
		} else {
			a.logger.Error("iteration finished with error", log.Error(err))
			a.metrics.SyncErrors.Update(1)
		}

		sleep := make(chan struct{}, 1)
		go func() {
			time.Sleep(time.Duration(a.config.IterationInterval) * time.Second)
			sleep <- struct{}{}
		}()

		select {
		case <-sleep:
			continue
		case <-ctx.Done():
			a.logger.Info("exiting app")
			return nil
		}
	}
}

func StartSynchronizer() *cobra.Command {
	return &cobra.Command{
		Use:   "abc-export",
		Short: "Start abc resources synchronizer",
		Long:  `Start abc resources synchronizer`,
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			logger := zap.Must(zap.NewProductionDeployConfig())

			appConfig, err := config.ReadConfig(args[0])
			if err != nil {
				logger.Fatal("failed parse config", log.Error(err))
			}

			bootstrap, err := util.NewBootstrap(logger, appConfig, NewGenericApp)
			if err != nil {
				logger.Fatal("failed to create bootstrap", log.Error(err))
			}

			ctx, cancel := context.WithCancel(context.Background())

			go util.HandleSignals(logger, cancel)

			err = bootstrap.Run(ctx)
			logger.Fatal("bootstrap exited", log.Error(err))
		},
	}
}
