package generate

import (
	"a.yandex-team.ru/infra/alert_controller/internal/solomon"
	"a.yandex-team.ru/infra/alert_controller/internal/util"
	"a.yandex-team.ru/infra/alert_controller/internal/yp"
	"a.yandex-team.ru/library/go/core/log/zap"
	"context"
	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"
	"io/ioutil"
	"math/rand"
	"strings"
)

type AppConfig struct {
	YpCluster string            `yaml:"yp_cluster"`
	Project   string            `yaml:"project"`
	Channel   string            `yaml:"channel"`
	Percent   int               `yaml:"percent"`
	Versions  map[string]string `yaml:"versions"`
}

func loadConfig(path string) *AppConfig {
	var config AppConfig
	data, err := ioutil.ReadFile(path)
	if err != nil {
		panic(err)
	}

	err = yaml.Unmarshal(data, &config)
	if err != nil {
		panic(err)
	}

	return &config
}

func Generate() *cobra.Command {
	return &cobra.Command{
		Use:   "generate",
		Short: "Generate alerts from template",
		Long:  `Generate alerts from template`,
		Run: func(cmd *cobra.Command, args []string) {
			logger := zap.Must(zap.NewProductionDeployConfig())

			ctx, cancel := context.WithCancel(context.Background())

			go util.HandleSignals(logger, cancel)

			config := loadConfig(args[0])

			c, err := yp.NewYpClient(config.YpCluster)

			if err != nil {
				panic(err)
			}

			// Select all stages
			cnt, err := c.ListStages(ctx)
			if err != nil {
				panic(err)
			}
			logger.Infof("total stages: %d", cnt)

			// Init solomon API client
			solomonClient, err := solomon.NewSolomonClient(logger)
			if err != nil {
				panic(err)
			}

			// Load templates

			templates := solomon.LoadExperimentalTemplates()

			//
			//publishedTemplates, err := solomonClient.SelectTemplate(ctx)
			//if err != nil {
			//	panic(err)
			//}
			//
			//templates = solomon.ConvertFromSolomonTemplates(publishedTemplates)
			//
			//if config.Versions != nil {
			//	templates = solomon.ConvertFromMap(config.Versions)
			//}

			logger.Infof("Clean old alerts for %s", config.Project)
			for {
				aCnt := solomonClient.CleanAlerts(ctx, config.Project, "", logger)
				if aCnt < 1 {
					break
				}
			}

			stages := c.FetchAllStages(ctx)
			for _, stage := range stages {
				if strings.Contains(stage.GetMeta().GetId(), "test") || strings.Contains(stage.GetMeta().GetId(), "prestable") {
					continue
				}

				if rand.Intn(100) >= config.Percent {
					continue
				}

				for duName, du := range stage.GetSpec().DeployUnits {
					itype := yp.SelectItype(du)
					diskID := yp.SelectMainDisk(du)
					locations := yp.SelectClusters(du)

					logger.Infof("du %s, geo: %s, %s", duName, strings.Join(locations, ","), itype)

					info := solomon.ServiceInfo{
						Stage:      stage.GetMeta().GetId(),
						DeployUnit: duName,
						Itype:      itype,
						DiskID:     diskID,
					}

					for _, template := range templates {
						solomonClient.CreateSubAlert(ctx, info, config.Project, template, config.Channel)
					}

				}
			}
		},
	}
}
