package project

import (
	"a.yandex-team.ru/infra/alert_controller/internal/abc"
	"a.yandex-team.ru/infra/alert_controller/internal/solomon"
	"a.yandex-team.ru/infra/alert_controller/internal/util"
	"a.yandex-team.ru/infra/alert_controller/internal/yp"
	"a.yandex-team.ru/library/go/core/log/zap"
	"context"
	"github.com/spf13/cobra"
	"os"
	"time"
)

var MainCluster = "xdc"

func listAbcServices(ctx context.Context, c *yp.Client, logger *zap.Logger) map[string][]string {
	projects := c.FetchAllProjects(ctx)
	abcServices := make(map[string][]string)
	for _, project := range projects {
		if project.Spec.AccountId == "tmp" {
			continue
		}
		if project.Spec.AccountId[0:4] == "abc:" {
			abcServices[project.Spec.AccountId] = append(abcServices[project.Spec.AccountId], project.Meta.GetId())
		} else {
			logger.Infof("abnormal service name: %s", project.Spec.AccountId)
		}

	}
	logger.Infof("%d", len(projects))
	logger.Infof("%d", len(abcServices))
	return abcServices
}

func listMonitoringProjects(ctx context.Context, c *yp.Client, logger *zap.Logger) {
	projects := c.FetchAllProjects(ctx)
	cnt := 0
	for _, p := range projects {
		if p.Spec.GetMonitoringProject() != "" {
			logger.Info(p.Meta.GetId() + " " + p.Spec.GetMonitoringProject())
			cnt++
		}
	}
	logger.Infof("%d", cnt)
}

func FillProjects() *cobra.Command {
	return &cobra.Command{
		Use:   "fill-projects",
		Short: "Autofill monitoring projects",
		Run: func(cmd *cobra.Command, args []string) {
			logger := zap.Must(zap.NewProductionDeployConfig())

			ctx, cancel := context.WithCancel(context.Background())

			go util.HandleSignals(logger, cancel)

			abcCache := make(map[string]string)

			c, err := yp.NewYpClient(MainCluster)

			if err != nil {
				panic(err)
			}

			abcToProjects := listAbcServices(ctx, c, logger)

			abcToken, found := os.LookupEnv("ABC_TOKEN")
			if !found {
				panic("no abc token")
			}

			abcClientConfig := &abc.ClientConfig{
				OauthToken: abcToken,
			}

			abcClient := abc.NewClient(abcClientConfig)

			// Init solomon API client
			solomonClient, err := solomon.NewSolomonClient(logger)
			if err != nil {
				panic(err)
			}
			monitoringProjects := solomonClient.ListMonitoringProjects(ctx)
			logger.Infof("monitoring projects: %d", len(monitoringProjects))

			canBeUpdatedProjects := 0
			serviceCanBeUpdated := 0
			emptyProjects := 0
			emptyService := 0

			for abcFullID, projects := range abcToProjects {
				if abcFullID == "tmp" {
					continue
				}
				abcID := abcFullID[len("abc:service:"):]
				var abcSlug string
				if _, found := abcCache[abcID]; !found {
					abcSlug, err = abcClient.GetServiceSlugByID(abcID)
					if err != nil {
						logger.Infof("%v", err)
					}
					time.Sleep(200 * time.Millisecond)
				} else {
					abcSlug = abcCache[abcID]
				}

				if len(monitoringProjects[abcSlug]) == 1 {
					canBeUpdatedProjects += len(projects)
					serviceCanBeUpdated++

					for _, project := range projects {
						logger.Infof(" will be updated %s to %s", project, monitoringProjects[abcSlug][0])
						err := c.UpdateProjectMonitoring(ctx, project, monitoringProjects[abcSlug][0])
						if err != nil {
							logger.Errorf("%v", err)
						} else {
							logger.Info("success")
						}
						time.Sleep(100 * time.Millisecond)
					}
				}

				if len(monitoringProjects[abcSlug]) == 0 {
					emptyProjects += len(projects)
					emptyService++
				}
			}
			logger.Infof("service can be updated: %d", serviceCanBeUpdated)
			logger.Infof("projects can be updated: %d", canBeUpdatedProjects)

			logger.Infof("empty service: %d", emptyService)
			logger.Infof("empty projects: %d", emptyProjects)
		},
	}
}
