package stats

import (
	"a.yandex-team.ru/infra/alert_controller/internal/abc"
	"a.yandex-team.ru/infra/alert_controller/internal/solomon"
	"a.yandex-team.ru/infra/alert_controller/internal/swagger"
	"a.yandex-team.ru/infra/alert_controller/internal/util"
	"a.yandex-team.ru/infra/alert_controller/internal/yp"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"context"
	"github.com/spf13/cobra"
	"os"
	"time"
)

var MainCluster = "xdc"

func listProjects(ctx context.Context, c *yp.Client, logger *zap.Logger) map[string]*ypapi.TProject {
	projects := c.FetchAllProjects(ctx)
	res := make(map[string]*ypapi.TProject)
	for _, project := range projects {
		res[project.GetMeta().GetId()] = project
	}
	return res
}

var abcClient *abc.Client
var solomonClient *solomon.Client
var abcCache = make(map[string]string)

var monitoringProjectChannels = make(map[string]int)

func convertAbcIDToSlug(abcFullID string) string {
	if abcFullID == "tmp" {
		return ""
	}
	abcID := abcFullID[len("abc:service:"):]
	var abcSlug string
	if _, found := abcCache[abcID]; !found {
		abcSlug, _ = abcClient.GetServiceSlugByID(abcID)
		abcCache[abcID] = abcSlug
		time.Sleep(50 * time.Millisecond)
	} else {
		abcSlug = abcCache[abcID]
	}
	return abcSlug
}

func countChannels(ctx context.Context, project string) int {
	if _, found := monitoringProjectChannels[project]; !found {
		opts := swagger.AlertingApiListNotificationUsingGETOpts{}
		page, _, err := solomonClient.AlertingApi.ListNotificationUsingGET(ctx, project, &opts)
		if err != nil {
			monitoringProjectChannels[project] = -1
		} else {
			monitoringProjectChannels[project] = len(page.Items)
		}

		time.Sleep(50 * time.Millisecond)
	}
	return monitoringProjectChannels[project]
}

func CalcStatistics() *cobra.Command {
	return &cobra.Command{
		Use:   "stats",
		Short: "Calc deep alert statistics",
		Run: func(cmd *cobra.Command, args []string) {
			logger := zap.Must(zap.NewProductionDeployConfig())

			ctx, cancel := context.WithCancel(context.Background())

			go util.HandleSignals(logger, cancel)

			// Init YP Client
			c, err := yp.NewYpClient(MainCluster)
			if err != nil {
				panic(err)
			}
			// Init ABC Client
			abcToken, found := os.LookupEnv("ABC_TOKEN")
			if !found {
				panic("no abc token")
			}
			abcClient = abc.NewClient(&abc.ClientConfig{
				OauthToken: abcToken,
			})
			// Init solomon API client
			solomonClient, err = solomon.NewSolomonClient(logger)
			if err != nil {
				panic(err)
			}

			// List projects map
			projects := listProjects(ctx, c, logger)

			monitoringProjectsIDs := solomonClient.ListMonitoringProjectsIDs(ctx)
			logger.Infof("monitoring projects: %d", len(monitoringProjectsIDs))

			stages := c.FetchAllStages(ctx)

			total := 0
			withMonitoringProject := 0
			cahnnelsCanBeUpdated := 0
			multipleChannel := 0
			zeroChannels := 0
			inaccessibleChannels := 0

			for _, stage := range stages {
				//stageID := stage.GetMeta().GetId()
				projectID := stage.GetMeta().GetProjectId()
				project := projects[projectID]
				monitoringProject := project.GetSpec().GetMonitoringProject()
				weight := len(stage.GetSpec().DeployUnits)
				total += weight
				if monitoringProject != "" {
					withMonitoringProject += weight
					switch countChannels(ctx, monitoringProject) {
					case -1:
						inaccessibleChannels += weight
					case 0:
						zeroChannels += weight
					case 1:
						cahnnelsCanBeUpdated += weight
					default:
						multipleChannel += weight
					}
				}
			}

			logger.Infof("deploy units total: %d", total)
			logger.Infof("deploy units with monitoring project: %d", withMonitoringProject)
			logger.Infof("deploy units with inaccessible projects: %d", inaccessibleChannels)
			logger.Infof("deploy units with empty notification channel: %d", zeroChannels)
			logger.Infof("deploy units with multiple notification channels: %d", multipleChannel)
			logger.Infof("deploy units can be updated: %d", cahnnelsCanBeUpdated)
		},
	}
}
