package abc

import (
	"fmt"
	"net/http"
	"time"

	"github.com/go-resty/resty/v2"
)

const (
	defaultURL            = "https://abc-back.yandex-team.ru/api/v4"
	defaultRequestTimeout = time.Second * 10
)

type ClientConfig struct {
	URL            string        `yaml:"url"`
	OauthToken     string        `yaml:"token"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
}

type Client struct {
	c     *resty.Client
	cache map[string]string
}

func NewClient(config *ClientConfig) *Client {
	requestTimeout := config.RequestTimeout
	if requestTimeout == 0 {
		requestTimeout = defaultRequestTimeout
	}

	URL := config.URL
	if URL == "" {
		URL = defaultURL
	}

	return &Client{
		cache: make(map[string]string),
		c: resty.New().
			SetTimeout(requestTimeout).
			SetHostURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(config.OauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}
}

type ServiceSlugResult struct {
	Slug string `json:"slug"`
}

type GetServiceSlugByIDResponse struct {
	Results []ServiceSlugResult `json:"results"`
}

func (c *Client) GetServiceSlugByID(abcServiceID string) (string, error) {
	if slug, found := c.cache[abcServiceID]; found {
		return slug, nil
	}

	params := map[string]string{
		"id":     abcServiceID,
		"fields": "slug",
	}
	var slugs GetServiceSlugByIDResponse
	resp, err := c.c.R().SetQueryParams(params).SetResult(&slugs).Get("/services/")
	if err != nil {
		return "", err
	}

	if resp.StatusCode() != http.StatusOK {
		return "", fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}
	if len(slugs.Results) != 1 {
		return "", fmt.Errorf("incorrect respounce items count: %d", len(slugs.Results))
	}

	c.cache[abcServiceID] = slugs.Results[0].Slug

	return slugs.Results[0].Slug, nil
}
