package service

import (
	"a.yandex-team.ru/infra/alert_controller/internal/solomon"
	"a.yandex-team.ru/infra/alert_controller/internal/unistat"
	"a.yandex-team.ru/infra/alert_controller/internal/yp"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"context"
	"errors"
	"reflect"
)

type Service struct {
	id       string
	provider string
	settings string
	project  string
	channel  string
	info     solomon.ServiceInfo
}

func (s *Service) GetProject() string {
	return s.project
}

func (s *Service) GetLabels() string {
	return "stage=" + s.info.Stage + ",deploy_unit=" + s.info.DeployUnit + ",service=deploy"
}

type Manager struct {
	solomonClient *solomon.Client
	logger        *zap.Logger
	metric        *unistat.Stats
}

func NewManager(solomonClient *solomon.Client, logger *zap.Logger, metric *unistat.Stats) *Manager {
	return &Manager{
		solomonClient: solomonClient,
		logger:        logger,
		metric:        metric,
	}
}

func (m *Manager) ListDeployService(ctx context.Context, projects []*ypapi.TProject, stages []*ypapi.TStage) map[string]Service {
	services := make(map[string]Service, 2*len(stages))
	monitoringProjects := make(map[string]string, len(projects))
	m.metric.ProjectsTotal.Update(float64(len(projects)))
	monitoredProjects := 0
	for _, project := range projects {
		monitoringProjects[project.GetMeta().GetId()] = project.GetSpec().GetMonitoringProject()
		if project.GetSpec().GetMonitoringProject() != "" {
			monitoredProjects++
		}
	}
	m.metric.ProjectsWithMonitoring.Update(float64(monitoredProjects))
	totalMonitoredServices := 0
	totalServices := 0
	for _, stage := range stages {
		totalServices += len(stage.GetSpec().DeployUnits)
		project := stage.GetMeta().GetProjectId()
		monitoringProject := monitoringProjects[project]
		if monitoringProject == "" {
			continue
		}

		for duName, du := range stage.GetSpec().DeployUnits {
			if settings, ok := stage.GetSpec().GetDeployUnitSettings()[duName]; ok &&
				settings.GetAlerting().GetState() == ypapi.TStageSpec_TDeployUnitSettings_TAlerting_ENABLED {
				totalMonitoredServices++

				m.logger.Info(stage.GetMeta().GetId() + "[" + duName + "]")

				service := Service{
					id:       "deploy." + stage.GetMeta().GetId() + "." + duName,
					provider: "deploy",
					project:  monitoringProject,
					channel:  settings.GetAlerting().NotificationChannels["ERROR"],
					info: solomon.ServiceInfo{
						Stage:      stage.GetMeta().GetId(),
						DeployUnit: duName,
						Itype:      yp.SelectItype(du),
						DiskID:     yp.SelectMainDisk(du),
					},
				}
				services[service.id] = service
			}
		}
	}
	m.metric.ServicesTotal.Update(float64(totalServices))
	m.metric.ServicesWithMonitoring.Update(float64(totalMonitoredServices))

	return services
}

func (m *Manager) SyncAlerts(ctx context.Context,
	old map[string]Service, new map[string]Service,
	templates []solomon.TemplateInfo,
	forceReCreateAlerts bool) {
	for id, service := range new {
		if oldSpec, found := old[id]; found {
			if reflect.DeepEqual(service, oldSpec) {
				continue
			}
			err := m.RemoveAlertsForService(ctx, oldSpec)
			if err != nil {
				m.logger.Error("cant remove alerts", log.Error(err))
			}
		}
		if forceReCreateAlerts {
			m.logger.Infof("removing all old alerts for %s", service.id)
			err := m.RemoveAlertsForService(ctx, service)
			if err != nil {
				m.logger.Error("cant remove alerts", log.Error(err))
			}
		}
		err := m.CreateAlertsForService(ctx, service, templates)
		if err != nil {
			m.logger.Error("cant create alerts", log.Error(err))
		}
	}

	for id, service := range old {
		if _, found := new[id]; !found {
			err := m.RemoveAlertsForService(ctx, service)
			if err != nil {
				m.logger.Error("cant remove alerts", log.Error(err))
			}
		}
	}
}

func (m *Manager) CreateAlertsForService(ctx context.Context, service Service, templates []solomon.TemplateInfo) error {
	ids := m.solomonClient.FetchAlertsID(ctx, m.logger, service.GetProject(), service.GetLabels())
	if len(ids) == len(templates) {
		m.logger.Infof("%s will be skipped", service.id)
		return nil
	}

	usedTemplates := map[string]struct{}{}
	for _, id := range ids {
		data := m.solomonClient.FetchAlertInfo(ctx, service.project, id)
		usedTemplates[data.TemplateID] = struct{}{}
	}

	for _, template := range templates {
		if _, ok := usedTemplates[template.ID]; !ok {
			m.solomonClient.CreateSubAlert(ctx, service.info, service.project, template, service.channel)
		}
	}

	return nil
}

func (m *Manager) RemoveAlertsForService(ctx context.Context, service Service) error {
	ids := m.solomonClient.FetchAlertsID(ctx, m.logger, service.GetProject(), service.GetLabels())
	cnt := m.solomonClient.RemoveAlerts(ctx, service.project, ids)
	if cnt != len(ids) {
		return errors.New("error during alerts removing")
	}
	return nil
}
