package solomon

import (
	"a.yandex-team.ru/infra/alert_controller/internal/antihax/optional"
	"a.yandex-team.ru/infra/alert_controller/internal/swagger"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"context"
	"fmt"
	"os"
)

const baseDeployURL = "https://deploy.yandex-team.ru"
const serviceProvider = "deploy"

type Client struct {
	logger *zap.Logger
	*swagger.APIClient
}

type TemplateInfo struct {
	ID      string
	Version string
	Name    string
}

type ServiceInfo struct {
	Stage      string
	DeployUnit string
	Geo        string
	Itype      string
	DiskID     string
}

type AlertInfo struct {
	ID              string
	TemplateID      string
	TemplateVersion string
}

func (s *Client) fillTextParameters(template TemplateInfo, info ServiceInfo) []swagger.TextParameterValue {
	textParams := []swagger.TextParameterValue{
		{Name: "stage", Value: info.Stage},
		{Name: "deployUnit", Value: info.DeployUnit},
		{Name: "itype", Value: info.Itype},
	}

	if template.ID == "deploy-main-disk-usage" {
		if info.DiskID != "" {
			textParams = append(textParams, swagger.TextParameterValue{Name: "diskId", Value: info.DiskID})
		}
	}

	return textParams
}

func (s *Client) CreateSubAlert(ctx context.Context,
	info ServiceInfo,
	projectID string,
	template TemplateInfo,
	chanelID string) {

	opts := swagger.AlertingApiCreateAlertUsingPOSTOpts{
		XServiceProvider: optional.NewString(serviceProvider),
	}

	textParams := s.fillTextParameters(template, info)

	model := swagger.ModelType{
		FromTemplate: &swagger.AlertFromTemplate{
			DoubleValueParameters:    nil,
			DoubleValueThresholds:    nil,
			IntValueParameters:       nil,
			IntValueThresholds:       nil,
			LabelListValueParameters: nil,
			LabelListValueThresholds: nil,
			ServiceProvider:          serviceProvider,
			TemplateId:               template.ID,
			TemplateVersionTag:       template.Version,
			TextListValueParameters:  nil,
			TextListValueThresholds:  nil,
			TextValueParameters:      textParams,
			TextValueThresholds:      nil,
		},
	}

	statuses := []string{"ALARM", "ERROR", "WARN", "OK"}

	channel := swagger.AssociatedChannel{
		Id: chanelID,
		Config: &swagger.ChannelConfig{
			NotifyAboutStatuses: statuses,
			RepeatDelaySecs:     0,
		},
	}

	channels := []swagger.AssociatedChannel{channel}

	labels := map[string]string{
		"stage":       info.Stage,
		"deploy_unit": info.DeployUnit,
		"service":     serviceProvider,
	}

	description := fmt.Sprintf("alert for %s/stages/%s/status/%s\n\n"+
		"documentation https://deploy.yandex-team.ru/docs/reference/alerts",
		baseDeployURL, info.Stage, info.DeployUnit)
	name := fmt.Sprintf("%s for %s.%s", template.Name, info.Stage, info.DeployUnit)

	alert := swagger.Alert{
		Annotations:                nil,
		Channels:                   channels,
		CreatedAt:                  nil,
		CreatedBy:                  "",
		DelaySecs:                  0,
		Description:                description,
		GroupByLabels:              nil,
		Labels:                     labels,
		Name:                       name,
		NoPointsPolicy:             "",
		ProjectId:                  projectID,
		ResolvedEmptyPolicy:        "",
		ServiceProviderAnnotations: nil,
		State:                      "ACTIVE",
		Type_:                      &model,
		UpdatedAt:                  nil,
		UpdatedBy:                  "",
		Version:                    0,
		WindowSecs:                 0,
	}

	resp, _, err := s.APIClient.AlertingApi.CreateAlertUsingPOST(ctx, alert, projectID, &opts)
	if err != nil {
		if ferr, ok := err.(swagger.GenericSwaggerError); ok {
			s.logger.Errorf("error: %s", string(ferr.Body()))
			return
		}
	}
	s.logger.Infof("created alert %s", resp.Id)
}

func (s *Client) RemoveAlert(ctx context.Context, alertID string, projectID string) {
	_, err := s.APIClient.AlertingApi.DeleteAlertUsingDELETE(ctx, alertID, projectID, &swagger.AlertingApiDeleteAlertUsingDELETEOpts{
		XServiceProvider: optional.NewString(serviceProvider),
	})
	if err != nil {
		s.logger.Error("failed to remove alert", log.Error(err))
	} else {
		s.logger.Infof("removed alert %s", alertID)
	}
}

func (s *Client) RemoveAlerts(ctx context.Context, project string, ids []string) int {
	removed := 0
	for _, id := range ids {
		s.RemoveAlert(ctx, id, project)
		removed++
	}
	return removed
}

func (s *Client) CleanAlerts(ctx context.Context, project string, filter string, logger *zap.Logger) int {
	ids := make([]string, 0)
	removed := 0
	data, _, err := s.AlertingApi.ListAlertsByProjectUsingGET1(ctx, project, &swagger.AlertingApiListAlertsByProjectUsingGET1Opts{
		PageSize: optional.NewInt32(500),
	})
	if err != nil {
		return 0
	}
	for _, alert := range data.Items {
		logger.Infof("%s", alert.Id)
		ids = append(ids, alert.Id)
	}
	for _, id := range ids {
		if id == filter {
			continue
		}
		s.RemoveAlert(ctx, id, project)
		removed++
	}
	return removed
}

func (s *Client) FetchAlertInfo(ctx context.Context, project string, alertID string) AlertInfo {
	var res AlertInfo
	data, _, err := s.AlertingApi.GetAlertUsingGET(ctx, alertID, project, &swagger.AlertingApiGetAlertUsingGETOpts{
		XServiceProvider: optional.NewString(serviceProvider),
	})
	if err != nil || data.Type_ == nil || data.Type_.FromTemplate == nil {
		return res
	}
	res.TemplateID = data.Type_.FromTemplate.TemplateId
	res.TemplateVersion = data.Type_.FromTemplate.TemplateVersionTag
	res.ID = data.Id
	return res
}

func (s *Client) FetchAlertsID(ctx context.Context, logger *zap.Logger, project string, labels string) []string {
	ids := make([]string, 0)

	data, _, err := s.AlertingApi.ListAlertsByProjectUsingGET1(ctx, project, &swagger.AlertingApiListAlertsByProjectUsingGET1Opts{
		PageSize:         optional.NewInt32(500),
		LabelsSelector:   optional.NewString(labels),
		XServiceProvider: optional.NewString(serviceProvider),
	})
	if err != nil {
		return ids
	}
	for _, alert := range data.Items {
		ids = append(ids, alert.Id)
	}
	return ids
}

func (s *Client) SelectTemplate(ctx context.Context) ([]swagger.AlertTemplate, error) {
	resp, _, err := s.AlertingApi.ListAlertTemplateUsingGET(ctx, &swagger.AlertingApiListAlertTemplateUsingGETOpts{
		ServiceProviderId: optional.NewString(serviceProvider),
	})
	return resp.Items, err
}

func (s *Client) ListMonitoringProjectsIDs(ctx context.Context) []string {
	projects := make([]string, 0)
	res, _, err := s.APIClient.ProjectsApi.AllProjectsUsingGET(ctx, &swagger.ProjectsApiAllProjectsUsingGETOpts{})
	if err != nil {
		panic(err)
	}
	for _, v := range res.([]interface{}) {
		projects = append(projects, v.(map[string]interface{})["id"].(string))
	}
	return projects
}

func (s *Client) ListMonitoringProjects(ctx context.Context) map[string][]string {
	res, _, err := s.APIClient.ProjectsApi.AllProjectsUsingGET(ctx, &swagger.ProjectsApiAllProjectsUsingGETOpts{})
	if err != nil {
		panic(err)
	}

	projects := make(map[string][]string)

	for _, v := range res.([]interface{}) {
		id := v.(map[string]interface{})["id"].(string)
		abc := v.(map[string]interface{})["abcService"].(string)
		projects[abc] = append(projects[abc], id)
	}
	return projects
}

func NewSolomonClient(logger *zap.Logger) (*Client, error) {
	cfg := swagger.NewConfiguration()

	solomonToken, ok := os.LookupEnv("SOLOMON_TOKEN")
	if !ok {
		panic("No solomon token")
	}

	cfg.DefaultHeader = map[string]string{
		"Authorization": "OAuth " + solomonToken,
	}

	return &Client{
		logger,
		swagger.NewAPIClient(cfg),
	}, nil
}
