package staff

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"time"

	"a.yandex-team.ru/library/go/core/log"
)

type ClientConfig struct {
	BaseURL string
	Token   string
}

type Client struct {
	config  *ClientConfig
	baseURL *url.URL
	client  *http.Client
	logger  log.Logger
}

func NewClient(cfg *ClientConfig, logger log.Logger) (*Client, error) {
	baseURL, err := url.Parse(cfg.BaseURL)
	if err != nil {
		return nil, err
	}
	if baseURL.Path != "" || baseURL.RawQuery != "" {
		return nil, fmt.Errorf("%s must be a base URL", baseURL)
	}

	return &Client{
		config: cfg,
		client: &http.Client{Timeout: time.Minute},
		logger: logger,
	}, nil
}

func (m *Client) prepareGetRequest(ctx context.Context, path string, query url.Values) (*http.Request, error) {
	requestURL := m.baseURL
	requestURL.Path = path
	requestURL.RawQuery = query.Encode()

	request, err := http.NewRequestWithContext(ctx, http.MethodGet, requestURL.String(), nil)
	if err != nil {
		return nil, err
	}

	request.Header.Set("Accept", "application/json")
	request.Header.Set("Authorization", fmt.Sprintf("OAuth %s", string(m.config.Token)))

	return request, nil
}

func (m *Client) doRequest(ctx context.Context, request *http.Request) ([]byte, error) {
	m.logger.Debug("executing HTTP", log.String("URL", request.URL.String()))

	response, err := m.client.Do(request)
	if response != nil {
		defer func() {
			if err := response.Body.Close(); err != nil {
				m.logger.Warn("failed to close response body", log.Error(err))
			}
		}()
	}
	if err != nil {
		return nil, err
	}

	data, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to read response body: %w", err)
	}

	if response.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("staff status is %v, not 200: %s", response.StatusCode, string(data))
	}

	return data, nil
}

func (m *Client) FetchAllMembership(ctx context.Context) (map[string]map[string]struct{}, error) {
	memberships, err := m.doFetchMembership(ctx)

	return memberships, err
}

func (m *Client) doFetchMembership(ctx context.Context) (map[string]map[string]struct{}, error) {
	query := url.Values{
		"person.official.is_dismissed": []string{"false"},
		"_fields":                      []string{"person.login,group.url"},
		"_limit":                       []string{"2000000000"}, // no limit (this number fits in int32)
	}

	request, err := m.prepareGetRequest(ctx, "/v3/groupmembership", query)
	if err != nil {
		return nil, err
	}

	data, err := m.doRequest(ctx, request)
	if err != nil {
		return nil, err
	}

	type PersonLogin struct {
		Login string `json:"login"`
	}
	type GroupURL struct {
		URL string `json:"url"`
	}
	type Membership struct {
		Person PersonLogin `json:"person"`
		Group  GroupURL    `json:"group"`
	}
	type MembershipResult struct {
		ResultArray []Membership `json:"result"`
	}

	membershipResponse := MembershipResult{}
	if err := json.Unmarshal(data, &membershipResponse); err != nil {
		return nil, fmt.Errorf("failed to decode staff response: %w", err)
	}

	out := make(map[string]map[string]struct{})
	for _, membership := range membershipResponse.ResultArray {
		if _, ok := out[membership.Person.Login]; !ok {
			out[membership.Person.Login] = make(map[string]struct{})
		}

		out[membership.Person.Login][membership.Group.URL] = struct{}{}
	}

	return out, nil
}
