package util

import (
	"a.yandex-team.ru/infra/alert_controller/internal/unistat"
	"a.yandex-team.ru/library/go/core/log/zap"
	"context"
	"fmt"
	"golang.org/x/sync/errgroup"
	"net/http"
	"os"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/ytlock"
)

type EngineConfig struct {
	baseTimeout time.Duration
	LeaderLock  *LeaderLockConfig `yaml:"leader_lock"`
	App         interface{}       `yaml:"app"`
}

type Config struct {
	DebugHTTPEndpoint string       `yaml:"debug_http_endpoint"`
	Engine            EngineConfig `yaml:"engine"`
}

type LeaderLockConfig struct {
	Path     string         `yaml:"path"`
	YTClient YTClientConfig `yaml:"yt_client"`
}

type Bootstrap struct {
	logger log.Logger
	config Config
	engine *Engine
}

type App interface {
	Run(ctx context.Context) error
}

type AppConstructor func(
	logger *zap.Logger,
	configPtr interface{},
	metrics *unistat.Stats,
) (App, error)

type Engine struct {
	logger   log.Logger
	registry metrics.Registry
	config   EngineConfig

	app          App
	isActive     bool
	isActiveLock *sync.RWMutex
}

func NewBootstrap(
	logger *zap.Logger,
	config Config,
	constructor AppConstructor,
) (*Bootstrap, error) {
	stats := unistat.MakeStats()

	http.HandleFunc("/unistat", stats.ServeHTTP)
	go func() {
		err := http.ListenAndServe(config.DebugHTTPEndpoint, nil)
		if err != nil {
			panic(fmt.Errorf("failed to start HTTP listener: %v", err))
		}
	}()

	logger.Info("started HTTP listener", log.String("address", config.DebugHTTPEndpoint))

	engine, err := NewEngine(logger, config.Engine, constructor, stats)
	if err != nil {
		return nil, fmt.Errorf("failed to create runner for app: %v", err)
	}

	return &Bootstrap{logger, config, engine}, nil
}

func (b *Bootstrap) Run(ctx context.Context) error {
	g, gctx := errgroup.WithContext(ctx)

	g.Go(func() error {
		return b.engine.RunApp(gctx)
	})

	err := g.Wait()
	return err
}

func NewEngine(
	logger *zap.Logger,
	config EngineConfig,
	constructor AppConstructor,
	metrics *unistat.Stats,
) (*Engine, error) {
	app, err := constructor(logger, config.App, metrics)
	if err != nil {
		return nil, fmt.Errorf("failed to create app: %v", err)
	}

	return &Engine{
		logger:       logger,
		config:       config,
		app:          app,
		isActive:     false,
		isActiveLock: &sync.RWMutex{},
	}, nil
}

func (e *Engine) RunApp(ctx context.Context) error {
	for {
		err := e.RunAppOnce(ctx)
		if err != nil {
			e.logger.Error("failed to run app", log.Error(err))
		}

		select {
		case <-ctx.Done():
			return err
		default:
			time.Sleep(e.config.baseTimeout)
		}
	}
}

func (e *Engine) RunAppOnce(ctx context.Context) error {
	lockCtx, lockCancel := context.WithCancel(ctx)
	defer lockCancel()

	lockLost, err := e.acquireLeaderLock(lockCtx)
	if err != nil {
		return err
	}

	e.logger.Info("starting app")

	e.setIsActive(true)
	defer e.setIsActive(false)

	appCtx, appCancel := context.WithCancel(ctx)
	defer appCancel()

	appErrChan := make(chan error)
	go func() {
		appErrChan <- e.app.Run(appCtx)
	}()

	select {
	case appErr := <-appErrChan:
		e.logger.Error("app has failed", log.Error(appErr))
	case <-lockLost:
		e.logger.Error("lock is lost, waiting for the app to shut down")
		appCancel()
		e.logger.Error("app is shut down (lock is lost)", log.Error(<-appErrChan))
	}

	return nil
}

func (e *Engine) acquireLeaderLock(ctx context.Context) (<-chan struct{}, error) {
	if e.config.LeaderLock == nil {
		e.logger.Info("leader lock is disabled via config")
		return make(chan struct{}), nil
	}

	ytClient, err := NewYTClient(e.logger, e.config.LeaderLock.YTClient, "")
	if err != nil {
		return nil, fmt.Errorf("failed to create YT client: %v", err)
	}

	hostname, _ := os.Hostname()
	lock := ytlock.NewLockOptions(
		ytClient,
		ypath.Path(e.config.LeaderLock.Path),
		ytlock.Options{
			CreateIfMissing: true,
			LockMode:        yt.LockExclusive,
			TxAttributes: map[string]interface{}{
				"title":    fmt.Sprintf("Leader lock for %s", hostname),
				"hostname": hostname,
			},
		},
	)

	e.logger.Info("trying to acquire lock", log.String("path", e.config.LeaderLock.Path))
	lost, err := lock.Acquire(ctx)
	if err == nil {
		e.logger.Info("lock acquired")
	} else {
		e.logger.Error("failed to acquire lock", log.Error(err))
	}

	return lost, err
}

func (e *Engine) IsActive() bool {
	e.isActiveLock.RLock()
	defer e.isActiveLock.RUnlock()
	return e.isActive
}

func (e *Engine) setIsActive(isActive bool) {
	e.isActiveLock.Lock()
	defer e.isActiveLock.Unlock()
	e.isActive = isActive
}
