package yp

import (
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
	"context"
	"fmt"
	"sort"
)

type Client struct {
	*yp.Client
}

func NewYpClient(cluster string) (*Client, error) {
	c, err := yp.NewClient(cluster, yp.WithSystemAuthToken())
	if err != nil {
		return nil, fmt.Errorf("create YP-client: %w", err)
	}

	return &Client{
		Client: c,
	}, nil
}

func (y *Client) StageInfo(ctx context.Context, stageName string) (uint32, error) {
	rsp, err := y.GetStage(ctx, yp.GetStageRequest{
		Format:    yp.PayloadFormatYson,
		ID:        stageName,
		Selectors: []string{"/status/revision"},
	})
	if err != nil {
		return 0, fmt.Errorf("get stage: %w", err)
	}

	var rev uint32

	if err := rsp.Fill(&rev); err != nil {
		return 0, fmt.Errorf("fill results: %w", err)
	}

	return rev, nil
}
func (y *Client) ListStages(ctx context.Context) (int, error) {
	rsp, err := y.SelectStages(ctx, yp.SelectStagesRequest{
		Filter:            "",
		Format:            0,
		Selectors:         []string{"/meta/id"},
		ContinuationToken: "",
		Offset:            0,
		Limit:             0,
		Timestamp:         0,
		FetchRootObject:   false,
	})
	if err != nil {
		return 0, err
	}

	return rsp.Count(), nil
}
func fixForMoscow(geo []string) []string {
	res := make([]string, 0)
	isMsk := false
	for _, loc := range geo {
		if loc == "myt" || loc == "iva" {
			if isMsk {
				continue
			} else {
				res = append(res, "msk")
				isMsk = true
			}
		} else {
			res = append(res, loc)
		}
	}
	sort.Strings(res)
	return res
}

func SelectClusters(du *ypapi.TDeployUnitSpec) []string {
	geo := make([]string, 0)
	for _, cluster := range du.GetMultiClusterReplicaSet().GetReplicaSet().GetClusters() {
		if cluster.GetSpec().GetReplicaCount() > 0 {
			geo = append(geo, cluster.GetCluster())
		}
	}
	for cluster, settings := range du.GetReplicaSet().GetPerClusterSettings() {
		if settings.GetPodCount() > 0 {
			geo = append(geo, cluster)
		}
	}

	return fixForMoscow(geo)
}

func getPodSpec(du *ypapi.TDeployUnitSpec) *ypapi.TPodSpec {
	if du.GetReplicaSet() != nil {
		return du.GetReplicaSet().GetReplicaSetTemplate().GetPodTemplateSpec().GetSpec()
	}

	return du.GetMultiClusterReplicaSet().GetReplicaSet().GetPodTemplateSpec().GetSpec()
}

func SelectItype(du *ypapi.TDeployUnitSpec) string {
	podSpec := getPodSpec(du)

	if podSpec.GetHostInfra().GetMonitoring().GetLabels()["itype"] != "" {
		return podSpec.GetHostInfra().GetMonitoring().GetLabels()["itype"]
	}
	return "deploy"
}

func SelectMainDisk(du *ypapi.TDeployUnitSpec) string {
	for _, disk := range getPodSpec(du).GetDiskVolumeRequests() {
		if len(disk.GetLabels().GetAttributes()) > 0 {
			return disk.GetId()
		}
	}
	return ""
}

func (y *Client) getStageBatch(ctx context.Context, ct string) ([]*ypapi.TStage, string, error) {
	rv := make([]*ypapi.TStage, 0, DefaultGetObjectsBatchSize)
	req := yp.SelectStagesRequest{
		Format:    yp.PayloadFormatProto,
		Selectors: DefaultGetObjectsSelectors,
		Limit:     DefaultGetObjectsBatchSize,
	}
	if ct != "" {
		req.ContinuationToken = ct
	}
	rsp, err := y.SelectStages(ctx, req)
	if err != nil {
		return nil, ct, err
	}
	for rsp.Next() {
		t := &ypapi.TStage{
			Meta: &ypapi.TStageMeta{},
			Spec: &ypapi.TStageSpec{},
		}
		err := rsp.Fill(t.Meta, t.Spec)
		if err != nil {
			return nil, ct, err
		}
		rv = append(rv, t)
	}
	ct = rsp.ContinuationToken()
	return rv, ct, nil
}

func (y *Client) getProjectBatch(ctx context.Context, ct string) ([]*ypapi.TProject, string, error) {
	rv := make([]*ypapi.TProject, 0, DefaultGetObjectsBatchSize)
	req := yp.SelectProjectsRequest{
		Format:    yp.PayloadFormatProto,
		Selectors: DefaultGetObjectsSelectors,
		Limit:     DefaultGetObjectsBatchSize,
	}
	if ct != "" {
		req.ContinuationToken = ct
	}
	rsp, err := y.SelectProjects(ctx, req)
	if err != nil {
		return nil, ct, err
	}
	for rsp.Next() {
		t := &ypapi.TProject{
			Meta: &ypapi.TProjectMeta{},
			Spec: &ypapi.TProjectSpec{},
		}
		err := rsp.Fill(t.Meta, t.Spec)
		if err != nil {
			return nil, ct, err
		}
		rv = append(rv, t)
	}
	ct = rsp.ContinuationToken()
	return rv, ct, nil
}

func (y *Client) FetchAllStages(ctx context.Context) []*ypapi.TStage {
	allStages := make([]*ypapi.TStage, 0, DefaultGetObjectsBatchSize)

	token := ""
	for {
		stages, newToken, err := y.getStageBatch(ctx, token)
		if err != nil {
			panic(fmt.Errorf("failed getting stages: %w", err))
		}
		token = newToken
		allStages = append(allStages, stages...)
		if len(stages) < DefaultGetObjectsBatchSize {
			break
		}
	}
	return allStages
}

func (y *Client) FetchAllProjects(ctx context.Context) []*ypapi.TProject {
	allProjects := make([]*ypapi.TProject, 0, DefaultGetObjectsBatchSize)

	token := ""
	for {
		projects, newToken, err := y.getProjectBatch(ctx, token)
		if err != nil {
			panic(fmt.Errorf("failed getting projects: %w", err))
		}
		token = newToken
		allProjects = append(allProjects, projects...)
		if len(projects) < DefaultGetObjectsBatchSize {
			break
		}
	}
	return allProjects
}

func (y *Client) UpdateProjectMonitoring(ctx context.Context, projectID string, monitoringProject string) error {
	_, err := y.UpdateProject(ctx, yp.UpdateProjectRequest{
		ID: projectID,
		SetUpdates: []yp.SetObjectUpdate{
			{
				Path:      "/spec/monitoring_project",
				Object:    monitoringProject,
				Recursive: false,
			},
		},
		RemoveUpdates: nil,
	})
	return err
}
