package main

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/controller"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/instancespec"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/confgen"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/log"
	ypcluster "a.yandex-team.ru/infra/allocation-ctl/pkg/yp"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/yputil"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	nannyrpc "a.yandex-team.ru/infra/nanny/go/nanny"
	internalpb "a.yandex-team.ru/infra/nanny/go/proto/nanny_internal"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	hqpb "a.yandex-team.ru/yp/go/proto/hq"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
	"bytes"
	"context"
	"flag"
	"fmt"
	"github.com/davecgh/go-spew/spew"
	"github.com/pmezard/go-difflib/difflib"
	"github.com/stretchr/testify/assert"
	"gopkg.in/yaml.v2"
	"io/ioutil"
	"reflect"
	"strings"
)

var (
	configPath string
)

var spewConfig = spew.ConfigState{
	Indent:                  " ",
	DisablePointerAddresses: true,
	DisableCapacities:       true,
	SortKeys:                true,
	DisableMethods:          true,
	MaxDepth:                10,
}

func typeAndKind(v interface{}) (reflect.Type, reflect.Kind) {
	t := reflect.TypeOf(v)
	k := t.Kind()

	if k == reflect.Ptr {
		t = t.Elem()
		k = t.Kind()
	}
	return t, k
}

// diff returns a diff of both values as long as both are of the same type and
// are a struct, map, slice, array or string. Otherwise it returns an empty string.
func diff(expected interface{}, actual interface{}) string {
	if expected == nil || actual == nil {
		return ""
	}

	et, ek := typeAndKind(expected)
	at, _ := typeAndKind(actual)

	if et != at {
		return ""
	}

	if ek != reflect.Struct && ek != reflect.Map && ek != reflect.Slice && ek != reflect.Array && ek != reflect.String {
		return ""
	}

	var e, a string
	if et != reflect.TypeOf("") {
		e = spewConfig.Sdump(expected)
		a = spewConfig.Sdump(actual)
	} else {
		e = reflect.ValueOf(expected).String()
		a = reflect.ValueOf(actual).String()
	}

	diff, _ := difflib.GetUnifiedDiffString(difflib.UnifiedDiff{
		A:        difflib.SplitLines(e),
		B:        difflib.SplitLines(a),
		FromFile: "Expected",
		FromDate: "",
		ToFile:   "Actual",
		ToDate:   "",
		Context:  1,
	})

	return "\nDiff:\n" + diff
}

type snapshot struct {
	ID           string
	ConfID       string
	RuntimeAttrs *nannyclient.RuntimeAttrs
	InstanceSpec instancespec.Interface
	ISSTemplate  *internalpb.IssConfTemplate
	RepoSnapshot *repopb.Snapshot
}

var processed map[string]struct{}

type service struct {
	ID        string
	Snapshots map[string]*snapshot
}

func listServiceIDs(ctx context.Context, c *nannyrpc.NannyRPCClient) ([]string, error) {
	req := &repopb.ListSummariesRequest{}
	rsp, err := c.ListSummaries(ctx, req)
	if err != nil {
		return nil, err
	}
	ids := make([]string, 0, len(rsp.Value))
	for _, s := range rsp.Value {
		ids = append(ids, s.GetServiceId())
	}
	return ids, nil
}

func isNotFound(err error) bool {
	cerr, ok := err.(*rpc.ClientError)
	return ok && cerr.Code == 404
}

func getService(ctx context.Context, ID string, nannyClient *nannyclient.NannyClient, nannyRPCClient *nannyrpc.NannyRPCClient) (*service, error) {
	rv := &service{
		ID: ID,
	}
	req := &repopb.GetServiceRequest{
		ServiceId: ID,
	}
	rsp, err := nannyRPCClient.GetService(ctx, req)
	if err != nil {
		if isNotFound(err) {
			return nil, nil
		}
		return nil, err
	}
	snapshots := make(map[string]*snapshot, len(rsp.Service.GetSpec().GetSnapshot()))
	for _, repoSn := range rsp.Service.GetSpec().GetSnapshot() {
		sn := &snapshot{
			ID:           repoSn.GetId(),
			ConfID:       repoSn.GetSnapshotMetaInfo().GetConfId(),
			RepoSnapshot: repoSn,
		}
		ra, err := nannyClient.GetRuntimeAttributeSnapshot(ctx, ID, sn.ID)
		if err != nil {
			return nil, fmt.Errorf("failed getting runtime attrs %s: %w", sn.ID, err)
		}
		if ra.Content.Engines.EngineType != "YP_LITE" {
			return nil, nil
		}
		sn.RuntimeAttrs = ra
		specReq := &repopb.GetSnapshotInstanceSpecRequest{
			SnapshotId: sn.ID,
		}
		specRsp, err := nannyRPCClient.GetSnapshotInstanceSpec(ctx, specReq)
		if err != nil {
			return nil, fmt.Errorf("failed getting instance spec %s: %w", sn.ID, err)
		}
		sn.InstanceSpec = instancespec.InstanceSpecFactory(specRsp.GetInstanceSpec())

		tplReq := &internalpb.GetIssConfTemplateRequest{
			SnapshotId: sn.ID,
		}
		tplRsp, err := nannyRPCClient.GetIssConfTemplate(ctx, tplReq)
		if err != nil {
			if isNotFound(err) {
				return nil, nil
			}
			return nil, fmt.Errorf("failed getting iss conf template %s: %w", sn.ID, err)
		}
		sn.ISSTemplate = tplRsp.GetTemplate()
		snapshots[sn.ConfID] = sn
	}
	rv.Snapshots = snapshots
	return rv, nil
}

func getServices(ctx context.Context, IDs []string, nannyClient *nannyclient.NannyClient, nannyRPCClient *nannyrpc.NannyRPCClient) map[string]*service {
	services := make([]*service, 0, len(IDs))
	for _, ID := range IDs {
		s, err := getService(ctx, ID, nannyClient, nannyRPCClient)
		if err != nil {
			log.Errorf("failed getting service %s: %s. Skipping service", ID, err)
			continue
		}
		if s != nil {
			services = append(services, s)
		}
	}
	rv := make(map[string]*service, len(services))
	for _, s := range services {
		rv[strings.ReplaceAll(s.ID, "_", "-")] = s
	}
	return rv
}

func getPodsBatch(ctx context.Context, c *yp.Client, ct string) ([]*api.Pod, string, error) {
	rv := make([]*api.Pod, 0, api.DefaultGetObjectsBatchSize)
	req := yp.SelectPodsRequest{
		Format:    yp.PayloadFormatProto,
		Filter:    "[/labels/deploy_engine]=\"YP_LITE\"",
		Selectors: api.DefaultGetObjectsSelectors,
		Limit:     api.DefaultGetObjectsBatchSize,
	}
	if ct != "" {
		req.ContinuationToken = ct
	}
	rsp, err := c.SelectPods(ctx, req)
	if err != nil {
		return nil, ct, err
	}
	for rsp.Next() {
		p := &api.Pod{
			TPod: ypapi.TPod{
				Labels: &ytree.TAttributeDictionary{},
				Meta:   &ypapi.TPodMeta{},
				Spec:   &ypapi.TPodSpec{},
			},
		}
		err := rsp.Fill(p.Labels, p.Meta, p.Spec)
		if err != nil {
			return nil, ct, err
		}
		rv = append(rv, p)
	}
	ct = rsp.ContinuationToken()
	return rv, ct, nil
}

type specDiff struct {
	Message          string
	MismatchedConfID string
	Diff             string
}

func (d *specDiff) String() string {
	b := &bytes.Buffer{}
	if d.Message != "" && d.MismatchedConfID != "" {
		fmt.Fprintf(b, "%s: Mismatched conf: %s", d.Message, d.MismatchedConfID)
	} else if d.MismatchedConfID != "" {
		fmt.Fprintf(b, "Mismatched conf: %s", d.MismatchedConfID)
	} else {
		fmt.Fprint(b, d.Message)
	}
	title := b.String()
	if d.Diff != "" {
		return fmt.Sprintf("%s%s", title, d.Diff)
	}
	return title
}

func findLogrotateCont(rev *hqpb.InstanceRevision) *hqpb.Container {
	for _, c := range rev.GetContainer() {
		if c.GetName() == "logrotate" {
			return c
		}
	}
	return nil
}

func prepareConfs(generated *clusterapi.HostConfigurationInstance, podConf *clusterapi.HostConfigurationInstance) {
	podRev := podConf.GetInstanceRevision()
	genRev := generated.GetInstanceRevision()
	if podRev == nil || genRev == nil {
		return
	}
	podCont := findLogrotateCont(podRev)
	genCont := findLogrotateCont(genRev)
	if podCont == nil || genCont == nil {
		return
	}
	if genCont.GetResourceAllocation().GetLimit() == nil && podCont.GetResourceAllocation().GetLimit() != nil {
		if genCont.ResourceAllocation != nil {
			genCont.ResourceAllocation.Limit = podCont.GetResourceAllocation().GetLimit()
		} else {
			genCont.ResourceAllocation = &hqpb.ResourceAllocation{Limit: podCont.GetResourceAllocation().GetLimit()}
		}
	}
}

func processPod(cluster *ypcluster.ClusterInfo, pod *api.Pod, s *service, gen *confgen.Generator) *specDiff {
	for _, podConf := range pod.GetSpec().GetIss().GetInstances() {
		confID := podConf.GetId().GetConfiguration().GetGroupStateFingerprint()
		sn, ok := s.Snapshots[confID]
		if !ok {
			return &specDiff{Message: fmt.Sprintf("conf %s is not found in service snapshots", confID)}
		}
		conf, err := gen.GenerateForPod(pod, cluster, sn.ISSTemplate, sn.RuntimeAttrs, sn.InstanceSpec, sn.RepoSnapshot)
		if err != nil {
			return &specDiff{Message: fmt.Sprintf("failed generating conf for sn %s: %s", sn.ID, err)}
		}
		conf.TargetState = podConf.TargetState
		if !assert.ObjectsAreEqual(conf, podConf) {
			prepareConfs(conf, podConf)
			if !assert.ObjectsAreEqual(conf, podConf) {
				return &specDiff{
					MismatchedConfID: confID,
					Diff:             diff(podConf, conf),
				}
			}
		}
	}
	return nil
}

func fetchAllServices(ctx context.Context, nannyClient *nannyclient.NannyClient, nannyRPCClient *nannyrpc.NannyRPCClient) (map[string]*service, error) {
	serviceIDs, err := listServiceIDs(ctx, nannyRPCClient)
	if err != nil {
		return nil, err
	}
	rv := make(map[string]*service)
	for i := 0; i < len(serviceIDs); i += 100 {
		end := i + 100
		if end > len(serviceIDs) {
			end = len(serviceIDs)
		}
		for k, v := range getServices(ctx, serviceIDs[i:end], nannyClient, nannyRPCClient) {
			rv[k] = v
		}
	}
	return rv, nil
}

func processCluster(ctx context.Context, nannyClient *nannyclient.NannyClient, nannyRPCClient *nannyrpc.NannyRPCClient, ypClient *yp.Client, gen *confgen.Generator, cluster *ypcluster.ClusterInfo) {
	ct := ""
	for {
		pods, newCT, err := getPodsBatch(ctx, ypClient, ct)
		if err != nil {
			panic(fmt.Errorf("failed getting pods: %w", err))
		}
		ct = newCT
		for _, pod := range pods {
			sid, err := yputil.FindLabelString(pod.GetLabels(), "nanny_service_id")
			if err != nil {
				log.Errorf("could not get service id for pod %s: %s", pod.GetMeta().GetId(), err)
				continue
			}
			_, found := processed[sid]
			if found {
				continue
			}
			s, err := getService(ctx, sid, nannyClient, nannyRPCClient)
			if err != nil {
				log.Errorf("could not get service %s: %s", sid, err)
				continue
			}
			if s == nil {
				processed[sid] = struct{}{}
				continue
			}
			d := processPod(cluster, pod, s, gen)
			if d != nil {
				log.Infof("FAILED: %s %s [%s]: %s", s.ID, pod.GetMeta().GetId(), cluster.Name, d)
			} else {
				log.Infof("OK: %s", s.ID)
			}
			processed[s.ID] = struct{}{}
		}
		if len(pods) < api.DefaultGetObjectsBatchSize {
			break
		}
	}
}

func main() {
	flag.Parse()

	cfg := controller.Config{}
	data, err := ioutil.ReadFile(configPath)
	if err != nil {
		log.Fatalf("Error reading controller config: %s", err.Error())
	}
	err = yaml.Unmarshal(data, &cfg)
	if err != nil {
		log.Fatalf("Error parsing controller config: %s", err.Error())
	}
	cfg.SetTokensFromEnv()
	log.SetupLogger(cfg.Log)

	processed = make(map[string]struct{}, 30000)
	nannyCfg := cfg.NannyClient
	nannyClient := nannyclient.NewNannyClient(nannyCfg.URL, nannyCfg.OAuthToken, nannyCfg.RequestTimeout, nannyCfg.ConnectionTimeout)
	nannyRPCClient := nannyrpc.NewNannyRPCClient(cfg.NannyRepoRPC, cfg.NannyInternalRPC)

	cfg.ISSConfGen.NannyRepoURL = nannyRPCClient.RepoURL
	gen := confgen.NewGeneratorFromConfig(&cfg.ISSConfGen)

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	for _, cluster := range cfg.YP.Clusters {
		ypClient, err := yp.NewClient(cluster.Name, yp.WithSystemAuthToken())
		if err != nil {
			panic(err)
		}
		processCluster(ctx, nannyClient, nannyRPCClient, ypClient, gen, cluster)
	}
}

func init() {
	flag.StringVar(&configPath, "config", "", "Path to a controller config.")
}
