package main

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/log"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/server"
	"bytes"
	"context"
	"flag"
	"fmt"
	"gopkg.in/yaml.v2"
	"io/ioutil"
	clientcmdapi "k8s.io/client-go/tools/clientcmd/api"
	"net/http"
	"os"
	"os/signal"
	"strings"
	"sync"
	"syscall"
	"text/template"

	"a.yandex-team.ru/infra/allocation-ctl/pkg/controller"
	"k8s.io/client-go/tools/clientcmd"
	// Uncomment the following line to load the gcp plugin (only required to authenticate against GKE clusters).
	// _ "k8s.io/client-go/plugin/pkg/client/auth/gcp"
)

var (
	masterURL  string
	kubeconfig string
	configPath string
)

func getEnvMap() map[string]string {
	env := os.Environ()
	rv := make(map[string]string, len(env))
	for _, item := range env {
		p := strings.Split(item, "=")
		rv[p[0]] = p[1]
	}
	return rv
}

func replaceWithEnv(t *template.Template) ([]byte, error) {
	b := &bytes.Buffer{}
	if err := t.Execute(b, getEnvMap()); err != nil {
		return nil, err
	}
	return b.Bytes(), nil
}

func kubeconfigGetter() (*clientcmdapi.Config, error) {
	tpl, err := template.ParseFiles(kubeconfig)
	if err != nil {
		return nil, err
	}
	data, err := replaceWithEnv(tpl)
	if err != nil {
		return nil, err
	}
	return clientcmd.Load(data)
}

func main() {
	flag.Parse()

	kubecfg, err := clientcmd.BuildConfigFromKubeconfigGetter(masterURL, kubeconfigGetter)
	if err != nil {
		panic(fmt.Sprintf("Error building kubeconfig: %s", err.Error()))
	}

	cfg := controller.Config{}

	data, err := ioutil.ReadFile(configPath)
	if err != nil {
		panic(fmt.Sprintf("Error reading controller config: %s", err.Error()))
	}
	err = yaml.Unmarshal(data, &cfg)
	if err != nil {
		panic(fmt.Sprintf("Error parsing controller config: %s", err.Error()))
	}
	cfg.SetTokensFromEnv()

	log.SetupLogger(cfg.Log)

	// set up signals so we handle the first shutdown signal gracefully
	ctx, cancel := context.WithCancel(context.Background())
	c := make(chan os.Signal, 1)
	signal.Notify(c, os.Interrupt, syscall.SIGTERM)
	defer func() {
		signal.Stop(c)
		cancel()
	}()

	go func() {
		select {
		case <-c:
			cancel()
		case <-ctx.Done():
		}
	}()

	uc := cfg.Web.Unistat
	srv := &http.Server{Addr: fmt.Sprintf(":%d", uc.Port)}
	srvExit := &sync.WaitGroup{}
	http.HandleFunc(uc.Path, server.UnistatHandle)

	go func() {
		defer srvExit.Done()
		// always returns error. ErrServerClosed on graceful close
		if err := srv.ListenAndServe(); err != http.ErrServerClosed {
			// unexpected error. port in use?
			log.Fatalf("[web-server] ListenAndServe(): %v, err)", err)
		}
	}()

	if err = controller.Main(ctx, kubecfg, &cfg); err != nil {
		log.Info("shutting down web-server")
		if shutdownErr := srv.Shutdown(ctx); shutdownErr != nil {
			log.Errorf("[web-server] failed to shutdown: %s", shutdownErr)
			panic(err)
		}
		srvExit.Wait()
		panic(err)
	}
}

func init() {
	flag.StringVar(&kubeconfig, "kubeconfig", "", "Path to a kubeconfig. Only required if out-of-cluster.")
	flag.StringVar(&masterURL, "master", "", "The address of the Kubernetes API server. Overrides any value in kubeconfig. Only required if out-of-cluster.")
	flag.StringVar(&configPath, "config", "", "Path to a controller config.")
}
