package controller

import (
	"context"
	"fmt"
	"os"
	"time"

	clientset "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/clientset/versioned"
	informers "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/informers/externalversions"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/confgen"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/specbuilder"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/log"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/hqsender"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/servicefetcher"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	hqrpc "a.yandex-team.ru/infra/nanny/go/hq"
	nannyrpc "a.yandex-team.ru/infra/nanny/go/nanny"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/client-go/kubernetes"
	restclient "k8s.io/client-go/rest"
	"k8s.io/client-go/tools/leaderelection"
	"k8s.io/client-go/tools/leaderelection/resourcelock"
)

type nannyClientConfig struct {
	URL               string        `yaml:"url"`
	OAuthToken        string        `yaml:"oauth_token"`
	RequestTimeout    time.Duration `yaml:"request_timeout"`
	ConnectionTimeout time.Duration `yaml:"connection_timeout"`
}

type unistatConfig struct {
	Path string `yaml:"path"`
	Port int    `yaml:"port"`
}

type webConfig struct {
	Unistat unistatConfig `yaml:"unistat"`
}

type Config struct {
	NannyClient       nannyClientConfig `yaml:"nanny_client"`
	NannyRepoRPC      rpc.ClientConfig  `yaml:"nanny_repo_rpc"`
	NannyInternalRPC  rpc.ClientConfig  `yaml:"nanny_internal_rpc"`
	HQSender          hqrpc.Config      `yaml:"hq_sender"`
	YP                yp.Config         `yaml:"yp"`
	ISSConfGen        confgen.Config    `yaml:"iss_conf_gen"`
	SendSpecToHQ      bool              `yaml:"send_spec_to_hq"`
	UpdatePodsWithCAS bool              `yaml:"update_pods_with_cas"`
	WorkersCount      int               `yaml:"workers_count"`
	Web               webConfig         `yaml:"web"`
	Log               *log.Config       `yaml:"log"`
}

func (cfg *Config) SetTokensFromEnv() {
	t := os.Getenv("NANNY_OAUTH_TOKEN")
	if t == "" {
		return
	}
	if cfg.NannyClient.OAuthToken == "" {
		cfg.NannyClient.OAuthToken = t
	}
	if cfg.NannyRepoRPC.OauthToken == "" {
		cfg.NannyRepoRPC.OauthToken = t
	}
	if cfg.NannyInternalRPC.OauthToken == "" {
		cfg.NannyInternalRPC.OauthToken = t
	}
}

func Main(ctx context.Context, kubecfg *restclient.Config, cfg *Config) error {
	kubeClient, err := kubernetes.NewForConfig(kubecfg)
	if err != nil {
		return fmt.Errorf("error building kubernetes clientset: %w", err)
	}

	allocClient, err := clientset.NewForConfig(kubecfg)
	if err != nil {
		return fmt.Errorf("error building example clientset: %w", err)
	}

	allocInformerFactory := informers.NewSharedInformerFactory(allocClient, time.Second*30)

	clusterMap, err := yp.NewYpClusterMap(ctx, &cfg.YP, cfg.UpdatePodsWithCAS, allocInformerFactory.Nanny().V1alpha1().Allocations().Lister())

	if err != nil {
		return fmt.Errorf("error building YP clusters: %w", err)
	}

	nannyCfg := cfg.NannyClient

	nannyClient := nannyclient.NewNannyClient(nannyCfg.URL, nannyCfg.OAuthToken, nannyCfg.RequestTimeout, nannyCfg.ConnectionTimeout)
	nannyRPCClient := nannyrpc.NewNannyRPCClient(cfg.NannyRepoRPC, cfg.NannyInternalRPC)
	nannyFetcher := servicefetcher.NewServiceFetcher(nannyClient, nannyRPCClient)

	cfg.ISSConfGen.NannyRepoURL = nannyRPCClient.RepoURL
	gen := confgen.NewGeneratorFromConfig(&cfg.ISSConfGen)
	builder := &specbuilder.Builder{}

	hqSender := hqsender.NewSenderFromConfig(&cfg.HQSender)

	ctl := NewController(kubeClient,
		allocClient,
		allocInformerFactory.Nanny().V1alpha1().Allocations(),
		clusterMap,
		cfg.UpdatePodsWithCAS,
		nannyClient,
		nannyRPCClient,
		nannyFetcher,
		gen,
		builder,
		hqSender,
		cfg.SendSpecToHQ)

	hostname, err := os.Hostname()
	if err != nil {
		return fmt.Errorf("error getting os.Hostname(): %w", err)
	}

	var lock = &resourcelock.LeaseLock{
		LeaseMeta: metav1.ObjectMeta{
			Name:      "allocation-ctl-lock",
			Namespace: "default",
		},
		Client: kubeClient.CoordinationV1(),
		LockConfig: resourcelock.ResourceLockConfig{
			Identity: hostname,
		},
	}
	leaderelection.RunOrDie(ctx, leaderelection.LeaderElectionConfig{
		Lock:            lock,
		ReleaseOnCancel: true,
		LeaseDuration:   15 * time.Second,
		RenewDeadline:   10 * time.Second,
		RetryPeriod:     2 * time.Second,
		Callbacks: leaderelection.LeaderCallbacks{
			OnStartedLeading: func(ctx context.Context) {
				// notice that there is no need to run Start methods in a separate goroutine. (i.e. go allocInformerFactory.Start(stopCh)
				// Start method is non-blocking and runs all registered informers in a dedicated goroutine.
				allocInformerFactory.Start(ctx.Done())
				if err = ctl.Run(ctx, cfg.WorkersCount); err != nil {
					log.Fatalf("error running controller: %s", err.Error())
				}
			},
			OnStoppedLeading: func() {
				log.Infof("leader lost: %s", hostname)
				os.Exit(0)
			},
			OnNewLeader: func(identity string) {
				if identity == hostname {
					return
				}
				log.Infof("new leader elected: %s", identity)
			},
		},
	})
	return nil
}
