package controller

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/apis/allocationctl/v1alpha1"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/controller/events"
	clientset "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/clientset/versioned"
	allocscheme "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/clientset/versioned/scheme"
	informers "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/informers/externalversions/allocationctl/v1alpha1"
	listers "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/listers/allocationctl/v1alpha1"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/confgen"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/specbuilder"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/log"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/hqsender"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/servicefetcher"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/yputil"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	nannyrpc "a.yandex-team.ru/infra/nanny/go/nanny"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	"context"
	"fmt"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/client-go/kubernetes"
	"k8s.io/client-go/kubernetes/scheme"
	typedcorev1 "k8s.io/client-go/kubernetes/typed/core/v1"
	"k8s.io/client-go/tools/cache"
	"k8s.io/client-go/tools/record"
	"k8s.io/client-go/util/workqueue"
	"strings"
	"time"
)

const controllerAgentName = "alloc-ctl"

// Controller is the controller implementation for Allocation resources
type Controller struct {
	// kubeclientset is a standard kubernetes clientset
	kubeclientset kubernetes.Interface
	// sampleclientset is a clientset for our own API group
	allocclientset clientset.Interface

	allocsLister      listers.AllocationLister
	allocsSynced      cache.InformerSynced
	deploymentsSynced cache.InformerSynced

	ypClusters        yp.YpClusterMap
	updatePodsWithCAS bool

	nannyClient    *nannyclient.NannyClient
	nannyRPCClient *nannyrpc.NannyRPCClient
	nannyFetcher   *servicefetcher.ServiceFetcher
	issConfGen     *confgen.Generator
	issSpecBuilder *specbuilder.Builder
	hqSender       *hqsender.Sender
	sendSpecToHQ   bool

	// workqueue is a rate limited work queue. This is used to queue work to be
	// processed instead of performing it as soon as a change happens. This
	// means we can ensure we only process a fixed amount of resources at a
	// time, and makes it easy to ensure we are never processing the same item
	// simultaneously in two different workers.
	workqueue workqueue.RateLimitingInterface
	// recorder is an event recorder for recording Event resources to the
	// Kubernetes API.
	recorder record.EventRecorder
}

func waitUntil(f func(), period time.Duration, stopCh <-chan struct{}) {
	for {
		select {
		case <-stopCh:
			return
		default:
		}

		f()

		t := time.NewTimer(period)
		select {
		case <-stopCh:
			return
		case <-t.C:
		}
	}
}

// NewController returns a new allocation controller
func NewController(
	kubeclientset kubernetes.Interface,
	allocclientset clientset.Interface,
	allocInformer informers.AllocationInformer,
	ypClusters yp.YpClusterMap,
	updatePodsWithCAS bool,
	nannyClient *nannyclient.NannyClient,
	nannyRPCClient *nannyrpc.NannyRPCClient,
	nannyFetcher *servicefetcher.ServiceFetcher,
	issConfGen *confgen.Generator,
	issSpecBuilder *specbuilder.Builder,
	hqSender *hqsender.Sender,
	sendSpecToHQ bool) *Controller {

	// Create event broadcaster
	// Add alloction-ctl types to the default Kubernetes Scheme so Events can be
	// logged for allocation-ctl types.
	if err := allocscheme.AddToScheme(scheme.Scheme); err != nil {
		panic(err)
	}
	log.Info("creating event broadcaster")
	eventBroadcaster := record.NewBroadcaster()
	eventBroadcaster.StartStructuredLogging(0)
	eventBroadcaster.StartRecordingToSink(&typedcorev1.EventSinkImpl{Interface: kubeclientset.CoreV1().Events("")})
	recorder := eventBroadcaster.NewRecorder(scheme.Scheme, corev1.EventSource{Component: controllerAgentName})

	workqueue.SetProvider(&UnistatMetricsProvider{})
	wq := workqueue.NewNamedRateLimitingQueue(workqueue.DefaultControllerRateLimiter(), "allocs")
	controller := &Controller{
		kubeclientset:     kubeclientset,
		allocclientset:    allocclientset,
		allocsLister:      allocInformer.Lister(),
		allocsSynced:      allocInformer.Informer().HasSynced,
		ypClusters:        ypClusters,
		updatePodsWithCAS: updatePodsWithCAS,
		nannyClient:       nannyClient,
		nannyRPCClient:    nannyRPCClient,
		nannyFetcher:      nannyFetcher,
		issConfGen:        issConfGen,
		issSpecBuilder:    issSpecBuilder,
		hqSender:          hqSender,
		sendSpecToHQ:      sendSpecToHQ,
		workqueue:         wq,
		recorder:          recorder,
	}

	log.Info("setting up event handlers")
	// Set up an event handler for when Allocation resources change
	allocInformer.Informer().AddEventHandler(cache.ResourceEventHandlerFuncs{
		AddFunc: controller.enqueueAllocation,
		UpdateFunc: func(old, new interface{}) {
			controller.enqueueAllocation(new)
		},
	})

	return controller
}

// Run will set up the event handlers for types we are interested in, as well
// as syncing informer caches and starting workers. It will block until stopCh
// is closed, at which point it will shutdown the workqueue and wait for
// workers to finish processing their current work items.
func (c *Controller) Run(ctx context.Context, threadiness int) error {
	defer c.workqueue.ShutDown()

	// Start the informer factories to begin populating the informer caches
	log.Info("starting Allocation controller")

	// Wait for the caches to be synced before starting workers
	log.Info("waiting for allocations informer cache to sync")
	if ok := cache.WaitForCacheSync(ctx.Done(), c.allocsSynced); !ok {
		return fmt.Errorf("failed to wait for allocations cache to sync")
	}

	c.ypClusters.Run(ctx)
	log.Info("waiting for YP storage caches to sync")
	if ok := cache.WaitForCacheSync(ctx.Done(), c.ypClusters.HasSynced); !ok {
		return fmt.Errorf("failed to wait for YP storage caches to sync")
	}

	log.Info("starting workers")
	// Launch workers to process Allocation resources

	for i := 0; i < threadiness; i++ {
		go waitUntil(func() {
			c.runWorker(ctx)
		}, time.Second, ctx.Done())
	}

	log.Info("started workers")
	<-ctx.Done()
	log.Info("shutting down workers")

	return nil
}

// Records event with stringify err and returns wrapped err.
func (c *Controller) recordError(object runtime.Object, err error, eventtype, reason, messageFmt string, args ...interface{}) error {
	args = append(args, err)
	c.recorder.Eventf(object, eventtype, reason, messageFmt+": %s", args...)
	return fmt.Errorf(messageFmt+": %w", args...)
}

// runWorker is a long-running function that will continually call the
// processNextWorkItem function in order to read and process a message on the
// workqueue.
func (c *Controller) runWorker(ctx context.Context) {
	for c.processNextWorkItem(ctx) {
	}
}

// processNextWorkItem will read a single work item off the workqueue and
// attempt to process it, by calling the syncHandler.
func (c *Controller) processNextWorkItem(ctx context.Context) bool {
	obj, shutdown := c.workqueue.Get()

	if shutdown {
		return false
	}

	// We wrap this block in a func so we can defer c.workqueue.Done.
	err := func(obj interface{}) error {
		// We call Done here so the workqueue knows we have finished
		// processing this item. We also must remember to call Forget if we
		// do not want this work item being re-queued. For example, we do
		// not call Forget if a transient error occurs, instead the item is
		// put back on the workqueue and attempted again after a back-off
		// period.
		defer c.workqueue.Done(obj)
		var key string
		var ok bool
		// We expect strings to come off the workqueue. These are of the
		// form namespace/name. We do this as the delayed nature of the
		// workqueue means the items in the informer cache may actually be
		// more up to date that when the item was initially put onto the
		// workqueue.
		if key, ok = obj.(string); !ok {
			// As the item in the workqueue is actually invalid, we call
			// Forget here else we'd go into a loop of attempting to
			// process a work item that is invalid.
			c.workqueue.Forget(obj)
			log.Errorf("expected string in workqueue but got %#v", obj)
			return nil
		}
		// Run the syncHandler, passing it the namespace/name string of the
		// Allocation resource to be synced.
		if err := c.syncHandler(ctx, key); err != nil {
			// Maybe it's better to put the item back on the workqueue to handle any transient errors,
			// but for now we rely on the next Resync event and just wait for it.
			// c.workqueue.AddRateLimited(key)
			c.workqueue.Forget(obj)
			return fmt.Errorf("error syncing %s: %w", key, err)
		}
		// Finally, if no error occurs we Forget this item so it does not
		// get queued again until another change happens.
		c.workqueue.Forget(obj)
		log.Infof("successfully synced %s", key)
		return nil
	}(obj)

	if err != nil {
		log.Errorf(err.Error())
		return true
	}

	return true
}

func isPodInAllocation(pod *api.Pod, cluster string, alloc *v1alpha1.Allocation) bool {
	for _, p := range alloc.Spec.YpPodIds.Pods {
		c, err := castPodClusterToYP(p.Cluster)
		if err != nil {
			return false
		}
		if c == cluster && p.Id == pod.GetMeta().GetId() {
			return true
		}
	}
	return false
}

func castPodClusterToYP(c string) (string, error) {
	lc := strings.ToLower(c)
	switch lc {
	case "test_sas":
		return "sas-test", nil
	case "man_pre":
		return "man-pre", nil
	case "sas", "man", "vla", "iva", "myt", "xdc":
		return lc, nil
	default:
		return "", fmt.Errorf("unknown YP cluster %s", c)
	}
}

type putHQRev struct {
	conf    *clusterapi.HostConfigurationInstance
	pod     *api.Pod
	service *servicefetcher.Service // for validation purpose
}

type deleteHQInstance struct {
	instanceID string
	pod        *api.Pod
	service    *servicefetcher.Service // for validation purpose
}

// syncHandler compares the actual state with the desired, and attempts to
// converge the two. It then updates the Status block of the Allocation resource
// with the current status of the resource.
func (c *Controller) syncHandler(ctx context.Context, key string) error {
	// Convert the namespace/name string into a distinct namespace and name
	namespace, name, err := cache.SplitMetaNamespaceKey(key)
	if err != nil {
		log.Errorf("invalid resource key: %s", key)
		return nil
	}

	// Get the Allocation resource with this namespace/name
	alloc, err := c.allocsLister.Allocations(namespace).Get(name)
	if err != nil {
		// The Allocation resource may no longer exist, in which case we stop
		// processing.
		if errors.IsNotFound(err) {
			log.Errorf("allocation %s in work queue no longer exists", key)
			return nil
		}
		return err
	}

	var service *servicefetcher.Service = nil

	for clusterName, cluster := range c.ypClusters {
		pods, err := cluster.PodIndexer.ByIndex("allocID", alloc.Name)
		if err != nil {
			return fmt.Errorf("failed to find pods by allocation ID %s: %w", alloc.Name, err)
		}
		log.Infof("allocation %s has %d pods in cluster %s", key, len(pods), clusterName)
		toFill := make([]*api.Pod, 0, len(pods))
		toReset := make([]*api.Pod, 0, len(pods))
		toHQCreate := make([]*putHQRev, 0, len(pods)*3) // 3 is an approximation for average number of snapshots in pod
		toHQDelete := make([]*deleteHQInstance, 0, len(pods))
		for _, p := range pods {
			pod, ok := p.(*api.Pod)
			if !ok {
				return fmt.Errorf("failed to cast stored object to Pod")
			}
			if service == nil {
				sID := alloc.Labels["nanny.apiserver.service_id"]
				service, err = c.nannyFetcher.FetchService(ctx, sID)
				if err != nil {
					return c.recordError(alloc, err, corev1.EventTypeWarning, events.FailedGetNannyService, "failed getting service %s from Nanny", sID)
				}
			}
			// Service is not managed by Allocation, skip service
			if !service.ManagedByAllocation() {
				return nil
			}
			if !isPodInAllocation(pod, clusterName, alloc) {
				if yputil.HasPodISSInstances(pod) {
					// Allocation does not contain pod and pod has ISS spec, reset pod
					toReset = append(toReset, yputil.MakePodToUpdateISSSpec(pod, &clusterapi.HostConfiguration{}))
					d := &deleteHQInstance{
						instanceID: c.issConfGen.MakeHQInstanceID(cluster.Info, service.ID, pod.GetMeta().GetId()),
						pod:        pod,
						service:    service,
					}
					toHQDelete = append(toHQDelete, d)
				}
				continue
			}
			if yputil.HasPodISSInstances(pod) {
				// Allocation contains pod and pod has ISS spec, skip pod
				continue
			}
			// Allocation contains pod and pod has no ISS spec, fill pod
			podID := pod.GetMeta().GetId()
			confs := make([]*clusterapi.HostConfigurationInstance, 0, len(service.Snapshots))
			for _, sn := range service.Snapshots {
				ct, err := sn.RuntimeAttrs.Content.Instances.ChosenTypeMapped()
				if err != nil {
					// Let's skip unknown chosen type, because it's not clear how to fill them
					continue
				}
				if ct != nannyclient.InstanceTypeAllocationPods {
					// We *must* fill only snapshots on allocations to avoid bug described in SWAT-7998
					continue
				}
				conf, err := c.issConfGen.GenerateForPod(pod, cluster.Info, sn.ISSTemplate, sn.RuntimeAttrs, sn.InstanceSpec, sn.RepoSnapshot)
				if err != nil {
					return c.recordError(alloc, err, corev1.EventTypeWarning, events.FailedGenerateISSConf, "failed generating ISS conf for pod %s, snapshot %s", podID, sn.ID)
				}
				if alloc.Spec.ISSActiveTargetStateReplacement != "" && sn.RepoSnapshot.Target == repopb.Snapshot_ACTIVE {
					conf.TargetState = alloc.Spec.ISSActiveTargetStateReplacement
				}
				confs = append(confs, conf)
				if sn.InstanceSpec.GetInstanceSpec().GetHqPolicy() == repopb.InstanceSpec_HQ_ENABLED && conf.GetInstanceRevision() != nil {
					toHQCreate = append(toHQCreate, &putHQRev{conf, pod, service})
				}
			}
			issSpec, err := c.issSpecBuilder.Build(confs)
			if err != nil {
				return c.recordError(alloc, err, corev1.EventTypeWarning, events.FailedBuildISSSpec, "failed building ISS spec for pod %s", podID)
			}
			toFill = append(toFill, yputil.MakePodToUpdateISSSpec(pod, issSpec))
		}
		if len(toFill) != 0 {
			log.Infof("%s: filling ISS spec for %d pods in cluster %s", alloc.Name, len(toFill), clusterName)
			if err := cluster.Client.Pods().UpdateISSSpecList(ctx, toFill, c.updatePodsWithCAS); err != nil {
				return c.recordError(alloc, err, corev1.EventTypeWarning, events.FailedUpdatePods, "failed filling %d pods in cluster %s", len(toFill), clusterName)
			} else if len(toFill) != 0 {
				c.recorder.Eventf(alloc, corev1.EventTypeNormal, events.UpdatedPods, "filled %d pods in cluster %s", len(toFill), clusterName)
			}
		}
		if len(toReset) != 0 {
			log.Infof("%s: resetting ISS spec for %d pods in cluster %s", alloc.Name, len(toReset), clusterName)
			if err := cluster.Client.Pods().UpdateISSSpecList(ctx, toReset, c.updatePodsWithCAS); err != nil {
				return c.recordError(alloc, err, corev1.EventTypeWarning, events.FailedUpdatePods, "failed resetting %d pods in cluster %s", len(toReset), clusterName)
			} else if len(toFill) != 0 {
				c.recorder.Eventf(alloc, corev1.EventTypeNormal, events.UpdatedPods, "reset %d pods in cluster %s", len(toFill), clusterName)
			}
		}
		if c.sendSpecToHQ {
			if len(toHQCreate) != 0 {
				log.Infof("[%s] sending %d instance revisions to HQ", cluster.Info.Location, len(toHQCreate))
				for _, hqCrt := range toHQCreate {
					if err = c.hqSender.SendRevision(ctx, cluster.Info.Location, hqCrt.conf, hqCrt.pod, hqCrt.service); err != nil {
						log.Errorf("[%s] failed to send instance revision %s to HQ: %s", cluster.Info.Location, hqCrt.conf.GetInstanceRevision().GetId(), err)
					}
				}
			}
			if len(toHQDelete) != 0 {
				log.Infof("deleting %d instances from HQ", len(toHQDelete))
				for _, hqDel := range toHQDelete {
					if err = c.hqSender.DeleteInstance(ctx, cluster.Info.Location, hqDel.instanceID, hqDel.pod, hqDel.service); err != nil {
						log.Errorf("failed to delete instance %s from HQ: %s", hqDel.instanceID, err)
					}
				}
			}
		}
	}
	log.Infof("sync handler for allocation %s is finished", key)
	return nil
}

// enqueueAllocation takes an Allocation resource and converts it into a namespace/name
// string which is then put onto the work queue. This method should *not* be
// passed resources of any type other than Allocation.
func (c *Controller) enqueueAllocation(obj interface{}) {
	var key string
	var err error
	if key, err = cache.MetaNamespaceKeyFunc(obj); err != nil {
		log.Errorf(err.Error())
		return
	}
	c.workqueue.Add(key)
}
