package controller

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/metricsutil"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
	"fmt"
	"k8s.io/client-go/util/workqueue"
)

// implemets workqueue.GaugeMetric interface
type Gauge struct {
	numeric *unistat.Numeric
}

func NewGauge(name string, aggr unistat.Aggregation) *Gauge {
	return &Gauge{
		numeric: unistat.NewNumeric(name, 0, aggr, unistat.Sum),
	}
}

func (g *Gauge) Inc() {
	g.numeric.Update(1)
}

func (g *Gauge) Dec() {
	g.numeric.Update(-1)
}

// implemets workqueue.SettableGaugeMetric interface
type SettableGauge struct {
	numeric *unistat.Numeric
}

func NewSettableGauge(name string) *SettableGauge {
	return &SettableGauge{
		numeric: unistat.NewNumeric(name, 0, aggr.Absolute(), unistat.Last),
	}
}

func (sg *SettableGauge) Set(v float64) {
	sg.numeric.Update(v)
}

// implemets workqueue.GaugeMetric interface
type Counter struct {
	numeric *unistat.Numeric
}

func NewCounter(name string) *Counter {
	return &Counter{
		numeric: unistat.NewNumeric(name, 0, aggr.Counter(), unistat.Sum),
	}
}

func (c *Counter) Inc() {
	c.numeric.Update(1)
}

// implements workqueue.HistogramMetric
type Histogram struct {
	histogram *unistat.Histogram
}

func NewHistogram(name string, intervals []float64) *Histogram {
	return &Histogram{
		histogram: unistat.NewHistogram(name, 0, aggr.Histogram(), intervals),
	}
}

func (h *Histogram) Observe(v float64) {
	h.histogram.Update(v)
}

// implements workqueue.MetricsProvider
type UnistatMetricsProvider struct {
}

func (mp *UnistatMetricsProvider) NewDepthMetric(name string) workqueue.GaugeMetric {
	g := NewGauge(fmt.Sprintf("%s-wq-depth", name), aggr.Absolute())
	unistat.Register(g.numeric)
	return g
}

func (mp *UnistatMetricsProvider) NewAddsMetric(name string) workqueue.CounterMetric {
	c := NewCounter(fmt.Sprintf("%s-wq-adds", name))
	unistat.Register(c.numeric)
	return c
}

func (mp *UnistatMetricsProvider) NewLatencyMetric(name string) workqueue.HistogramMetric {
	h := NewHistogram(fmt.Sprintf("%s-wq-latency", name), metricsutil.DefaultHistogramIntervalsSeconds)
	unistat.Register(h.histogram)
	return h
}

func (mp *UnistatMetricsProvider) NewWorkDurationMetric(name string) workqueue.HistogramMetric {
	h := NewHistogram(fmt.Sprintf("%s-wq-work-duration", name), metricsutil.DefaultHistogramIntervalsSeconds)
	unistat.Register(h.histogram)
	return h
}

func (mp *UnistatMetricsProvider) NewUnfinishedWorkSecondsMetric(name string) workqueue.SettableGaugeMetric {
	sg := NewSettableGauge(fmt.Sprintf("%s-wq-unfinished-work-secs", name))
	unistat.Register(sg.numeric)
	return sg
}

func (mp *UnistatMetricsProvider) NewLongestRunningProcessorSecondsMetric(name string) workqueue.SettableGaugeMetric {
	sg := NewSettableGauge(fmt.Sprintf("%s-wq-longest-processor-secs", name))
	unistat.Register(sg.numeric)
	return sg
}

func (mp *UnistatMetricsProvider) NewRetriesMetric(name string) workqueue.CounterMetric {
	c := NewCounter(fmt.Sprintf("%s-wq-retries", name))
	unistat.Register(c.numeric)
	return c
}
