package instancespec

import repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"

type Interface interface {
	GetHostProvidedDaemons() []*repopb.HostProvidedDaemon
	GetAUXDaemons() []*repopb.AuxDaemon
	GetInstanceAccess() *repopb.InstanceAccess
	GetInstanceCtl() *repopb.InstanceCtl
	GetLayersConfig() *repopb.LayersConfig
	IsHQPollEnabled() bool
	GetHQTags() []string
	GetNetworkProperties() *repopb.ContainerNetworkProperties
	IsDockerInstance() bool
	GetInitContainers() []*repopb.Container
	GetVolume() []*repopb.Volume
	GetNotifyAction() *repopb.NotifyAction
	GetInstanceSpec() *repopb.InstanceSpec
}

type InstanceSpec struct {
	instanceSpec *repopb.InstanceSpec
}

func (s *InstanceSpec) GetHostProvidedDaemons() []*repopb.HostProvidedDaemon {
	return s.instanceSpec.GetHostProvidedDaemons()
}

func (s *InstanceSpec) GetAUXDaemons() []*repopb.AuxDaemon {
	return s.instanceSpec.GetAuxDaemons()
}

func (s *InstanceSpec) GetInstanceAccess() *repopb.InstanceAccess {
	return s.instanceSpec.GetInstanceAccess()
}

func (s *InstanceSpec) GetInstanceCtl() *repopb.InstanceCtl {
	return s.instanceSpec.GetInstancectl()
}

func (s *InstanceSpec) GetLayersConfig() *repopb.LayersConfig {
	return s.instanceSpec.GetLayersConfig()
}

func (s *InstanceSpec) GetHQTags() []string {
	t := make([]string, 0, 3)
	for _, c := range s.instanceSpec.GetContainers() {
		if len(c.GetCommand()) != 0 {
			t = append(t, "use_hq_spec")
			break
		}
	}
	if len(t) == 0 {
		for _, c := range s.instanceSpec.GetInitContainers() {
			if len(c.GetCommand()) != 0 {
				t = append(t, "use_hq_spec")
				break
			}
		}
	}
	t = append(t, "enable_hq_report")
	if s.IsHQPollEnabled() || len(s.instanceSpec.GetAuxDaemons()) > 0 {
		t = append(t, "enable_hq_poll")
	}
	return t
}

func (s *InstanceSpec) IsHQPollEnabled() bool {
	return len(s.instanceSpec.GetContainers()) > 0 || len(s.instanceSpec.GetVolume()) > 0
}

func (s *InstanceSpec) GetNetworkProperties() *repopb.ContainerNetworkProperties {
	return s.instanceSpec.GetNetworkProperties()
}

func (s *InstanceSpec) IsDockerInstance() bool {
	return s.instanceSpec.GetType() == repopb.InstanceSpec_DOCKER_IMAGE || s.instanceSpec.GetType() == repopb.InstanceSpec_DOCKER_LAYERS
}

func (s *InstanceSpec) GetInitContainers() []*repopb.Container {
	return s.instanceSpec.GetInitContainers()
}

func (s *InstanceSpec) GetVolume() []*repopb.Volume {
	return s.instanceSpec.GetVolume()
}

func (s *InstanceSpec) GetNotifyAction() *repopb.NotifyAction {
	return s.instanceSpec.GetNotifyAction()
}

func (s *InstanceSpec) GetInstanceSpec() *repopb.InstanceSpec {
	return s.instanceSpec
}

type OSContainerInstanceSpec struct {
	instanceSpec *repopb.InstanceSpec
}

func (s *OSContainerInstanceSpec) GetHostProvidedDaemons() []*repopb.HostProvidedDaemon {
	return s.instanceSpec.OsContainerSpec.GetHostProvidedDaemons()
}

func (s *OSContainerInstanceSpec) GetAUXDaemons() []*repopb.AuxDaemon {
	return s.instanceSpec.OsContainerSpec.GetAuxDaemons()
}

func (s *OSContainerInstanceSpec) GetInstanceAccess() *repopb.InstanceAccess {
	return s.instanceSpec.OsContainerSpec.GetInstanceAccess()
}

func (s *OSContainerInstanceSpec) GetInstanceCtl() *repopb.InstanceCtl {
	return s.instanceSpec.OsContainerSpec.GetInstancectl()
}

func (s *OSContainerInstanceSpec) GetLayersConfig() *repopb.LayersConfig {
	return s.instanceSpec.OsContainerSpec.GetLayersConfig()
}

func (s *OSContainerInstanceSpec) GetHQTags() []string {
	return []string{"enable_hq_poll", "enable_hq_report", "use_hq_spec"}
}

func (s *OSContainerInstanceSpec) IsHQPollEnabled() bool {
	return true
}

func (s *OSContainerInstanceSpec) IsDockerInstance() bool {
	return s.instanceSpec.GetType() == repopb.InstanceSpec_DOCKER_IMAGE ||
		s.instanceSpec.GetType() == repopb.InstanceSpec_DOCKER_LAYERS ||
		s.instanceSpec.OsContainerSpec.GetType() == repopb.OsContainerSpec_DOCKER_LAYERS
}

func (s *OSContainerInstanceSpec) GetNetworkProperties() *repopb.ContainerNetworkProperties {
	return s.instanceSpec.OsContainerSpec.GetNetworkProperties()
}

func (s *OSContainerInstanceSpec) GetInitContainers() []*repopb.Container {
	return s.instanceSpec.OsContainerSpec.GetInitContainers()
}

func (s *OSContainerInstanceSpec) GetVolume() []*repopb.Volume {
	return s.instanceSpec.OsContainerSpec.GetVolume()
}

func (s *OSContainerInstanceSpec) GetNotifyAction() *repopb.NotifyAction {
	return s.instanceSpec.OsContainerSpec.GetNotifyAction()
}

func (s *OSContainerInstanceSpec) GetInstanceSpec() *repopb.InstanceSpec {
	return s.instanceSpec
}

func InstanceSpecFactory(s *repopb.InstanceSpec) Interface {
	if s.Type == repopb.InstanceSpec_OS_CONTAINER {
		return &OSContainerInstanceSpec{instanceSpec: s}
	}
	return &InstanceSpec{instanceSpec: s}
}
