package confgen

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/ictlversions"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/instancespec"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/hqcaster"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/issutil"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/nannyutil"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/yputil"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	internalpb "a.yandex-team.ru/infra/nanny/go/proto/nanny_internal"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	hqpb "a.yandex-team.ru/yp/go/proto/hq"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"fmt"
	"google.golang.org/protobuf/proto"
	"strconv"
	"strings"
)

const (
	defaultDiskQuotaBind               = 99999 * 1024 * 1024 * 1024
	skynetVolumePath                   = "/Berkanavt/supervisor"
	ypLiteBackboneIPPlaceholder        = "%%BACKBONE_IP_ADDRESS%%"
	DefaultYPInstancePort              = 80
	yasmAgentMTNPort                   = 11003
	jugglerAgentMTNPort                = 31579
	tagITypeRTCSmall                   = "a_itype_rtcsmall"
	tagPrjNone                         = "a_prj_none"
	genCfgPortRange                    = 20
	defaultCPUWeight                   = "1"
	defaultYPULimit                    = "memlock: 549755813888 549755813888;"
	defaultYPULimitHighNofileHardLimit = defaultYPULimit + " nofile: 102400 1000000;"
	slotMemoryMargin                   = 16 * 1024 * 1024
	nat64ResolvConfContent             = `search search.yandex.net
nameserver 2a02:6b8:0:3400::5005
nameserver 2a02:6b8::1:1
nameserver 2a02:6b8:0:3400::1`
	nat64LocalResolvConfContent = `search search.yandex.net
nameserver fd64::1
nameserver 2a02:6b8:0:3400::5005
options attempts:1 timeout:1
`
)

var issHooks = []string{
	"iss_hook_start",
	"iss_hook_stop",
	"iss_hook_status",
	"iss_hook_install",
	"iss_hook_uninstall",
	"iss_hook_validate",
	"iss_hook_notify",
	"iss_hook_reopenlogs",
}

var mainISSHooks = []string{
	"iss_hook_start",
	"iss_hook_install",
	"iss_hook_uninstall",
}

var inParallelISSHooks = []string{
	"iss_hook_stop",
	"iss_hook_status",
	"iss_hook_validate",
	"iss_hook_notify",
	"iss_hook_reopenlogs",
}

func contains(s string, a []string) bool {
	for _, e := range a {
		if e == s {
			return true
		}
	}
	return false
}

func max(x, y uint64) uint64 {
	if x > y {
		return x
	}
	return y
}

type hooksTimeLimitsCfg struct {
	Default *nannyclient.ISSHookTimeLimits `yaml:"default"`
}

type Config struct {
	NannyRepoURL                 string              `yaml:"nanny_repo_url"`
	LogsMountPoint               string              `yaml:"logs_mount_point"`
	HooksTimeLimits              *hooksTimeLimitsCfg `yaml:"hooks_time_limits"`
	StrippedHQInstanceIDServices []string            `yaml:"stripped_hq_instance_id_services"`
	StrippedHQIDYPClusters       []string            `yaml:"stripped_hq_instance_id_yp_clusters"`
	InstanceSpecToHQCaster       *hqcaster.Config    `yaml:"instance_spec_to_hq_caster"`
}

type networkSettings struct {
	Hostname   string
	BackboneIP string
}

func newNetworkSettingsFromPod(pod *api.Pod, cluster *yp.ClusterInfo, ictlTaskID int) (*networkSettings, error) {
	var bbIP string
	if ictlTaskID >= ictlversions.MinYPLiteNodeAgnosticPayload {
		bbIP = ypLiteBackboneIPPlaceholder
	} else {
		ip, found := yputil.GetPodBackboneIP(pod)
		if !found {
			return nil, fmt.Errorf("failed generating BACKBONE_IP_ADDRESS ISS property: pod %s has no backbone IP address allocated by YP", pod.GetMeta().GetId())
		}
		bbIP = ip
	}
	s := &networkSettings{
		Hostname:   yputil.MakePodPersistentHostname(pod.GetMeta().GetId(), cluster.Name, cluster.HostnameSuffix),
		BackboneIP: bbIP,
	}
	return s, nil
}

func getStorageClass(r *ypapi.TPodSpec_TDiskVolumeRequest) string {
	if r.GetStorageClass() == "ssd" {
		return "/ssd"
	}
	return "/place"
}

func isYTRBind(b *repopb.HostPathBind) bool {
	return b.Mode == repopb.HostPathBind_VOLUME && b.Path == "/yt" && b.MountPath == "/yt"
}

func isAllowedRBind(b *repopb.HostPathBind) bool {
	if isYTRBind(b) {
		return true
	}
	return b.Mode == repopb.HostPathBind_VOLUME && b.Path == "/basesearch" && b.MountPath == "/basesearch"
}

func hasYTRBind(spec *repopb.InstanceSpec) bool {
	for _, b := range spec.GetLayersConfig().GetBind() {
		if isYTRBind(b) {
			return true
		}
	}
	return false
}

func castTimeLimitsToCAPI(l *nannyclient.ISSHookTimeLimits, defaultLimits *clusterapi.TimeLimit) *clusterapi.TimeLimit {
	if l == nil && defaultLimits == nil {
		return nil
	}
	rv := &clusterapi.TimeLimit{}
	if l != nil && l.MinRestartPeriod != nil {
		rv.MinRestartPeriodMs = *l.MinRestartPeriod * 1000
	} else if defaultLimits != nil {
		rv.MinRestartPeriodMs = defaultLimits.MinRestartPeriodMs
	}
	if l != nil && l.MaxRestartPeriod != nil {
		rv.MaxRestartPeriodMs = *l.MaxRestartPeriod * 1000
	} else if defaultLimits != nil {
		rv.MaxRestartPeriodMs = defaultLimits.MaxRestartPeriodMs
	}
	if l != nil && l.RestartPeriodBackoff != nil {
		rv.RestartPeriodBackOff = *l.RestartPeriodBackoff
	} else if defaultLimits != nil {
		rv.RestartPeriodBackOff = defaultLimits.RestartPeriodBackOff
	}
	if l != nil && l.MaxRestartPeriod != nil {
		rv.MaxRestartPeriodMs = *l.MaxRestartPeriod * 1000
	} else if defaultLimits != nil {
		rv.MaxRestartPeriodMs = defaultLimits.MaxRestartPeriodMs
	}
	if l != nil && l.MaxExecutionTime != nil {
		rv.MaxExecutionTimeMs = *l.MaxExecutionTime * 1000
	} else if defaultLimits != nil {
		rv.MaxExecutionTimeMs = defaultLimits.MaxExecutionTimeMs
	}
	if l != nil && l.RestartPeriodScale != nil {
		rv.RestartPeriodScaleMs = *l.RestartPeriodScale * 1000
	} else if defaultLimits != nil {
		rv.RestartPeriodScaleMs = defaultLimits.RestartPeriodScaleMs
	}
	return rv
}

func validateOrthogonalTags(ra *nannyclient.RuntimeAttrs) error {
	tags := ra.Content.Instances.YPPodIDs.OrthogonalTags
	if tags.IType == "" {
		return fmt.Errorf("instance tag itype cannot be empty")
	}
	if tags.CType == "" {
		return fmt.Errorf("instance tag ctype cannot be empty")
	}
	if tags.Prj == "" {
		return fmt.Errorf("instance tag prj cannot be empty")
	}
	if tags.MetaPrj == "" {
		return fmt.Errorf("instance tag metaprj cannot be empty")
	}
	return nil
}

func validateITags(tags []string) error {
	if contains(tagITypeRTCSmall, tags) && contains(tagPrjNone, tags) {
		return fmt.Errorf("unistat endpoints cannot be used itype=%s and prj=%s: https://st.yandex-team.ru/GOLOVANSUPPORT-5", tagITypeRTCSmall, tagPrjNone)
	}
	return nil
}

// Generator generates ISS conf (HostConfigurationInstance) by the given pod, runtime attrs and instance spec.
type Generator struct {
	containerAccessURL           string
	logMountPoint                string
	defaultHooksTimeLimits       *clusterapi.TimeLimit
	strippedHQInstanceIDServices []string
	strippedHQIDYPClusters       []string
	hqCaster                     *hqcaster.Caster
}

func NewGeneratorFromConfig(cfg *Config) *Generator {
	url := fmt.Sprintf("%s/CheckContainerAccess/", cfg.NannyRepoURL)
	defaultHooksTimeLimits := castTimeLimitsToCAPI(cfg.HooksTimeLimits.Default, nil)
	t := &Generator{
		containerAccessURL:           url,
		logMountPoint:                cfg.LogsMountPoint,
		defaultHooksTimeLimits:       defaultHooksTimeLimits,
		strippedHQInstanceIDServices: cfg.StrippedHQInstanceIDServices,
		strippedHQIDYPClusters:       cfg.StrippedHQIDYPClusters,
		hqCaster:                     hqcaster.NewCasterFromConfig(cfg.InstanceSpecToHQCaster),
	}
	return t
}

func (g *Generator) findLogVolume(volumes []*clusterapi.Volume) *clusterapi.Volume {
	for _, v := range volumes {
		if v.GetMountPoint() == g.logMountPoint {
			return v
		}
	}
	return nil
}

func (g *Generator) getLogVolumeSize(volumes []*clusterapi.Volume) uint64 {
	v := g.findLogVolume(volumes)
	if v == nil {
		return 0
	}
	return v.GetQuotaBytes()
}

func (g *Generator) generateRootVolume(r *ypapi.TPodSpec_TDiskVolumeRequest, tpl *internalpb.IssConfTemplate, instanceSpec instancespec.Interface) (*clusterapi.Volume, error) {
	v := &clusterapi.Volume{}
	bind, err := issutil.MakeBindProperty(instanceSpec.GetHostProvidedDaemons())
	if err != nil {
		return nil, fmt.Errorf("failed to make bind property for root volume: %w", err)
	}
	if bind != "" {
		v.Properties = map[string]string{"bind": bind}
	}

	v.MountPoint = "/"
	v.Storage = getStorageClass(r)
	quota, err := yputil.FindLabelUint64(r.GetLabels(), "root_fs_snapshot_quota")
	if err != nil {
		return nil, fmt.Errorf("failed to get root_fs_snapshot_quota for root volume: %w", err)
	}
	v.QuotaBytes = quota
	cwdQuota, err := yputil.FindLabelUint64(r.GetLabels(), "work_dir_snapshot_quota")
	if err != nil {
		return nil, fmt.Errorf("failed to get work_dir_snapshot_quota for root volume: %w", err)
	}
	v.QuotaCwdBytes = cwdQuota
	v.Layers = make([]*clusterapi.Resource, 0, len(tpl.GetLayers()))
	for _, tl := range tpl.GetLayers() {
		l := proto.Clone(tl).(*clusterapi.Resource)
		l.Storage = v.Storage
		v.Layers = append(v.GetLayers(), l)
	}
	return v, nil
}

func (g *Generator) generatePersistentVolume(r *ypapi.TPodSpec_TDiskVolumeRequest) (*clusterapi.Volume, error) {
	v := &clusterapi.Volume{}
	v.Uuid = r.GetId()
	mp, err := yputil.FindLabelString(r.GetLabels(), "mount_path")
	if err != nil {
		return nil, fmt.Errorf("failed to find mount_path label for volume %s: %w", r.GetId(), err)
	}
	v.MountPoint = mp
	v.Storage = getStorageClass(r)
	v.QuotaBytes = r.GetQuotaPolicy().GetCapacity()
	v.QuotaCwdBytes = r.GetQuotaPolicy().GetCapacity()
	return v, nil
}

func (g *Generator) generateBindVolume(b *repopb.HostPathBind) *clusterapi.Volume {
	v := &clusterapi.Volume{}
	v.MountPoint = b.MountPath
	v.QuotaBytes = defaultDiskQuotaBind
	v.QuotaCwdBytes = defaultDiskQuotaBind
	v.Properties = map[string]string{
		"backend": "rbind",
		"storage": b.Path,
	}
	return v
}

func (g *Generator) generateSkynetVolume() *clusterapi.Volume {
	v := &clusterapi.Volume{}
	v.MountPoint = skynetVolumePath
	v.QuotaBytes = defaultDiskQuotaBind
	v.QuotaCwdBytes = defaultDiskQuotaBind
	v.Properties = map[string]string{
		"backend":   "bind",
		"storage":   skynetVolumePath,
		"read_only": "true",
	}
	return v
}

func (g *Generator) generateHQInstanceRevision(confID string, instanceSpec instancespec.Interface, ra *nannyclient.RuntimeAttrs, volumes []*clusterapi.Volume, tags []string, netSet *networkSettings) (*hqpb.InstanceRevision, error) {
	r, err := g.hqCaster.Cast(instanceSpec, ra, g.getLogVolumeSize(volumes))
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec to HQ: %w", err)
	}
	r.Id = confID
	r.Tags = append(r.Tags, tags...)
	r.Hostname = netSet.Hostname
	return r, nil
}

func (g *Generator) generateVolumes(pod *api.Pod, tpl *internalpb.IssConfTemplate, instanceSpec instancespec.Interface) (allVolumes []*clusterapi.Volume, rootVolume *clusterapi.Volume, err error) {
	// disk_volume_requests volumes + bind volumes + 1 skynet volume
	size := len(pod.GetSpec().GetDiskVolumeRequests()) + len(instanceSpec.GetLayersConfig().GetBind()) + 1
	allVolumes = make([]*clusterapi.Volume, 0, size)
	for _, r := range pod.GetSpec().GetDiskVolumeRequests() {
		vt, err := yputil.FindLabelString(r.GetLabels(), "volume_type")
		if err != nil {
			return nil, nil, fmt.Errorf("failed to find volume_type label for volume %s: %w", r.GetId(), err)
		}
		if vt == "root_fs" {
			rootVolume, err = g.generateRootVolume(r, tpl, instanceSpec)
			if err != nil {
				return nil, nil, fmt.Errorf("failed generating root volume to ISS: %w", err)
			}
			allVolumes = append(allVolumes, rootVolume)
		} else if vt == "persistent" {
			v, err := g.generatePersistentVolume(r)
			if err != nil {
				return nil, nil, fmt.Errorf("failed generating persistent volume %s to ISS: %w", r.GetId(), err)
			}
			allVolumes = append(allVolumes, v)
		}
	}
	if rootVolume == nil {
		return nil, nil, fmt.Errorf("failed generating volumes for pod %s: pod must have root volume", pod.GetMeta().GetId())
	}
	for _, b := range instanceSpec.GetLayersConfig().GetBind() {
		// SWAT-5650: allow /yt for YP-lite
		if !isAllowedRBind(b) {
			return nil, nil, fmt.Errorf("YP-lite does not support explicit host path binds")
		}
		v := g.generateBindVolume(b)
		allVolumes = append(allVolumes, v)
	}
	for _, d := range instanceSpec.GetHostProvidedDaemons() {
		if d.GetType() == repopb.HostProvidedDaemon_HOST_SKYNET {
			v := g.generateSkynetVolume()
			allVolumes = append(allVolumes, v)
		} else if d.GetType() != repopb.HostProvidedDaemon_YASM_AGENT {
			return nil, nil, fmt.Errorf("unknown container host provided subagent type")
		}
	}
	return allVolumes, rootVolume, nil
}

func (g *Generator) generateDynProperties(instanceSpec instancespec.Interface, snapshotID string) map[string]string {
	p := make(map[string]string)
	p["NANNY_SNAPSHOT_ID"] = snapshotID
	p["HBF_NAT"] = "disabled"
	p["nanny_container_access_url"] = g.containerAccessURL
	a := instanceSpec.GetInstanceAccess()
	if a != nil && a.SkynetSsh == repopb.InstanceAccess_ENABLED {
		p["SKYNET_SSH"] = "enabled"
	} else {
		p["SKYNET_SSH"] = "disabled"
	}
	return p
}

func (g *Generator) generateSlot(pod *api.Pod, ictlTaskID int) (*clusterapi.Slot, error) {
	s := &clusterapi.Slot{}
	if ictlTaskID < ictlversions.MinYPLiteEmptySlotPayload {
		s.Host = strings.ReplaceAll(pod.GetSpec().GetNodeId(), "yandex.ru", "search.yandex.net")
	}
	s.Service = pod.GetMeta().GetId()
	return s, nil
}

func (g *Generator) generateHooksTimeLimits(ra *nannyclient.RuntimeAttrs) (map[string]*clusterapi.TimeLimit, error) {
	limitsMap := make(map[string]*clusterapi.TimeLimit)
	for _, n := range issHooks {
		l, err := ra.Content.Instances.ISSSettings.HookTimeLimits.GetTimeLimitsByName(n)
		if err != nil {
			return limitsMap, fmt.Errorf("failed generating ISS hooks time limits: %w", err)
		}
		var cl *clusterapi.TimeLimit
		if n == "iss_hook_status" {
			cl = castTimeLimitsToCAPI(l, g.defaultHooksTimeLimits)
		} else {
			cl = castTimeLimitsToCAPI(l, nil)
		}
		if cl != nil {
			limitsMap[n] = cl
		}
	}
	return limitsMap, nil
}

func (g *Generator) generateTags(ra *nannyclient.RuntimeAttrs, instanceSpec instancespec.Interface, cluster *yp.ClusterInfo) ([]string, error) {
	if err := validateOrthogonalTags(ra); err != nil {
		return nil, fmt.Errorf("orthogonal tags validation error: %w", err)
	}
	oTags := ra.Content.Instances.YPPodIDs.OrthogonalTags
	var tier string
	if oTags.Tier != "" {
		tier = oTags.Tier
	} else {
		tier = "none"
	}
	iTags := []string{
		fmt.Sprintf("a_geo_%s", strings.ToLower(cluster.Location)),
		fmt.Sprintf("a_dc_%s", strings.ToLower(cluster.DC)),
		fmt.Sprintf("a_itype_%s", strings.ToLower(oTags.IType)),
		fmt.Sprintf("a_ctype_%s", strings.ToLower(oTags.CType)),
		fmt.Sprintf("a_prj_%s", strings.ToLower(oTags.Prj)),
		fmt.Sprintf("a_metaprj_%s", strings.ToLower(oTags.MetaPrj)),
		fmt.Sprintf("a_tier_%s", strings.ToLower(tier)),
	}
	iTags = append(iTags, instanceSpec.GetHQTags()...)
	err := validateITags(iTags)
	if err != nil {
		return nil, fmt.Errorf("instance tags validation failed: %w", err)
	}
	return iTags, nil
}

func (g *Generator) generateYASMUnistatURLProperty(netSet *networkSettings, containers []*repopb.Container) (string, error) {
	eps := make([]*repopb.YasmUnistatEndpoint, 0, 50)
	for _, c := range containers {
		eps = append(eps, c.GetUnistatEndpoints()...)
	}
	if len(eps) == 0 {
		return "", nil
	}

	urls := make([]string, 0, len(eps))
	h := fmt.Sprintf("[%s]", netSet.BackboneIP)
	replArgs := make([]string, 0, genCfgPortRange*2)
	replArgs = append(replArgs, "{BSCONFIG_IPORT}", strconv.Itoa(DefaultYPInstancePort))
	for i := 1; i < genCfgPortRange+1; i++ {
		replArgs = append(replArgs, fmt.Sprintf("{BSCONFIG_IPORT_PLUS_%d}", i), strconv.Itoa(DefaultYPInstancePort+i))
	}
	r := strings.NewReplacer(replArgs...)
	for _, ep := range eps {
		port, err := strconv.Atoi(r.Replace(ep.Port))
		if err != nil {
			return "", fmt.Errorf("port must be digit or BSCONFIG_IPORT... matched, got: %s", ep.Port)
		}
		url := fmt.Sprintf("http://%s:%d", h, port)
		if ep.Path != "" {
			url = fmt.Sprintf("%s/%s", url, ep.Path)
		}
		urls = append(urls, url)
	}
	return strings.Join(urls, ";"), nil
}

func (g *Generator) makeHQID(cluster *yp.ClusterInfo, serviceID string, podID string) string {
	if contains(strings.ToUpper(cluster.Name), g.strippedHQIDYPClusters) {
		return podID
	}
	if contains(serviceID, g.strippedHQInstanceIDServices) {
		return podID
	}
	return yputil.MakePodPersistentHostname(podID, cluster.HostnameSegment, cluster.HostnameSuffix)
}

func (g *Generator) MakeHQInstanceID(cluster *yp.ClusterInfo, serviceID string, podID string) string {
	return fmt.Sprintf("%s@%s", g.makeHQID(cluster, serviceID, podID), serviceID)
}

func (g *Generator) generateProperties(pod *api.Pod,
	cluster *yp.ClusterInfo,
	ictlTaskID int,
	ra *nannyclient.RuntimeAttrs,
	instanceSpec instancespec.Interface,
	tags []string,
	netSet *networkSettings) (map[string]string, error) {

	p := map[string]string{
		"tags":                     strings.Join(tags, " "),
		"DEPLOY_ENGINE":            "YP_LITE",
		"NANNY_SERVICE_ID":         ra.ServiceID,
		"HOSTNAME":                 netSet.Hostname,
		"BACKBONE_IP_ADDRESS":      netSet.BackboneIP,
		"HQ_INSTANCE_ID":           g.MakeHQInstanceID(cluster, ra.ServiceID, pod.GetMeta().GetId()),
		"INSTANCE_TAG_ITYPE":       ra.Content.Instances.YPPodIDs.OrthogonalTags.IType,
		"INSTANCE_TAG_CTYPE":       ra.Content.Instances.YPPodIDs.OrthogonalTags.CType,
		"INSTANCE_TAG_PRJ":         ra.Content.Instances.YPPodIDs.OrthogonalTags.Prj,
		"yasmUnistatFallbackPort":  strconv.Itoa(DefaultYPInstancePort),
		"yasmInstanceFallbackPort": strconv.Itoa(DefaultYPInstancePort), // https://st.yandex-team.ru/GOLOVANSUPPORT-504
	}
	if instanceSpec.GetInstanceSpec().GetHqPolicy() == repopb.InstanceSpec_HQ_STATUS_ONLY {
		p["HQ_REPORT_VERSION"] = "2"
	}
	h, err := nannyutil.ComputeInstanceSpecHashMD5(instanceSpec.GetInstanceSpec())
	if err != nil {
		return nil, fmt.Errorf("failed computing InstanceSpec hash: %w", err)
	}
	if instanceSpec.IsHQPollEnabled() {
		p["HQ_INSTANCE_SPEC_HASH"] = fmt.Sprintf("%x", h)
	}

	yasmURL := fmt.Sprintf("http://[%s]:%d/", netSet.BackboneIP, yasmAgentMTNPort)
	for _, d := range instanceSpec.GetHostProvidedDaemons() {
		if d.GetType() == repopb.HostProvidedDaemon_YASM_AGENT {
			p["monitoringYasmagentEndpoint"] = yasmURL
		} else if d.GetType() == repopb.HostProvidedDaemon_HOST_SKYNET {
			p["HOST_SKYNET"] = "enabled"
		} else {
			return nil, fmt.Errorf("unknown container host provided subagent type")
		}
	}
	for _, a := range instanceSpec.GetAUXDaemons() {
		if a.GetType() != repopb.AuxDaemon_JUGGLER_AGENT {
			continue
		}
		p["JUGGLER_AGENT_PORT"] = strconv.Itoa(jugglerAgentMTNPort)
		p["monitoringJugglerEndpoint"] = fmt.Sprintf("http://[%s]:%d/", netSet.BackboneIP, jugglerAgentMTNPort)
		p["monitoringInstanceName"] = ""
		p["monitoringHostname"] = netSet.Hostname
		break
	}
	unistatURLs, err := g.generateYASMUnistatURLProperty(netSet, instanceSpec.GetInstanceSpec().GetContainers())
	if err != nil {
		return nil, fmt.Errorf("failed generating YASM unistat urls ISS propery: %w", err)
	}
	if unistatURLs != "" {
		p["yasmUnistatUrl"] = unistatURLs
	}

	node := pod.GetSpec().GetNodeId()
	if ictlTaskID < ictlversions.MinYPLiteNodeAgnosticPayload {
		if node == "" {
			return nil, fmt.Errorf("failed node name ISS property: pod %s has not been assigned to node by YP", pod.GetMeta().GetId())
		}
		p["NODE_NAME"] = node
	}
	return p, nil
}

func (g *Generator) generateConstraints(pod *api.Pod, ra *nannyclient.RuntimeAttrs, instanceSpec instancespec.Interface, ictlTaskID int) map[string]string {
	c := make(map[string]string)
	if instanceSpec.IsDockerInstance() {
		// Files in docker images usually have permissions for root only and
		// main process in docker container is expected to be run under root
		// The only way to start process in porto container under root is to use virt_mode: os
		// https://wiki.yandex-team.ru/porto/propertiesanddata/virtmode/
		c["iss_hook_uninstall.virt_mode"] = "os"
		c["iss_hook_install.virt_mode"] = "os"
		c["iss_hook_notify.virt_mode"] = "os"
		c["iss_hook_start.virt_mode"] = "os"
		c["iss_hook_stop.virt_mode"] = "os"
		c["iss_hook_status.virt_mode"] = "os"
		c["iss_hook_reopenlogs.virt_mode"] = "os"
	}
	c["iss_hook_start.capabilities_ambient"] = "NET_BIND_SERVICE"
	// SWAT-5686: allow enable_porto=true for YP-lite
	if v := instanceSpec.GetInstanceSpec().GetPortoSettings().GetEnablePorto(); v != "" {
		c["meta.enable_porto"] = v
	} else if !hasYTRBind(instanceSpec.GetInstanceSpec()) {
		c["meta.enable_porto"] = "isolate"
	}
	var uLimit string
	if ictlTaskID >= ictlversions.MinInstanceCtlYPLiteHighNofileHardLimit {
		uLimit = defaultYPULimitHighNofileHardLimit
	} else {
		uLimit = defaultYPULimit
	}
	c["meta.ulimit"] = uLimit
	c["ulimit"] = uLimit
	rc := instanceSpec.GetNetworkProperties().GetResolvConf()
	if rc == repopb.ContainerNetworkProperties_KEEP_RESOLV_CONF {
		// Use resolv.conf in container as is.
		// See: https://st.yandex-team.ru/SWAT-4587
		c["meta.resolv_conf"] = "keep"
	} else if rc == repopb.ContainerNetworkProperties_USE_NAT64 {
		c["meta.resolv_conf"] = nat64ResolvConfContent
	} else if rc == repopb.ContainerNetworkProperties_USE_NAT64_LOCAL {
		c["meta.resolv_conf"] = nat64LocalResolvConfContent
	}
	for _, h := range inParallelISSHooks {
		c[fmt.Sprintf("%s.enable_porto", h)] = "false"
		// iss_hook_{stop,status,notify,reopenlogs} make just one thing: calls main instancectl process
		// via unixsocket. They must run in empty network namespace: https://st.yandex-team.ru/SWAT-3361
		c[fmt.Sprintf("%s.net", h)] = "inherited"
	}

	if w := ra.Content.Instances.ISSSettings.HookResourceLimits.ISSHookStartCPUWeight; w != nil {
		// https://st.yandex-team.ru/SWAT-5288
		c["iss_hook_start.cpu_weight"] = strconv.FormatUint(*w, 10)
		for _, h := range issHooks {
			if h == "iss_hook_start" {
				continue
			}
			c[fmt.Sprintf("%s.cpu_weight", h)] = defaultCPUWeight
		}
	}
	var hookMemLim *uint64
	if memLim := pod.GetSpec().GetResourceRequests().MemoryLimit; memLim != nil {
		hookMemLim = new(uint64)
		*hookMemLim = max(*memLim-slotMemoryMargin, 0)
	}
	// we need to exclude setting cpu limit in hooks:
	// https://st.yandex-team.ru/SWAT-4980
	excludeCPULimit := ictlTaskID >= ictlversions.MinInstanceCtlExcludeCPULimitFromHooks
	// Currently we exclude memory_limit in hooks:
	// https://st.yandex-team.ru/SWAT-5077.
	// Before that we set hook's memory limit equal to slot's
	// memory limit - 16 MB: https://st.yandex-team.ru/SWAT-4743
	excludeMemLimit := ictlTaskID >= ictlversions.MinInstanceCtlSetHostnameExcludeHooksMemLimit
	for _, h := range mainISSHooks {
		// Set "meta.enable_porto: isolate" only without "iss_hook_start.enable_porto: isolate"
		// For details see: SWAT-5688
		if ictlTaskID < ictlversions.MinIsolatePortoOnMetaOnly {
			c[fmt.Sprintf("%s.enable_porto", h)] = "isolate"
		}
		// iss_hook_{start,install,iss_hook_uninstall} must run in the same slot net ns.
		c[fmt.Sprintf("%s.net", h)] = "inherited"

		// https://st.yandex-team.ru/SWAT-5988
		if ictlTaskID >= ictlversions.MinYPLiteSetHooksNetLimit {
			c[fmt.Sprintf("%s.net_limit", h)] = "default: 0"
		}
		if !excludeMemLimit && hookMemLim != nil {
			c[fmt.Sprintf("%s.memory_limit", h)] = strconv.FormatUint(*hookMemLim, 10)
		}
		rr := pod.GetSpec().GetResourceRequests()
		if !excludeCPULimit {
			if cpuLim := rr.VcpuLimit; cpuLim != nil {
				c[fmt.Sprintf("%s.cpu_limit", h)] = fmt.Sprintf("%6fc", float64(*cpuLim)/1000.0)
			}
		}
		c[fmt.Sprintf("%s.oom_is_fatal", h)] = "false"
	}
	return c
}

func castSnapshotTargetStateToCAPI(s repopb.Snapshot_Target) string {
	switch s {
	case repopb.Snapshot_ACTIVE:
		return "ACTIVE"
	case repopb.Snapshot_PREPARED:
		return "PREPARED"
	default:
		return "REMOVED"
	}
}

// Create HostConfigurationInstance spec from pod, service runtime attrs and instance spec.
func (g *Generator) GenerateForPod(pod *api.Pod, cluster *yp.ClusterInfo, tpl *internalpb.IssConfTemplate, ra *nannyclient.RuntimeAttrs, instanceSpec instancespec.Interface, sn *repopb.Snapshot) (*clusterapi.HostConfigurationInstance, error) {
	podConf := fmt.Sprintf("%s#%s", pod.GetMeta().GetId(), sn.GetSnapshotMetaInfo().GetConfId())
	ictlTaskID, err := nannyutil.GetInstanceCtlTaskID(ra, instanceSpec.GetInstanceCtl())
	if err != nil {
		return nil, fmt.Errorf("[%s]: failed to get InstanceCtl task ID: %w", podConf, err)
	}
	rv := &clusterapi.HostConfigurationInstance{}
	rv.Id = &clusterapi.WorkloadId{}
	slot, err := g.generateSlot(pod, ictlTaskID)
	if err != nil {
		return nil, fmt.Errorf("[%s]: error generating ISS slot: %w", podConf, err)
	}
	rv.Id.Slot = slot
	rv.Id.Configuration = &clusterapi.ConfigurationId{
		GroupId:               ra.ServiceID,
		GroupStateFingerprint: sn.GetSnapshotMetaInfo().GetConfId(),
	}
	rv.TargetState = castSnapshotTargetStateToCAPI(sn.GetTarget())

	rv.DynamicProperties = g.generateDynProperties(instanceSpec, ra.SnapshotID)

	volumes, rootVol, err := g.generateVolumes(pod, tpl, instanceSpec)
	if err != nil {
		return nil, fmt.Errorf("[%s]: error generating ISS volumes: %w", podConf, err)
	}
	rv.Entity = &clusterapi.Entity{}
	rv.Entity.Kind = &clusterapi.Entity_Instance{Instance: &clusterapi.Instance{}}
	rv.GetEntity().GetInstance().Volumes = volumes
	rv.GetEntity().GetInstance().Storage = rootVol.GetStorage()

	limits, err := g.generateHooksTimeLimits(ra)
	if err != nil {
		return nil, fmt.Errorf("[%s]: failed generating ISS hook time limits: %w", podConf, err)
	}
	rv.GetEntity().GetInstance().TimeLimits = limits

	tags, err := g.generateTags(ra, instanceSpec, cluster)
	if err != nil {
		return nil, fmt.Errorf("[%s]: failed generating tags: %w", podConf, err)
	}
	netSet, err := newNetworkSettingsFromPod(pod, cluster, ictlTaskID)
	if err != nil {
		return nil, fmt.Errorf("[%s]: failed making network settings: %w", podConf, err)
	}
	props, err := g.generateProperties(pod, cluster, ictlTaskID, ra, instanceSpec, tags, netSet)
	if err != nil {
		return nil, fmt.Errorf("[%s]: failed generating ISS properties: %w", podConf, err)
	}
	rv.Properties = props

	rv.GetEntity().GetInstance().Container = &clusterapi.Container{
		Constraints: g.generateConstraints(pod, ra, instanceSpec, ictlTaskID),
	}
	resources := make(map[string]*clusterapi.Resourcelike, len(tpl.Resources))
	for _, r := range tpl.Resources {
		resources[r.GetName()] = r.GetResource()
	}
	rv.GetEntity().GetInstance().Resources = resources

	ictlVer := instanceSpec.GetInstanceCtl().GetVersion()
	if nannyutil.CompareInstanceCtlVersions(ictlVer, ictlversions.MinInstanceCtlStoreHQSpecInYP) >= 0 {
		hqRev, err := g.generateHQInstanceRevision(sn.GetSnapshotMetaInfo().GetConfId(), instanceSpec, ra, volumes, tags, netSet)
		if err != nil {
			return nil, fmt.Errorf("[%s]: failed generating HQ instance spec: %w", podConf, err)
		}
		rv.InstanceRevision = hqRev
	}

	return rv, nil
}
