package confgen

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"strconv"
	"testing"

	"a.yandex-team.ru/infra/allocation-ctl/pkg/ictlversions"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/instancespec"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/yputil"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	internalpb "a.yandex-team.ru/infra/nanny/go/proto/nanny_internal"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
	"github.com/stretchr/testify/assert"
)

func uint64ptr(x uint64) *uint64 {
	return &x
}

func strptr(x string) *string {
	return &x
}

func makeDiskVolReq(ID string, storage string, volType string, mountPoint string, capacity uint64, extraLabelsCap uint64) *ypapi.TPodSpec_TDiskVolumeRequest {
	rv := &ypapi.TPodSpec_TDiskVolumeRequest{}
	rv.Id = &ID
	rv.StorageClass = &storage
	rv.ConcretePolicy = &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{
		QuotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
			Capacity: &capacity,
		},
	}
	labels := &ytree.TAttributeDictionary{Attributes: make([]*ytree.TAttribute, 0, 2+extraLabelsCap)}
	if err := yputil.SetLabelString(labels, "volume_type", volType); err != nil {
		panic(err)
	}
	if err := yputil.SetLabelString(labels, "mount_path", mountPoint); err != nil {
		panic(err)
	}
	rv.Labels = labels
	return rv
}

func makeRootDiskVolReq(ID string, storage string, rootSnQuota uint64, workDirSnQuota uint64, capacity uint64) *ypapi.TPodSpec_TDiskVolumeRequest {
	rv := makeDiskVolReq(ID, storage, "root_fs", "/", capacity, 2)
	if err := yputil.SetLabelUint64(rv.GetLabels(), "root_fs_snapshot_quota", rootSnQuota); err != nil {
		panic(err)
	}
	if err := yputil.SetLabelUint64(rv.GetLabels(), "work_dir_snapshot_quota", workDirSnQuota); err != nil {
		panic(err)
	}
	return rv
}

func makePersistentDiskVolReq(ID string, storage string, capacity uint64) *ypapi.TPodSpec_TDiskVolumeRequest {
	return makeDiskVolReq(ID, storage, "persistent", "/logs", capacity, 0)
}

func makeISSConfTpl() *internalpb.IssConfTemplate {
	rv := &internalpb.IssConfTemplate{}
	rv.Resources = []*internalpb.NamedResource{
		{
			Resource: &clusterapi.Resourcelike{
				Kind: &clusterapi.Resourcelike_Resource{
					Resource: &clusterapi.Resource{
						Verification: &clusterapi.Verification{
							Checksum:    "MD5:fake-md5",
							CheckPeriod: "0d0h0m",
						},
						Urls: []string{"fake-yrl"},
						TrafficClass: &clusterapi.TrafficClass{
							DownloadSpeedLimit: 0,
						},
					},
				},
			},
		},
	}
	rv.Layers = []*clusterapi.Resource{
		{
			Uuid: "layer_c1236af22f8e0bf453df2142615e45ab504bf519",
			Verification: &clusterapi.Verification{
				Checksum:    "EMPTY:",
				CheckPeriod: "0d0h0m",
			},
			Urls: []string{"top_layer"},
			TrafficClass: &clusterapi.TrafficClass{
				DownloadSpeedLimit: 0,
			},
		},
		{
			Uuid: "layer_55dfdd8922ad698bc7a2637d635337f7eaa6a2e6",
			Verification: &clusterapi.Verification{
				Checksum:    "EMPTY:",
				CheckPeriod: "0d0h0m",
			},
			Urls: []string{"bottom_layer"},
			TrafficClass: &clusterapi.TrafficClass{
				DownloadSpeedLimit: 0,
			},
		},
	}
	return rv
}

func TestGenerateVolumes(t *testing.T) {
	t.Parallel()

	p := &api.Pod{
		TPod: ypapi.TPod{
			Meta: &ypapi.TPodMeta{Id: "pod-1"},
			Spec: &ypapi.TPodSpec{
				DiskVolumeRequests: []*ypapi.TPodSpec_TDiskVolumeRequest{
					makeRootDiskVolReq("root-fs-vol", "hdd", 2*1024*1024, 1024*1024, 10*1024*1024),
					makePersistentDiskVolReq("persistent-vol", "ssd", 1024*1024),
				},
			},
		},
	}
	s := &repopb.InstanceSpec{
		Type: repopb.InstanceSpec_SANDBOX_LAYERS,
		HostProvidedDaemons: []*repopb.HostProvidedDaemon{
			{Type: repopb.HostProvidedDaemon_HOST_SKYNET},
			{Type: repopb.HostProvidedDaemon_YASM_AGENT},
		},
		LayersConfig: &repopb.LayersConfig{
			Bind: []*repopb.HostPathBind{
				{Mode: repopb.HostPathBind_VOLUME, Path: "/yt", MountPath: "/yt"},
			},
		},
	}
	instanceSpec := instancespec.InstanceSpecFactory(s)
	tpl := makeISSConfTpl()

	g := &Generator{
		containerAccessURL:           "/access-url",
		defaultHooksTimeLimits:       nil,
		strippedHQInstanceIDServices: []string{},
		strippedHQIDYPClusters:       []string{},
		hqCaster:                     nil,
	}
	vols, rootVol, err := g.generateVolumes(p, tpl, instanceSpec)
	expectedRootVol := &clusterapi.Volume{
		MountPoint:    "/",
		Storage:       "/place",
		QuotaBytes:    2 * 1024 * 1024,
		QuotaCwdBytes: 1024 * 1024,

		Properties: map[string]string{
			"bind": "/usr/local/yasmagent /usr/local/yasmagent ro",
		},
		Layers: []*clusterapi.Resource{
			{
				Uuid:    "layer_c1236af22f8e0bf453df2142615e45ab504bf519",
				Storage: "/place",
				Verification: &clusterapi.Verification{
					Checksum:    "EMPTY:",
					CheckPeriod: "0d0h0m",
				},
				Urls: []string{"top_layer"},
				TrafficClass: &clusterapi.TrafficClass{
					DownloadSpeedLimit: 0,
				},
			},
			{
				Uuid:    "layer_55dfdd8922ad698bc7a2637d635337f7eaa6a2e6",
				Storage: "/place",
				Verification: &clusterapi.Verification{
					Checksum:    "EMPTY:",
					CheckPeriod: "0d0h0m",
				},
				Urls: []string{"bottom_layer"},
				TrafficClass: &clusterapi.TrafficClass{
					DownloadSpeedLimit: 0,
				},
			},
		},
	}
	expectedVols := []*clusterapi.Volume{
		expectedRootVol,
		{
			Uuid:          "persistent-vol",
			MountPoint:    "/logs",
			Storage:       "/ssd",
			QuotaBytes:    1024 * 1024,
			QuotaCwdBytes: 1024 * 1024,
		},
		{
			MountPoint:    "/yt",
			QuotaBytes:    99999 * 1024 * 1024 * 1024,
			QuotaCwdBytes: 99999 * 1024 * 1024 * 1024,
			Properties: map[string]string{
				"backend": "rbind",
				"storage": "/yt",
			},
		},
		{
			MountPoint:    "/Berkanavt/supervisor",
			QuotaBytes:    99999 * 1024 * 1024 * 1024,
			QuotaCwdBytes: 99999 * 1024 * 1024 * 1024,
			Properties: map[string]string{
				"backend":   "bind",
				"storage":   "/Berkanavt/supervisor",
				"read_only": "true",
			},
		},
	}
	assert.NoError(t, err)
	assert.Equal(t, expectedRootVol, rootVol)
	assert.Equal(t, expectedVols, vols)
}

func TestGenerateHooksTimeLimits(t *testing.T) {
	t.Parallel()
	g := &Generator{
		containerAccessURL: "/access-url",
		defaultHooksTimeLimits: &clusterapi.TimeLimit{
			MinRestartPeriodMs:   30 * 1000,
			MaxRestartPeriodMs:   60 * 1000,
			RestartPeriodBackOff: 2,
			MaxExecutionTimeMs:   1800 * 1000,
			RestartPeriodScaleMs: 1000,
		},
		strippedHQInstanceIDServices: []string{},
		strippedHQIDYPClusters:       []string{},
		hqCaster:                     nil,
	}
	ra := &nannyclient.RuntimeAttrs{
		Content: nannyclient.RuntimeAttrsContent{
			Instances: nannyclient.Instances{
				ISSSettings: nannyclient.ISSSettings{
					HookTimeLimits: nannyclient.ISS3HooksTimeLimits{
						// RestartPeriodBackoff is missed, it must not be set with default value
						ISSHookStart: &nannyclient.ISSHookTimeLimits{
							MinRestartPeriod:   uint64ptr(1),
							MaxExecutionTime:   uint64ptr(50),
							RestartPeriodScale: uint64ptr(1),
							MaxRestartPeriod:   uint64ptr(10),
						},
						// all missed hooks must be set with default values
						ISSHookStatus: &nannyclient.ISSHookTimeLimits{
							MaxRestartPeriod: uint64ptr(10),
						},
					},
				},
			},
		},
	}

	expectedLimits := map[string]*clusterapi.TimeLimit{
		"iss_hook_start": {
			MinRestartPeriodMs:   1 * 1000,
			MaxExecutionTimeMs:   50 * 1000,
			RestartPeriodScaleMs: 1000,
			MaxRestartPeriodMs:   10 * 1000,
		},
		"iss_hook_status": {
			MinRestartPeriodMs:   30 * 1000,
			MaxRestartPeriodMs:   10 * 1000,
			RestartPeriodBackOff: 2,
			MaxExecutionTimeMs:   1800 * 1000,
			RestartPeriodScaleMs: 1000,
		},
	}

	limits, err := g.generateHooksTimeLimits(ra)
	assert.NoError(t, err)
	assert.Equal(t, expectedLimits, limits)
}

func TestGenerateYASMUnistatURLProperty(t *testing.T) {
	t.Parallel()
	netSet := &networkSettings{
		Hostname:   "hostname",
		BackboneIP: "bbIP",
	}
	containers := []*repopb.Container{
		{
			UnistatEndpoints: []*repopb.YasmUnistatEndpoint{
				{
					Path: "stat1",
					Port: "8080",
				},
				{
					Port: "{BSCONFIG_IPORT}",
				},
			},
		},
		{
			UnistatEndpoints: []*repopb.YasmUnistatEndpoint{
				{
					Path: "stat3",
					Port: "{BSCONFIG_IPORT_PLUS_2}",
				},
			},
		},
	}

	g := &Generator{
		containerAccessURL:           "/access-url",
		defaultHooksTimeLimits:       nil,
		strippedHQInstanceIDServices: []string{},
		strippedHQIDYPClusters:       []string{},
		hqCaster:                     nil,
	}
	expectedURLs := "http://[bbIP]:8080/stat1;http://[bbIP]:80;http://[bbIP]:82/stat3"
	urls, err := g.generateYASMUnistatURLProperty(netSet, containers)
	assert.NoError(t, err)
	assert.Equal(t, expectedURLs, urls)

	bad := &repopb.Container{
		UnistatEndpoints: []*repopb.YasmUnistatEndpoint{
			{
				Path: "stat1",
				Port: "abc",
			},
		},
	}
	containers = append(containers, bad)
	_, err = g.generateYASMUnistatURLProperty(netSet, containers)
	assert.Error(t, err)
}

func TestGenerateProperties(t *testing.T) {
	t.Parallel()

	p := &api.Pod{
		TPod: ypapi.TPod{
			Meta: &ypapi.TPodMeta{Id: "pod-1"},
			Spec: &ypapi.TPodSpec{
				NodeId: func(s string) *string { return &s }("node-id"),
			},
			Status: &ypapi.TPodStatus{
				Ip6AddressAllocations: []*ypapi.TPodStatus_TIP6AddressAllocation{
					{VlanId: strptr("backbone"), Address: strptr("bbIP")},
				},
			},
		},
	}
	ra := &nannyclient.RuntimeAttrs{
		ServiceID: "service-1",
		Content: nannyclient.RuntimeAttrsContent{
			Instances: nannyclient.Instances{
				YPPodIDs: nannyclient.YPPodIDs{
					OrthogonalTags: nannyclient.OrthogonalTags{
						IType: "test-itype",
						CType: "test-ctype",
						Prj:   "test-prj",
					},
				},
			},
		},
	}
	s := &repopb.InstanceSpec{
		Type:     repopb.InstanceSpec_SANDBOX_LAYERS,
		HqPolicy: repopb.InstanceSpec_HQ_STATUS_ONLY,
		HostProvidedDaemons: []*repopb.HostProvidedDaemon{
			{Type: repopb.HostProvidedDaemon_HOST_SKYNET},
			{Type: repopb.HostProvidedDaemon_YASM_AGENT},
		},
		AuxDaemons: []*repopb.AuxDaemon{
			{Type: repopb.AuxDaemon_JUGGLER_AGENT},
		},
	}
	instanceSpec := instancespec.InstanceSpecFactory(s)
	c := &yp.ClusterInfo{
		Name:            "sas-test",
		HostnameSegment: "sas",
	}
	netSet := &networkSettings{
		Hostname:   "hostname",
		BackboneIP: "bbIP",
	}
	tags := []string{"a", "b", "c"}

	g := &Generator{
		containerAccessURL:           "/access-url",
		defaultHooksTimeLimits:       nil,
		strippedHQInstanceIDServices: []string{},
		strippedHQIDYPClusters:       []string{},
		hqCaster:                     nil,
	}
	props, err := g.generateProperties(p, c, ictlversions.MinYPLiteNodeAgnosticPayload, ra, instanceSpec, tags, netSet)
	assert.NoError(t, err)

	expectedProps := map[string]string{
		"tags":                        "a b c",
		"DEPLOY_ENGINE":               "YP_LITE",
		"NANNY_SERVICE_ID":            "service-1",
		"HOSTNAME":                    "hostname",
		"BACKBONE_IP_ADDRESS":         "bbIP",
		"HQ_INSTANCE_ID":              "pod-1.sas.yp-c.yandex.net@service-1",
		"INSTANCE_TAG_ITYPE":          "test-itype",
		"INSTANCE_TAG_CTYPE":          "test-ctype",
		"INSTANCE_TAG_PRJ":            "test-prj",
		"yasmUnistatFallbackPort":     "80",
		"yasmInstanceFallbackPort":    "80",
		"HQ_REPORT_VERSION":           "2",
		"monitoringYasmagentEndpoint": "http://[bbIP]:11003/",
		"HOST_SKYNET":                 "enabled",
		"JUGGLER_AGENT_PORT":          "31579",
		"monitoringJugglerEndpoint":   "http://[bbIP]:31579/",
		"monitoringInstanceName":      "",
		"monitoringHostname":          "hostname",
	}
	assert.Equal(t, expectedProps, props)

	props, err = g.generateProperties(p, c, ictlversions.MinYPLiteNodeAgnosticPayload-1, ra, instanceSpec, tags, netSet)
	assert.NoError(t, err)
	expectedProps["NODE_NAME"] = "node-id"
	assert.Equal(t, expectedProps, props)
}

func TestGenerateConstraints(t *testing.T) {
	t.Parallel()

	p := &api.Pod{
		TPod: ypapi.TPod{
			Meta: &ypapi.TPodMeta{Id: "pod-1"},
			Spec: &ypapi.TPodSpec{
				ResourceRequests: &ypapi.TPodSpec_TResourceRequests{
					MemoryLimit:          uint64ptr(100 * 1024 * 1024),
					AnonymousMemoryLimit: uint64ptr(5 * 1024 * 1024),
					VcpuLimit:            uint64ptr(1000),
				},
			},
		},
	}
	ra := &nannyclient.RuntimeAttrs{
		ServiceID: "service-1",
		Content: nannyclient.RuntimeAttrsContent{
			Instances: nannyclient.Instances{
				ISSSettings: nannyclient.ISSSettings{
					HookResourceLimits: nannyclient.ISS3HooksResourceLimits{
						ISSHookStartCPUWeight: uint64ptr(2),
					},
				},
			},
		},
	}
	s := &repopb.InstanceSpec{
		Type: repopb.InstanceSpec_SANDBOX_LAYERS,
		PortoSettings: &repopb.InstanceSpec_PortoSettings{
			EnablePorto: "true",
		},
		NetworkProperties: &repopb.ContainerNetworkProperties{
			ResolvConf: repopb.ContainerNetworkProperties_KEEP_RESOLV_CONF,
		},
	}
	instanceSpec := instancespec.InstanceSpecFactory(s)

	g := &Generator{
		containerAccessURL:           "/access-url",
		defaultHooksTimeLimits:       nil,
		strippedHQInstanceIDServices: []string{},
		strippedHQIDYPClusters:       []string{},
		hqCaster:                     nil,
	}
	constrs := g.generateConstraints(p, ra, instanceSpec, ictlversions.MinInstanceCtlYPLiteHighNofileHardLimit)
	expectedConstrs := map[string]string{
		"iss_hook_start.capabilities_ambient": "NET_BIND_SERVICE",
		"meta.enable_porto":                   "true",
		"meta.ulimit":                         "memlock: 549755813888 549755813888; nofile: 102400 1000000;",
		"ulimit":                              "memlock: 549755813888 549755813888; nofile: 102400 1000000;",
		"meta.resolv_conf":                    "keep",
		"iss_hook_stop.enable_porto":          "false",
		"iss_hook_stop.net":                   "inherited",
		"iss_hook_status.enable_porto":        "false",
		"iss_hook_status.net":                 "inherited",
		"iss_hook_validate.enable_porto":      "false",
		"iss_hook_validate.net":               "inherited",
		"iss_hook_notify.enable_porto":        "false",
		"iss_hook_notify.net":                 "inherited",
		"iss_hook_reopenlogs.enable_porto":    "false",
		"iss_hook_reopenlogs.net":             "inherited",
		"iss_hook_start.cpu_weight":           "2",
		"iss_hook_stop.cpu_weight":            "1",
		"iss_hook_status.cpu_weight":          "1",
		"iss_hook_install.cpu_weight":         "1",
		"iss_hook_uninstall.cpu_weight":       "1",
		"iss_hook_validate.cpu_weight":        "1",
		"iss_hook_notify.cpu_weight":          "1",
		"iss_hook_reopenlogs.cpu_weight":      "1",
		"iss_hook_start.net":                  "inherited",
		"iss_hook_start.net_limit":            "default: 0",
		"iss_hook_start.oom_is_fatal":         "false",
		"iss_hook_install.net":                "inherited",
		"iss_hook_install.net_limit":          "default: 0",
		"iss_hook_install.oom_is_fatal":       "false",
		"iss_hook_uninstall.net":              "inherited",
		"iss_hook_uninstall.net_limit":        "default: 0",
		"iss_hook_uninstall.oom_is_fatal":     "false",
	}
	assert.Equal(t, expectedConstrs, constrs)

	constrs = g.generateConstraints(p, ra, instanceSpec, ictlversions.MinInstanceCtlExcludeCPULimitFromHooks-1)
	expectedConstrs = map[string]string{
		"iss_hook_start.capabilities_ambient": "NET_BIND_SERVICE",
		"meta.enable_porto":                   "true",
		"meta.ulimit":                         "memlock: 549755813888 549755813888;",
		"ulimit":                              "memlock: 549755813888 549755813888;",
		"meta.resolv_conf":                    "keep",
		"iss_hook_stop.enable_porto":          "false",
		"iss_hook_stop.net":                   "inherited",
		"iss_hook_status.enable_porto":        "false",
		"iss_hook_status.net":                 "inherited",
		"iss_hook_validate.enable_porto":      "false",
		"iss_hook_validate.net":               "inherited",
		"iss_hook_notify.enable_porto":        "false",
		"iss_hook_notify.net":                 "inherited",
		"iss_hook_reopenlogs.enable_porto":    "false",
		"iss_hook_reopenlogs.net":             "inherited",
		"iss_hook_start.cpu_weight":           "2",
		"iss_hook_stop.cpu_weight":            "1",
		"iss_hook_status.cpu_weight":          "1",
		"iss_hook_install.cpu_weight":         "1",
		"iss_hook_uninstall.cpu_weight":       "1",
		"iss_hook_validate.cpu_weight":        "1",
		"iss_hook_notify.cpu_weight":          "1",
		"iss_hook_reopenlogs.cpu_weight":      "1",
		"iss_hook_start.enable_porto":         "isolate",
		"iss_hook_start.net":                  "inherited",
		"iss_hook_start.memory_limit":         "88080384",
		"iss_hook_start.cpu_limit":            "1.000000c",
		"iss_hook_start.oom_is_fatal":         "false",
		"iss_hook_install.enable_porto":       "isolate",
		"iss_hook_install.net":                "inherited",
		"iss_hook_install.memory_limit":       "88080384",
		"iss_hook_install.cpu_limit":          "1.000000c",
		"iss_hook_install.oom_is_fatal":       "false",
		"iss_hook_uninstall.enable_porto":     "isolate",
		"iss_hook_uninstall.net":              "inherited",
		"iss_hook_uninstall.memory_limit":     "88080384",
		"iss_hook_uninstall.cpu_limit":        "1.000000c",
		"iss_hook_uninstall.oom_is_fatal":     "false",
	}
	assert.Equal(t, expectedConstrs, constrs)
}

func TestGenerateForPod(t *testing.T) {
	t.Parallel()

	p := &api.Pod{
		TPod: ypapi.TPod{
			Meta: &ypapi.TPodMeta{Id: "pod-1"},
			Spec: &ypapi.TPodSpec{
				NodeId: strptr("node-id"),
				DiskVolumeRequests: []*ypapi.TPodSpec_TDiskVolumeRequest{
					makeRootDiskVolReq("root-fs-vol", "hdd", 2*1024*1024, 1024*1024, 10*1024*1024),
					makePersistentDiskVolReq("persistent-vol", "ssd", 1024*1024),
				},
				ResourceRequests: &ypapi.TPodSpec_TResourceRequests{},
			},
		},
	}

	sn := &repopb.Snapshot{
		SnapshotMetaInfo: &repopb.RuntimeAttrsSnapshotMeta{
			ConfId: "conf-id",
		},
		Target: repopb.Snapshot_PREPARED,
	}

	ra := &nannyclient.RuntimeAttrs{
		ServiceID:  "service-id",
		SnapshotID: "snapshot-id",
		Content: nannyclient.RuntimeAttrsContent{
			Instances: nannyclient.Instances{
				ISSSettings: nannyclient.ISSSettings{
					HookTimeLimits: nannyclient.ISS3HooksTimeLimits{
						// RestartPeriodBackoff is missed, it must not be set with default value
						ISSHookStart: &nannyclient.ISSHookTimeLimits{
							MinRestartPeriod:   uint64ptr(1),
							MaxExecutionTime:   uint64ptr(50),
							RestartPeriodScale: uint64ptr(1),
							MaxRestartPeriod:   uint64ptr(10),
						},
						// all missed hooks must be set with default values
						ISSHookStatus: &nannyclient.ISSHookTimeLimits{
							MaxRestartPeriod: uint64ptr(10),
						},
					},
				},
				YPPodIDs: nannyclient.YPPodIDs{
					OrthogonalTags: nannyclient.OrthogonalTags{
						IType:   "test-itype",
						CType:   "test-ctype",
						Prj:     "test-prj",
						MetaPrj: "test-metaprj",
					},
				},
			},
		},
	}

	s := &repopb.InstanceSpec{
		Type: repopb.InstanceSpec_SANDBOX_LAYERS,
		NetworkProperties: &repopb.ContainerNetworkProperties{
			ResolvConf: repopb.ContainerNetworkProperties_KEEP_RESOLV_CONF,
		},
		InstanceAccess: &repopb.InstanceAccess{
			SkynetSsh: repopb.InstanceAccess_ENABLED,
		},
		Instancectl: &repopb.InstanceCtl{
			FetchableMeta: &repopb.FetchableMeta{
				SandboxResource: &repopb.SandboxResource{
					TaskId: strconv.Itoa(ictlversions.MinInstanceCtlYPLiteHighNofileHardLimit),
				},
			},
			Version: "0.0", // do not generate HQ instance revision
		},
	}
	instanceSpec := instancespec.InstanceSpecFactory(s)

	c := &yp.ClusterInfo{
		Name:            "sas-test",
		HostnameSegment: "sas",
		Location:        "sas",
		DC:              "sas",
	}

	tpl := makeISSConfTpl()

	g := &Generator{
		containerAccessURL: "/access-url",
		defaultHooksTimeLimits: &clusterapi.TimeLimit{
			MinRestartPeriodMs:   30 * 1000,
			MaxRestartPeriodMs:   60 * 1000,
			RestartPeriodBackOff: 2,
			MaxExecutionTimeMs:   1800 * 1000,
			RestartPeriodScaleMs: 1000,
		},
		strippedHQInstanceIDServices: []string{},
		strippedHQIDYPClusters:       []string{},
		hqCaster:                     nil,
	}

	conf, err := g.GenerateForPod(p, c, tpl, ra, instanceSpec, sn)
	assert.NoError(t, err)
	expectedConf := &clusterapi.HostConfigurationInstance{
		Id: &clusterapi.WorkloadId{
			Slot: &clusterapi.Slot{
				Service: "pod-1",
			},
			Configuration: &clusterapi.ConfigurationId{
				GroupId:               "service-id",
				GroupStateFingerprint: "conf-id",
			},
		},
		TargetState: "PREPARED",
		DynamicProperties: map[string]string{
			"NANNY_SNAPSHOT_ID":          "snapshot-id",
			"HBF_NAT":                    "disabled",
			"nanny_container_access_url": "/access-url",
			"SKYNET_SSH":                 "enabled",
		},
		Entity: &clusterapi.Entity{
			Kind: &clusterapi.Entity_Instance{
				Instance: &clusterapi.Instance{
					Volumes: []*clusterapi.Volume{
						{
							MountPoint:    "/",
							Storage:       "/place",
							QuotaBytes:    2 * 1024 * 1024,
							QuotaCwdBytes: 1024 * 1024,
							Layers: []*clusterapi.Resource{
								{
									Uuid:    "layer_c1236af22f8e0bf453df2142615e45ab504bf519",
									Storage: "/place",
									Verification: &clusterapi.Verification{
										Checksum:    "EMPTY:",
										CheckPeriod: "0d0h0m",
									},
									Urls: []string{"top_layer"},
									TrafficClass: &clusterapi.TrafficClass{
										DownloadSpeedLimit: 0,
									},
								},
								{
									Uuid:    "layer_55dfdd8922ad698bc7a2637d635337f7eaa6a2e6",
									Storage: "/place",
									Verification: &clusterapi.Verification{
										Checksum:    "EMPTY:",
										CheckPeriod: "0d0h0m",
									},
									Urls: []string{"bottom_layer"},
									TrafficClass: &clusterapi.TrafficClass{
										DownloadSpeedLimit: 0,
									},
								},
							},
						},
						{
							Uuid:          "persistent-vol",
							MountPoint:    "/logs",
							Storage:       "/ssd",
							QuotaBytes:    1024 * 1024,
							QuotaCwdBytes: 1024 * 1024,
						},
					},
					Storage: "/place",
					TimeLimits: map[string]*clusterapi.TimeLimit{
						"iss_hook_start": {
							MinRestartPeriodMs:   1 * 1000,
							MaxExecutionTimeMs:   50 * 1000,
							RestartPeriodScaleMs: 1000,
							MaxRestartPeriodMs:   10 * 1000,
						},
						"iss_hook_status": {
							MinRestartPeriodMs:   30 * 1000,
							MaxRestartPeriodMs:   10 * 1000,
							RestartPeriodBackOff: 2,
							MaxExecutionTimeMs:   1800 * 1000,
							RestartPeriodScaleMs: 1000,
						},
					},
					Container: &clusterapi.Container{
						Constraints: map[string]string{
							"iss_hook_start.capabilities_ambient": "NET_BIND_SERVICE",
							"meta.enable_porto":                   "isolate",
							"meta.ulimit":                         "memlock: 549755813888 549755813888; nofile: 102400 1000000;",
							"ulimit":                              "memlock: 549755813888 549755813888; nofile: 102400 1000000;",
							"meta.resolv_conf":                    "keep",
							"iss_hook_stop.enable_porto":          "false",
							"iss_hook_stop.net":                   "inherited",
							"iss_hook_status.enable_porto":        "false",
							"iss_hook_status.net":                 "inherited",
							"iss_hook_validate.enable_porto":      "false",
							"iss_hook_validate.net":               "inherited",
							"iss_hook_notify.enable_porto":        "false",
							"iss_hook_notify.net":                 "inherited",
							"iss_hook_reopenlogs.enable_porto":    "false",
							"iss_hook_reopenlogs.net":             "inherited",
							"iss_hook_start.net":                  "inherited",
							"iss_hook_start.net_limit":            "default: 0",
							"iss_hook_start.oom_is_fatal":         "false",
							"iss_hook_install.net":                "inherited",
							"iss_hook_install.net_limit":          "default: 0",
							"iss_hook_install.oom_is_fatal":       "false",
							"iss_hook_uninstall.net":              "inherited",
							"iss_hook_uninstall.net_limit":        "default: 0",
							"iss_hook_uninstall.oom_is_fatal":     "false",
						},
					},
				},
			},
		},
		Properties: map[string]string{
			"tags":                     "a_geo_sas a_dc_sas a_itype_test-itype a_ctype_test-ctype a_prj_test-prj a_metaprj_test-metaprj a_tier_none enable_hq_report",
			"DEPLOY_ENGINE":            "YP_LITE",
			"NANNY_SERVICE_ID":         "service-id",
			"HOSTNAME":                 "pod-1.sas-test.yp-c.yandex.net",
			"BACKBONE_IP_ADDRESS":      "%%BACKBONE_IP_ADDRESS%%",
			"HQ_INSTANCE_ID":           "pod-1.sas.yp-c.yandex.net@service-id",
			"INSTANCE_TAG_ITYPE":       "test-itype",
			"INSTANCE_TAG_CTYPE":       "test-ctype",
			"INSTANCE_TAG_PRJ":         "test-prj",
			"yasmUnistatFallbackPort":  "80",
			"yasmInstanceFallbackPort": "80",
		},
	}

	resources := make(map[string]*clusterapi.Resourcelike, len(tpl.Resources))
	for _, r := range tpl.Resources {
		resources[r.GetName()] = r.GetResource()
	}
	expectedConf.GetEntity().GetInstance().Resources = resources
	assert.Equal(t, expectedConf, conf)
}
