package hqcaster

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/ictlversions"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/instancespec"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/nannyutil"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	hqpb "a.yandex-team.ru/yp/go/proto/hq"
	"fmt"
	"google.golang.org/protobuf/proto"
	"strings"
)

const (
	LogrotateAUXDaemonCommand = `
while true; do
    /bin/sleep %d
    # if rotation is not required, logrotate exits with 1, so don't crash whole section
    ./logrotate -f -s state/logrotate_status logrotate.conf || true
done
`
	LogrotateAUXInitCommand = `
mkdir -p /place/db/www
TARGET=/logs
LINK_NAME=/place/db/www/logs
ln -sT $TARGET $LINK_NAME || (
    # errexit semantics made me do it
    [ -L $LINK_NAME ] && [ $(readlink $LINK_NAME) = $TARGET ]
)
sed '
s|CURRENT_LOG_PATTERN|%s|;
s|CURRENT_LOG_SIZE_MAX|%d|;
s|FREE_SPACE_MIN|%d|;
' < logrotate.conf.tmpl > logrotate.conf
chmod 0644 logrotate.conf
`
	SkynetAUXDaemonCommand = `
/etc/init.d/skynetd start
/etc/init.d/skycored start
while true; do ` + `
    /Berkanavt/skynet/bin/gosky
    /bin/sleep 1200 ` + `
    /etc/init.d/skynetd autostart
    /etc/init.d/skycored autostart
done
`
	SkynetAUXDaemonCommandWithoutSkynetd = `
/etc/init.d/skycored start
while true; do ` + `
    /Berkanavt/skynet/bin/gosky
    /bin/sleep 1200 ` + `
    /etc/init.d/skycored autostart
done
`
	SandboxGDBPath = "/gdb/bin/gdb"
)

func builtinGDBMustBeUsed(ictlVersion string, gdbSource *repopb.GdbSource) bool {
	return nannyutil.CompareInstanceCtlVersions(ictlVersion, ictlversions.MinInstanceCtlGDB) > 0 && gdbSource.GetType() != repopb.GdbSource_CUSTOM
}

func makeJugglerRootPath(rootFSEnables bool) string {
	if rootFSEnables {
		return "/juggler"
	}
	return "{BSCONFIG_IDIR}/juggler"
}

type resourcePath struct {
	LocalPath   string
	ExtractPath string
}

func collectResourcePaths(ra *nannyclient.RuntimeAttrs) []*resourcePath {
	r := ra.Content.Resources
	size := len(r.SandboxFiles) + len(r.StaticFiles) + len(r.URLFiles)
	rv := make([]*resourcePath, 0, size)
	for _, sf := range r.SandboxFiles {
		rp := &resourcePath{
			LocalPath:   sf.LocalPath,
			ExtractPath: sf.ExtractPath,
		}
		rv = append(rv, rp)
	}
	for _, uf := range r.URLFiles {
		rp := &resourcePath{
			LocalPath:   uf.LocalPath,
			ExtractPath: uf.ExtractPath,
		}
		rv = append(rv, rp)
	}
	for _, stf := range r.StaticFiles {
		rp := &resourcePath{
			LocalPath:   stf.LocalPath,
			ExtractPath: stf.ExtractPath,
		}
		rv = append(rv, rp)
	}
	return rv
}

type scalarResourceValue struct {
	Value int64 `yaml:"value"`
}

func (s *scalarResourceValue) ToProto() *hqpb.Scalar {
	return &hqpb.Scalar{
		Value: s.Value,
	}
}

type computeResource struct {
	Name   string               `yaml:"name"`
	Scalar *scalarResourceValue `yaml:"scalar"`
}

func (r *computeResource) ToProto() *hqpb.ComputeResource {
	rv := &hqpb.ComputeResource{
		Name: r.Name,
	}
	if r.Scalar != nil {
		rv.Scalar = r.Scalar.ToProto()
	}
	return rv
}

type resourceAllocation struct {
	Limit   []*computeResource `yaml:"limit"`
	Request []*computeResource `yaml:"request"`
}

func resourcesToProto(r []*computeResource) []*hqpb.ComputeResource {
	rv := make([]*hqpb.ComputeResource, 0, len(r))
	for _, item := range r {
		if item == nil {
			rv = append(rv, nil)
		} else {
			rv = append(rv, item.ToProto())
		}
	}
	return rv
}

func (a *resourceAllocation) ToProto() *hqpb.ResourceAllocation {
	rv := &hqpb.ResourceAllocation{}
	if a.Limit != nil {
		rv.Limit = resourcesToProto(a.Limit)
	}
	if a.Request != nil {
		rv.Request = resourcesToProto(a.Request)
	}
	return rv
}

type defaultLogrotateSettings struct {
	IterTimeout       uint64  `yaml:"iter_timeout"`
	CurrentLogPattern string  `yaml:"current_log_pattern"`
	LogSizeMaxFactor  float64 `yaml:"log_size_max_factor"`
}

type logrotate struct {
	Default *defaultLogrotateSettings `yaml:"default"`
}

type auxDaemons struct {
	Logrotate *logrotate `yaml:"logrotate"`
}

type Config struct {
	JugglerResourceAllocation   *resourceAllocation `yaml:"juggler_resource_allocation"`
	SkynetResourceAllocation    *resourceAllocation `yaml:"skynet_resource_allocation"`
	LogrotateResourceAllocation *resourceAllocation `yaml:"logrotate_resource_allocation"`
	AUXDaemons                  *auxDaemons         `yaml:"aux_daemons"`
}

type Caster struct {
	defaultJugglerResourceAllocation   *hqpb.ResourceAllocation
	defaultSkynetResourceAllocation    *hqpb.ResourceAllocation
	defaultLogrotateResourceAllocation *hqpb.ResourceAllocation
	defaultAUXDaemonsReadinessProbe    *hqpb.Probe
	defaultAUXDaemonsRestartPolicy     *hqpb.RestartPolicy
	defaultAUXDaemonsLifecycle         *hqpb.Lifecycle
	aux                                *auxDaemons
}

func NewCasterFromConfig(cfg *Config) *Caster {
	probe := &hqpb.Probe{}
	probe.InitialDelaySeconds = 5
	probe.MinPeriodSeconds = 5
	probe.MaxPeriodSeconds = 60
	probe.PeriodBackoff = 2
	probe.SuccessThreshold = 1
	probe.FailureThreshold = 1

	pol := &hqpb.RestartPolicy{}
	pol.Type = hqpb.RestartPolicy_ALWAYS
	pol.MinPeriodSeconds = 1
	pol.MaxPeriodSeconds = 60
	pol.PeriodBackoff = 2
	pol.PeriodJitterSeconds = 20
	rv := &Caster{
		aux:                             cfg.AUXDaemons,
		defaultAUXDaemonsReadinessProbe: probe,
		defaultAUXDaemonsRestartPolicy:  pol,
		defaultAUXDaemonsLifecycle:      &hqpb.Lifecycle{TerminationGracePeriodSeconds: 10},
	}

	if cfg.JugglerResourceAllocation != nil {
		rv.defaultJugglerResourceAllocation = cfg.JugglerResourceAllocation.ToProto()
	}
	if cfg.SkynetResourceAllocation != nil {
		rv.defaultSkynetResourceAllocation = cfg.SkynetResourceAllocation.ToProto()
	}
	if cfg.LogrotateResourceAllocation != nil {
		rv.defaultLogrotateResourceAllocation = cfg.LogrotateResourceAllocation.ToProto()
	}
	return rv
}

func (c *Caster) makeAUXDaemonsContainers(daemons []*repopb.AuxDaemon, rootFSEnabled bool, logSize uint64, ictlVer string, secCtx *hqpb.SecurityContext) (containers []*hqpb.Container, initContainers []*hqpb.Container, err error) {
	if len(daemons) == 0 {
		return nil, nil, nil
	}
	containers = make([]*hqpb.Container, 0, len(daemons))
	initContainers = make([]*hqpb.Container, 0, len(daemons))
	path := makeJugglerRootPath(rootFSEnabled)
	for _, d := range daemons {
		container := &hqpb.Container{}
		switch t := d.GetType(); t {
		case repopb.AuxDaemon_JUGGLER_AGENT:
			ra, err := c.castResourceRequestToHQ(d.GetJugglerAgent().GetResourceRequest())
			if err != nil {
				return nil, nil, fmt.Errorf("failed casting resource request for juggler agent container: %w", err)
			}
			if ra != nil {
				container.ResourceAllocation = ra
			} else {
				container.ResourceAllocation = proto.Clone(c.defaultJugglerResourceAllocation).(*hqpb.ResourceAllocation)
			}
			container.Name = "juggler"
			container.Command = []string{
				"{BSCONFIG_IDIR}/juggler-scheduler",
				"--instance-port",
				"{JUGGLER_AGENT_PORT}",
				"--base-dir",
				path,
				"--config-dir",
				"configs",
				"--checks-dir",
				"checks",
				"--varlib-dir",
				"state",
				"--run-dir",
				"state",
				"--logs-dir",
				"logs",
				"--no-console-log",
			}

			container.ReadinessProbe = proto.Clone(c.defaultAUXDaemonsReadinessProbe).(*hqpb.Probe)
			container.RestartPolicy = proto.Clone(c.defaultAUXDaemonsRestartPolicy).(*hqpb.RestartPolicy)
			container.Lifecycle = proto.Clone(c.defaultAUXDaemonsLifecycle).(*hqpb.Lifecycle)

			if secCtx != nil {
				container.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
			}
			ic := &hqpb.Container{}
			ic.Name = "init_juggler"
			ic.Command = []string{
				"mkdir",
				"-p",
				fmt.Sprintf("%s/configs", path),
				fmt.Sprintf("%s/checks", path),
				fmt.Sprintf("%s/state", path),
			}
			if secCtx != nil {
				ic.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
			}
			initContainers = append(initContainers, ic)
		case repopb.AuxDaemon_SKYNET:
			ra, err := c.castResourceRequestToHQ(d.GetSkynetDaemon().GetResourceRequest())
			if err != nil {
				return nil, nil, fmt.Errorf("failed casting resource request for skynet agent container: %w", err)
			}
			if ra != nil {
				container.ResourceAllocation = ra
			} else {
				container.ResourceAllocation = proto.Clone(c.defaultSkynetResourceAllocation).(*hqpb.ResourceAllocation)
			}

			container.Name = "skynet"
			cmd := make([]string, 0, 3)
			cmd = append(cmd, "/bin/sh", "-ec")
			if nannyutil.CompareInstanceCtlVersions(ictlVer, ictlversions.MinInstanceCtlNoSkynetd) >= 0 {
				cmd = append(cmd, SkynetAUXDaemonCommandWithoutSkynetd)
			} else {
				cmd = append(cmd, SkynetAUXDaemonCommand)
			}
			container.Command = cmd

			container.ReadinessProbe = proto.Clone(c.defaultAUXDaemonsReadinessProbe).(*hqpb.Probe)
			container.RestartPolicy = proto.Clone(c.defaultAUXDaemonsRestartPolicy).(*hqpb.RestartPolicy)
			container.Lifecycle = proto.Clone(c.defaultAUXDaemonsLifecycle).(*hqpb.Lifecycle)

			container.SecurityContext = &hqpb.SecurityContext{
				RunAsUser:         "root",
				PortoAccessPolicy: "true",
			}
		case repopb.AuxDaemon_LOGROTATE:
			ra, err := c.castResourceRequestToHQ(d.GetLogRotateDaemon().GetResourceRequest())
			if err != nil {
				return nil, nil, fmt.Errorf("failed casting resource request for logrotate agent container: %w", err)
			}
			if ra != nil {
				container.ResourceAllocation = ra
			} else {
				container.ResourceAllocation = proto.Clone(c.defaultLogrotateResourceAllocation).(*hqpb.ResourceAllocation)
			}

			container.Name = "logrotate"
			container.Command = []string{
				"/bin/sh",
				"-ec",
				fmt.Sprintf(LogrotateAUXDaemonCommand, c.aux.Logrotate.Default.IterTimeout),
			}

			container.ReadinessProbe = proto.Clone(c.defaultAUXDaemonsReadinessProbe).(*hqpb.Probe)
			container.RestartPolicy = proto.Clone(c.defaultAUXDaemonsRestartPolicy).(*hqpb.RestartPolicy)
			container.Lifecycle = proto.Clone(c.defaultAUXDaemonsLifecycle).(*hqpb.Lifecycle)

			if secCtx != nil {
				container.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
			}

			ic := &hqpb.Container{}
			logMax := int(float64(logSize) * c.aux.Logrotate.Default.LogSizeMaxFactor)
			cmd := fmt.Sprintf(LogrotateAUXInitCommand, c.aux.Logrotate.Default.CurrentLogPattern, logMax, logMax)
			ic.Name = "init_logrotate"
			ic.Command = []string{"/bin/sh", "-exc", cmd}
			if secCtx != nil {
				ic.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
			}
			initContainers = append(initContainers, ic)
		default:
			return nil, nil, fmt.Errorf("unknown AUX daemon type %s", d.GetType())
		}
		containers = append(containers, container)
	}
	return containers, initContainers, nil
}

func (c *Caster) castEnvVarToHQ(src *repopb.EnvVar) (*hqpb.EnvVar, error) {
	rv := &hqpb.EnvVar{}
	rv.Name = src.GetName()
	valFrom := src.GetValueFrom()
	hqValFrom := &hqpb.EnvVarSource{}
	switch t := valFrom.GetType(); t {
	case repopb.EnvVarSource_SECRET_ENV:
		hqValFrom.Type = hqpb.EnvVarSource_SECRET_ENV
		secEnv := valFrom.SecretEnv
		hqSecEnv := &hqpb.SecretEnvSelector{}
		hqSecEnv.Field = secEnv.GetField()
		hqSecEnv.SecretName = secEnv.GetSecretName()
		keychan := secEnv.GetKeychainSecret()
		hqKeychan := &hqpb.KeychainSecret{}
		hqKeychan.KeychainId = keychan.GetKeychainId()
		hqKeychan.SecretId = keychan.GetSecretId()
		hqKeychan.SecretRevisionId = keychan.GetSecretRevisionId()
		hqSecEnv.KeychainSecret = hqKeychan
		hqValFrom.SecretEnv = hqSecEnv
	case repopb.EnvVarSource_VAULT_SECRET_ENV:
		hqValFrom.Type = hqpb.EnvVarSource_VAULT_SECRET_ENV
		secEnv := valFrom.VaultSecretEnv
		hqSecEnv := &hqpb.VaultSecretEnvSelector{}
		hqSecEnv.Field = secEnv.GetField()
		sec := secEnv.GetVaultSecret()
		hqSec := &hqpb.VaultSecret{}
		hqSec.SecretId = sec.GetSecretId()
		hqSec.SecretName = sec.GetSecretName()
		hqSec.SecretVer = sec.GetSecretVer()
		hqSec.DelegationToken = sec.GetDelegationToken()
		hqSecEnv.VaultSecret = hqSec
		hqValFrom.VaultSecretEnv = hqSecEnv
	case repopb.EnvVarSource_LITERAL_ENV:
		hqValFrom.Type = hqpb.EnvVarSource_LITERAL_ENV
		hqEnv := &hqpb.LiteralEnvSelector{}
		hqEnv.Value = valFrom.GetLiteralEnv().GetValue()
		hqValFrom.LiteralEnv = hqEnv
	default:
		return nil, fmt.Errorf("unknown env var source type %s", valFrom.GetType())
	}
	rv.ValueFrom = hqValFrom
	return rv, nil
}

func castTemplateVolumeToHQ(src *repopb.TemplateVolume) *hqpb.TemplateVolume {
	tpls := src.GetTemplate()
	if len(tpls) == 0 {
		return nil
	}
	rv := &hqpb.TemplateVolume{Template: make([]*hqpb.Template, 0, len(tpls))}
	for _, tpl := range tpls {
		hqTpl := &hqpb.Template{}
		hqTpl.SrcPath = tpl.GetSrcPath()
		hqTpl.DstPath = tpl.GetDstPath()
		rv.Template = append(rv.GetTemplate(), hqTpl)
	}
	return rv
}

func (c *Caster) castVolumesToHQ(src []*repopb.Volume) ([]*hqpb.Volume, error) {
	if len(src) == 0 {
		return nil, nil
	}
	rv := make([]*hqpb.Volume, 0, len(src))
	for _, v := range src {
		hqV := &hqpb.Volume{}
		hqV.Name = v.GetName()
		switch t := v.GetType(); t {
		case repopb.Volume_SECRET:
			hqV.Type = hqpb.Volume_SECRET
			hqSecV := &hqpb.SecretVolume{}
			hqSecV.SecretName = v.GetSecretVolume().GetSecretName()
			keychan := v.GetSecretVolume().GetKeychainSecret()
			hqKeychan := &hqpb.KeychainSecret{}
			hqKeychan.KeychainId = keychan.GetKeychainId()
			hqKeychan.SecretId = keychan.GetSecretId()
			hqKeychan.SecretRevisionId = keychan.GetSecretRevisionId()
			hqSecV.KeychainSecret = hqKeychan
			hqV.SecretVolume = hqSecV
		case repopb.Volume_VAULT_SECRET:
			hqV.Type = hqpb.Volume_VAULT_SECRET
			sec := v.GetVaultSecretVolume().GetVaultSecret()
			hqSec := &hqpb.VaultSecret{}
			hqSec.SecretId = sec.GetSecretId()
			hqSec.SecretName = sec.GetSecretName()
			hqSec.SecretVer = sec.GetSecretVer()
			hqSec.DelegationToken = sec.GetDelegationToken()
			hqV.VaultSecretVolume = &hqpb.VaultSecretVolume{VaultSecret: hqSec}
		case repopb.Volume_TEMPLATE:
			hqV.Type = hqpb.Volume_TEMPLATE
			hqV.TemplateVolume = castTemplateVolumeToHQ(v.GetTemplateVolume())
		case repopb.Volume_ITS:
			hqV.Type = hqpb.Volume_ITS
			its := v.GetItsVolume()
			hqITS := &hqpb.ItsVolume{}
			hqITS.ItsUrl = its.GetItsUrl()
			hqITS.MaxRetryPeriodSeconds = its.GetMaxRetryPeriodSeconds()
			hqITS.UseSharedMemory = its.GetUseSharedMemory()
			hqITS.PeriodSeconds = its.GetPeriodSeconds()
			hqV.ItsVolume = hqITS
		default:
			return nil, fmt.Errorf("unknown volume source type: %s", t)
		}
		rv = append(rv, hqV)
	}
	return rv, nil
}

func (c *Caster) castNotifyActionToHQ(src *repopb.NotifyAction) *hqpb.NotifyAction {
	if len(src.GetHandlers()) == 0 {
		return nil
	}
	rv := &hqpb.NotifyAction{}
	rv.Handlers = make([]*hqpb.Handler, 0, len(src.GetHandlers()))
	for _, h := range src.GetHandlers() {
		rv.Handlers = append(rv.Handlers, c.castHandlerToHQ(h))
	}
	return rv
}

func (c *Caster) makeExtractContainersForResource(jugglerPath string, resPath *resourcePath, secCtx *hqpb.SecurityContext) []*hqpb.Container {
	if resPath.ExtractPath == "" {
		return nil
	}
	rv := make([]*hqpb.Container, 0, 2)
	path := strings.ReplaceAll(resPath.ExtractPath, "{JUGGLER_CHECKS_PATH}", fmt.Sprintf("%s/checks", jugglerPath))
	c1 := &hqpb.Container{}
	c1.Name = fmt.Sprintf("mkdir_%s", resPath.LocalPath)
	c1.Command = []string{"mkdir", "-p", path}
	if secCtx != nil {
		c1.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
	}
	rv = append(rv, c1)
	c2 := &hqpb.Container{}
	c2.Name = fmt.Sprintf("extract_%s", resPath.LocalPath)
	c2.Command = []string{"tar", "xf", resPath.LocalPath, "-C", path}
	if secCtx != nil {
		c2.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
	}
	rv = append(rv, c2)
	return rv
}

func (c *Caster) makeExtractResourcesContainers(ra *nannyclient.RuntimeAttrs, rootFSEnabled bool, secCtx *hqpb.SecurityContext) []*hqpb.Container {
	jPath := makeJugglerRootPath(rootFSEnabled)
	rps := collectResourcePaths(ra)
	if len(rps) == 0 {
		return nil
	}
	containers := make([]*hqpb.Container, 0, len(rps)*2)
	for _, rp := range rps {
		containers = append(containers, c.makeExtractContainersForResource(jPath, rp, secCtx)...)
	}
	return containers
}

func castHTTPHeadersToHQ(src []*repopb.HTTPHeader) []*hqpb.HTTPHeader {
	if len(src) == 0 {
		return nil
	}
	rv := make([]*hqpb.HTTPHeader, 0, len(src))
	for _, h := range src {
		hqH := &hqpb.HTTPHeader{}
		hqH.Name = h.GetName()
		hqH.Value = h.GetValue()
		rv = append(rv, hqH)
	}
	return rv
}

func (c *Caster) castHandlerToHQ(src *repopb.Handler) *hqpb.Handler {
	rv := &hqpb.Handler{}
	switch t := src.GetType(); t {
	case repopb.Handler_NONE:
		rv.Type = hqpb.Handler_NONE
	case repopb.Handler_EXEC:
		rv.Type = hqpb.Handler_EXEC
		exec := &hqpb.ExecAction{}
		if len(src.GetExecAction().GetCommand()) != 0 {
			exec.Command = make([]string, 0, len(src.GetExecAction().GetCommand()))
			exec.Command = append(exec.Command, src.GetExecAction().GetCommand()...)
		}
		rv.ExecAction = exec
	case repopb.Handler_TCP_SOCKET:
		rv.Type = hqpb.Handler_TCP_SOCKET
		tcp := &hqpb.TCPSocketAction{}
		tcp.Port = src.GetTcpSocket().Port
		tcp.Host = src.GetTcpSocket().Host
		rv.TcpSocket = tcp
	case repopb.Handler_HTTP_GET:
		rv.Type = hqpb.Handler_HTTP_GET
		http := &hqpb.HTTPGetAction{}
		http.Path = src.GetHttpGet().GetPath()
		http.Port = src.GetHttpGet().GetPort()
		http.Host = src.GetHttpGet().GetHost()
		http.UriScheme = src.GetHttpGet().GetUriScheme()
		http.HttpHeaders = castHTTPHeadersToHQ(src.GetHttpGet().GetHttpHeaders())
		rv.HttpGet = http
	}
	return rv
}

func (c *Caster) castCoredumpPolicyToHQ(src *repopb.CoredumpPolicy, ictlVersion string, gdbSource *repopb.GdbSource) (*hqpb.CoredumpPolicy, error) {
	rv := &hqpb.CoredumpPolicy{}
	switch t := src.GetType(); t {
	case repopb.CoredumpPolicy_COREDUMP:
		rv.Type = hqpb.CoredumpPolicy_COREDUMP
		proc := src.GetCoredumpProcessor()
		hqProc := &hqpb.CoredumpProcessor{}
		hqProc.Path = proc.GetPath()
		hqProc.TotalSizeLimit = proc.GetTotalSizeLimit()
		hqProc.CountLimit = proc.GetCountLimit()
		hqProc.Probability = proc.GetProbability()

		pol := proc.GetCleanupPolicy()
		if pol.GetType() == repopb.CoredumpCleanupPolicy_TTL {
			hqPol := &hqpb.CoredumpCleanupPolicy{}
			hqPol.Type = hqpb.CoredumpCleanupPolicy_Type(pol.GetType())
			hqPol.Ttl = &hqpb.CoredumpCleanupPolicyTtl{}
			hqPol.Ttl.Seconds = int64(pol.GetTtl().GetSeconds())
			hqProc.CleanupPolicy = hqPol
		} else if pol.GetType() != repopb.CoredumpCleanupPolicy_DISABLED {
			return nil, fmt.Errorf("uknown coredump cleanup policy type: %s", pol.GetType())
		}

		agg := proc.GetAggregator()
		if agg.GetType() == repopb.CoredumpAggregator_SAAS_AGGREGATOR {
			hqAgg := &hqpb.CoredumpAggregator{}
			hqAgg.Type = hqpb.CoredumpAggregator_Type(agg.GetType())
			saas := agg.GetSaas()
			hqSaas := &hqpb.SaasCoredumpAggregator{}
			hqSaas.Url = saas.GetUrl()
			hqSaas.ServiceName = saas.GetServiceName()
			if saas.GetGdb().GetType() == repopb.CoredumpAggregatorGdb_LITERAL {
				hqSaas.Gdb = &hqpb.CoredumpAggregatorGdb{}
				if builtinGDBMustBeUsed(ictlVersion, gdbSource) {
					hqSaas.Gdb.ExecPath = SandboxGDBPath
				} else {
					hqSaas.Gdb.ExecPath = saas.GetGdb().GetExecPath()
				}
				hqAgg.Saas = hqSaas
			} else {
				return nil, fmt.Errorf("unknown coredump aggregator gdb type: %s", saas.GetGdb().GetType())
			}
			hqProc.Aggregator = hqAgg
		} else if agg.GetType() != repopb.CoredumpAggregator_DISABLED {
			return nil, fmt.Errorf("unknown coredump aggregator type: %s", agg.GetType())
		}
		rv.CoredumpProcessor = hqProc
	case repopb.CoredumpPolicy_CUSTOM_CORE_COMMAND:
		rv.Type = hqpb.CoredumpPolicy_CUSTOM_CORE_COMMAND
		rv.CustomProcessor = &hqpb.CustomCommandCoredumpProcessor{}
		rv.GetCustomProcessor().Command = src.GetCustomProcessor().GetCommand()
	case repopb.CoredumpPolicy_NONE:
		return nil, nil
	default:
		return nil, fmt.Errorf("unknown coredump policy type: %s", src.GetType())
	}
	return rv, nil
}

func castComputeResourcesToHQ(src []*repopb.ComputeResource) ([]*hqpb.ComputeResource, error) {
	if len(src) == 0 {
		return nil, nil
	}
	rv := make([]*hqpb.ComputeResource, 0, len(src))
	for _, r := range src {
		if r.GetType() != repopb.ComputeResource_SCALAR {
			return nil, fmt.Errorf("unknown compute resource type: %s", r.GetType())
		}
		hqR := &hqpb.ComputeResource{}
		hqR.Type = hqpb.ComputeResource_Type(r.GetType())
		hqR.Name = r.GetName()
		hqR.Scalar = &hqpb.Scalar{}
		hqR.GetScalar().Value = r.GetScalar().GetValue()
		rv = append(rv, hqR)
	}
	return rv, nil
}

func (c *Caster) castResourceRequestToHQ(src *repopb.ResourceRequest) (*hqpb.ResourceAllocation, error) {
	if len(src.GetLimit()) == 0 && len(src.GetRequest()) == 0 {
		return nil, nil
	}
	rv := &hqpb.ResourceAllocation{}
	limit, err := castComputeResourcesToHQ(src.GetLimit())
	if err != nil {
		return nil, fmt.Errorf("failed casting resource limit: %w", err)
	}
	rv.Limit = limit
	req, err := castComputeResourcesToHQ(src.GetRequest())
	if err != nil {
		return nil, fmt.Errorf("failed casting resource request: %w", err)
	}
	rv.Request = req
	return rv, nil
}

func (c *Caster) castEnvToHQ(src []*repopb.EnvVar) ([]*hqpb.EnvVar, error) {
	if len(src) == 0 {
		return nil, nil
	}
	rv := make([]*hqpb.EnvVar, 0, len(src))
	for _, e := range src {
		d, err := c.castEnvVarToHQ(e)
		if err != nil {
			return nil, fmt.Errorf("failed casting env var %s to HQ: %w", e.GetName(), err)
		}
		rv = append(rv, d)
	}
	return rv, nil

}

func (c *Caster) castHandlersToHQ(src []*repopb.Handler) []*hqpb.Handler {
	if len(src) == 0 {
		return nil
	}
	rv := make([]*hqpb.Handler, 0, len(src))
	for _, h := range src {
		rv = append(rv, c.castHandlerToHQ(h))
	}
	return rv
}

func (c *Caster) castContainersToHQ(src []*repopb.Container, ictlVersion string, gdbSource *repopb.GdbSource, secCtx *hqpb.SecurityContext) ([]*hqpb.Container, error) {
	if len(src) == 0 {
		return nil, nil
	}
	rv := make([]*hqpb.Container, 0, len(src))
	for _, container := range src {
		r := &hqpb.Container{}
		r.Name = container.GetName()
		if len(container.GetCommand()) != 0 {
			r.Command = make([]string, 0, len(container.GetCommand()))
			r.Command = append(r.Command, container.GetCommand()...)
		}
		hqEnv, err := c.castEnvToHQ(container.GetEnv())
		if err != nil {
			return nil, fmt.Errorf("failed casting container %s: %w", container.GetName(), err)
		}
		r.Env = hqEnv
		probe := container.GetReadinessProbe()
		hqProbe := &hqpb.Probe{}
		hqProbe.InitialDelaySeconds = probe.GetInitialDelaySeconds()
		hqProbe.MinPeriodSeconds = probe.GetMinPeriodSeconds()
		hqProbe.MaxPeriodSeconds = probe.GetMaxPeriodSeconds()
		hqProbe.PeriodBackoff = probe.GetPeriodBackoff()
		hqProbe.SuccessThreshold = probe.GetSuccessThreshold()
		hqProbe.FailureThreshold = probe.GetFailureThreshold()
		hqProbe.Handlers = c.castHandlersToHQ(probe.GetHandlers())
		r.ReadinessProbe = hqProbe

		pol := container.GetRestartPolicy()
		hqPol := &hqpb.RestartPolicy{}
		switch t := pol.GetType(); t {
		case repopb.RestartPolicy_ALWAYS:
			hqPol.Type = hqpb.RestartPolicy_ALWAYS
		case repopb.RestartPolicy_ON_FAILURE:
			hqPol.Type = hqpb.RestartPolicy_ON_FAILURE
		case repopb.RestartPolicy_NEVER:
			hqPol.Type = hqpb.RestartPolicy_NEVER
		}
		hqPol.MinPeriodSeconds = pol.GetMinPeriodSeconds()
		hqPol.MaxPeriodSeconds = pol.GetMaxPeriodSeconds()
		hqPol.PeriodBackoff = pol.GetPeriodBackoff()
		hqPol.PeriodJitterSeconds = pol.GetPeriodJitterSeconds()
		r.RestartPolicy = hqPol

		r.Lifecycle = &hqpb.Lifecycle{}
		r.Lifecycle.PreStop = c.castHandlerToHQ(container.GetLifecycle().GetPreStop())
		r.Lifecycle.StopGracePeriodSeconds = container.GetLifecycle().GetStopGracePeriodSeconds()
		r.Lifecycle.TerminationGracePeriodSeconds = container.GetLifecycle().GetTerminationGracePeriodSeconds()
		r.Lifecycle.TermBarrier = hqpb.Lifecycle_TermBarrier(container.GetLifecycle().GetTermBarrier())

		if container.GetReopenLogAction() != nil {
			r.ReopenLogAction = &hqpb.ReopenLogAction{}
			r.ReopenLogAction.Handler = c.castHandlerToHQ(container.GetReopenLogAction().GetHandler())
		}

		r.SecurityContext = &hqpb.SecurityContext{}
		if secCtx != nil {
			copy(r.SecurityContext.Capabilities, secCtx.GetCapabilities())
			r.SecurityContext.RunAsUser = secCtx.GetRunAsUser()
			r.SecurityContext.PortoAccessPolicy = secCtx.GetPortoAccessPolicy()
		} else {
			r.SecurityContext.RunAsUser = container.GetSecurityPolicy().GetRunAsUser()
		}

		p, err := c.castCoredumpPolicyToHQ(container.CoredumpPolicy, ictlVersion, gdbSource)
		if err != nil {
			return nil, fmt.Errorf("failed casting coredump policy for container %s to HQ: %w", container.GetName(), err)
		}
		r.CoredumpPolicy = p

		a, err := c.castResourceRequestToHQ(container.GetResourceRequest())
		if err != nil {
			return nil, fmt.Errorf("failed casting resource request to HQ: %w", err)
		}
		r.ResourceAllocation = a
		rv = append(rv, r)
	}
	return rv, nil
}

func (c *Caster) castDockerLayersToHQ(instanceSpec instancespec.Interface, ra *nannyclient.RuntimeAttrs, logSize uint64) (*hqpb.InstanceRevision, error) {
	rv := &hqpb.InstanceRevision{}
	rv.Type = hqpb.InstanceRevision_NO_SUBCONTAINERS
	ictlVersion := instanceSpec.GetInstanceCtl().GetVersion()
	gdb := instanceSpec.GetInstanceSpec().GetGdbSource()
	containers, err := c.castContainersToHQ(instanceSpec.GetInstanceSpec().GetContainers(), ictlVersion, gdb, nil)
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec containers to HQ: %w", err)
	}
	initContainers, err := c.castContainersToHQ(instanceSpec.GetInitContainers(), ictlVersion, gdb, nil)
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec init containers to HQ: %w", err)
	}
	auxConts, auxInitConts, err := c.makeAUXDaemonsContainers(instanceSpec.GetAUXDaemons(), true, logSize, ictlVersion, nil)
	if err != nil {
		return nil, fmt.Errorf("failed casting AUX daemons containers: %w", err)
	}
	extractInitConts := c.makeExtractResourcesContainers(ra, true, nil)
	if len(containers) != 0 || len(auxConts) != 0 {
		rv.Container = make([]*hqpb.Container, 0, len(containers)+len(auxConts))
		rv.Container = append(rv.GetContainer(), containers...)
		rv.Container = append(rv.GetContainer(), auxConts...)
	}
	if len(auxInitConts) != 0 || len(initContainers) != 0 || len(extractInitConts) != 0 {
		rv.InitContainers = make([]*hqpb.Container, 0, len(auxInitConts)+len(initContainers)+len(extractInitConts))
		rv.InitContainers = append(rv.GetInitContainers(), auxInitConts...)
		rv.InitContainers = append(rv.GetInitContainers(), initContainers...)
		rv.InitContainers = append(rv.GetInitContainers(), extractInitConts...)
	}

	volumes, err := c.castVolumesToHQ(instanceSpec.GetVolume())
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec volumes to HQ: %w", err)
	}
	rv.Volume = volumes

	rv.NotifyAction = c.castNotifyActionToHQ(instanceSpec.GetNotifyAction())
	return rv, nil
}

func (c *Caster) castSanboxLayers(instanceSpec instancespec.Interface, ra *nannyclient.RuntimeAttrs, logSize uint64) (*hqpb.InstanceRevision, error) {
	rv := &hqpb.InstanceRevision{}
	rv.Type = hqpb.InstanceRevision_NO_SUBCONTAINERS
	ictlVersion := instanceSpec.GetInstanceCtl().GetVersion()
	gdb := instanceSpec.GetInstanceSpec().GetGdbSource()
	containers, err := c.castContainersToHQ(instanceSpec.GetInstanceSpec().GetContainers(), ictlVersion, gdb, nil)
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec containers to HQ: %w", err)
	}
	initContainers, err := c.castContainersToHQ(instanceSpec.GetInitContainers(), ictlVersion, gdb, nil)
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec init containers to HQ: %w", err)
	}
	rootFSEnabled := len(instanceSpec.GetLayersConfig().GetLayer()) != 0
	auxConts, auxInitConts, err := c.makeAUXDaemonsContainers(instanceSpec.GetAUXDaemons(), rootFSEnabled, logSize, ictlVersion, nil)
	if err != nil {
		return nil, fmt.Errorf("failed casting AUX daemons containers: %w", err)
	}
	extractInitConts := c.makeExtractResourcesContainers(ra, rootFSEnabled, nil)
	if len(containers) != 0 || len(auxConts) != 0 {
		rv.Container = make([]*hqpb.Container, 0, len(containers)+len(auxConts))
		rv.Container = append(rv.GetContainer(), containers...)
		rv.Container = append(rv.GetContainer(), auxConts...)
	}
	if len(auxInitConts) != 0 || len(initContainers) != 0 || len(extractInitConts) != 0 {
		rv.InitContainers = make([]*hqpb.Container, 0, len(auxInitConts)+len(initContainers)+len(extractInitConts))
		rv.InitContainers = append(rv.GetInitContainers(), auxInitConts...)
		rv.InitContainers = append(rv.GetInitContainers(), initContainers...)
		rv.InitContainers = append(rv.GetInitContainers(), extractInitConts...)
	}

	volumes, err := c.castVolumesToHQ(instanceSpec.GetVolume())
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec volumes to HQ: %w", err)
	}
	rv.Volume = volumes

	rv.NotifyAction = c.castNotifyActionToHQ(instanceSpec.GetNotifyAction())
	return rv, nil
}

func (c *Caster) castQemuKVMToHQ() *hqpb.InstanceRevision {
	rv := &hqpb.InstanceRevision{}
	rv.Type = hqpb.InstanceRevision_APP_CONTAINER
	container := &hqpb.Container{}
	container.Name = "qemu-launcher"
	container.Command = []string{"/usr/sbin/qemu_launcher.sh"}
	container.RestartPolicy = proto.Clone(c.defaultAUXDaemonsRestartPolicy).(*hqpb.RestartPolicy)
	container.Lifecycle = proto.Clone(c.defaultAUXDaemonsLifecycle).(*hqpb.Lifecycle)
	secCtx := &hqpb.SecurityContext{}
	secCtx.PortoAccessPolicy = "isolate"
	container.SecurityContext = secCtx
	e := &hqpb.EnvVar{}
	e.Name = "IMG_PATH"
	v := &hqpb.EnvVarSource{}
	v.Type = hqpb.EnvVarSource_LITERAL_ENV
	v.LiteralEnv = &hqpb.LiteralEnvSelector{Value: "/image"}
	e.ValueFrom = v
	container.Env = []*hqpb.EnvVar{e}
	rv.Container = []*hqpb.Container{container}
	ic := &hqpb.Container{}
	ic.Name = "copy-image"
	ic.Command = []string{"/bin/bash", "-c", "ln -s \\$(pwd)/image /image"}
	rv.InitContainers = []*hqpb.Container{ic}
	return rv
}

func (c *Caster) castOSContainerToHQ(instanceSpec instancespec.Interface, ra *nannyclient.RuntimeAttrs, logSize uint64) (*hqpb.InstanceRevision, error) {
	rv := &hqpb.InstanceRevision{}
	rv.Type = hqpb.InstanceRevision_OS_CONTAINER

	ictlVersion := instanceSpec.GetInstanceCtl().GetVersion()
	gdb := instanceSpec.GetInstanceSpec().GetGdbSource()
	secCtx := &hqpb.SecurityContext{
		RunAsUser:         "root",
		PortoAccessPolicy: "isolate",
	}
	sbinCont := &hqpb.Container{}
	sbinCont.Name = "os"
	sbinCont.Command = []string{"/sbin/init"}
	sbinCont.ReadinessProbe = proto.Clone(c.defaultAUXDaemonsReadinessProbe).(*hqpb.Probe)
	sbinCont.RestartPolicy = proto.Clone(c.defaultAUXDaemonsRestartPolicy).(*hqpb.RestartPolicy)
	sbinCont.Lifecycle = proto.Clone(c.defaultAUXDaemonsLifecycle).(*hqpb.Lifecycle)
	sbinCont.SecurityContext = proto.Clone(secCtx).(*hqpb.SecurityContext)
	initContainers, err := c.castContainersToHQ(instanceSpec.GetInitContainers(), ictlVersion, gdb, secCtx)
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec init containers to HQ: %w", err)
	}
	auxConts, auxInitConts, err := c.makeAUXDaemonsContainers(instanceSpec.GetAUXDaemons(), true, logSize, ictlVersion, secCtx)
	if err != nil {
		return nil, fmt.Errorf("failed casting AUX daemons containers: %w", err)
	}
	extractInitConts := c.makeExtractResourcesContainers(ra, true, secCtx)
	rv.Container = make([]*hqpb.Container, 0, len(auxConts)+1)
	rv.Container = append(rv.GetContainer(), sbinCont)
	rv.Container = append(rv.GetContainer(), auxConts...)
	if len(auxInitConts) != 0 || len(initContainers) != 0 || len(extractInitConts) != 0 {
		rv.InitContainers = make([]*hqpb.Container, 0, len(auxInitConts)+len(initContainers)+len(extractInitConts))
		rv.InitContainers = append(rv.GetInitContainers(), auxInitConts...)
		rv.InitContainers = append(rv.GetInitContainers(), initContainers...)
		rv.InitContainers = append(rv.GetInitContainers(), extractInitConts...)
	}

	volumes, err := c.castVolumesToHQ(instanceSpec.GetVolume())
	if err != nil {
		return nil, fmt.Errorf("failed casting instance spec volumes to HQ: %w", err)
	}
	rv.Volume = volumes

	rv.NotifyAction = c.castNotifyActionToHQ(instanceSpec.GetNotifyAction())
	return rv, nil
}

func (c *Caster) Cast(instanceSpec instancespec.Interface, ra *nannyclient.RuntimeAttrs, logSize uint64) (*hqpb.InstanceRevision, error) {
	switch t := instanceSpec.GetInstanceSpec().GetType(); t {
	case repopb.InstanceSpec_DOCKER_LAYERS:
		return c.castDockerLayersToHQ(instanceSpec, ra, logSize)
	case repopb.InstanceSpec_SANDBOX_LAYERS:
		return c.castSanboxLayers(instanceSpec, ra, logSize)
	case repopb.InstanceSpec_QEMU_KVM:
		return c.castQemuKVMToHQ(), nil
	case repopb.InstanceSpec_OS_CONTAINER:
		return c.castOSContainerToHQ(instanceSpec, ra, logSize)
	default:
		return nil, fmt.Errorf("unknown instance spec type: %s", t)
	}
}
