package log

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/log/lumberjack"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	zp "go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

type Config struct {
	Filename   string         `yaml:"filename"`
	MaxSize    int            `yaml:"maxsize"`
	MaxBackups int            `yaml:"maxbackups"`
	Level      zp.AtomicLevel `yaml:"level"`
}

var encCfg = zapcore.EncoderConfig{
	MessageKey:     "msg",
	LevelKey:       "level",
	TimeKey:        "ts",
	CallerKey:      "caller",
	EncodeLevel:    zapcore.CapitalLevelEncoder,
	EncodeTime:     zapcore.ISO8601TimeEncoder,
	EncodeDuration: zapcore.StringDurationEncoder,
	EncodeCaller:   zapcore.ShortCallerEncoder,
}

func (cfg *Config) openSinks() (zapcore.WriteSyncer, zapcore.WriteSyncer, error) {
	errSink, errClose, err := zp.Open("stderr")
	if err != nil {
		return nil, nil, err
	}
	var sink zapcore.WriteSyncer
	if cfg.Filename == "stdout" {
		sink, _, err = zp.Open(cfg.Filename)
		if err != nil {
			errClose()
			return nil, nil, err
		}
	} else {
		sink = zapcore.AddSync(&lumberjack.Logger{
			Filename:   cfg.Filename,
			MaxSize:    cfg.MaxSize, // megabytes
			MaxBackups: cfg.MaxBackups,
		})
	}
	return sink, errSink, nil
}

func SetupLogger(cfg *Config) *zap.Logger {
	sink, errSink, err := cfg.openSinks()
	if err != nil {
		panic(err)
	}
	core := zapcore.NewCore(
		zapcore.NewConsoleEncoder(encCfg),
		sink,
		cfg.Level,
	)
	logger = &zap.Logger{L: zp.New(core, zp.ErrorOutput(errSink))}
	logger.AddCallerSkip(1)
	return logger
}

var logger *zap.Logger

func Debug(msg string, fields ...log.Field) {
	logger.Debug(msg, fields...)
}

func Info(msg string, fields ...log.Field) {
	logger.Info(msg, fields...)
}

func Warn(msg string, fields ...log.Field) {
	logger.Warn(msg, fields...)
}

func Error(msg string, fields ...log.Field) {
	logger.Error(msg, fields...)
}

func Fatal(msg string, fields ...log.Field) {
	logger.Fatal(msg, fields...)
}

func Debugf(msg string, args ...interface{}) {
	logger.Debugf(msg, args...)
}

func Infof(msg string, args ...interface{}) {
	logger.Infof(msg, args...)
}

func Warnf(msg string, args ...interface{}) {
	logger.Warnf(msg, args...)
}

func Errorf(msg string, args ...interface{}) {
	logger.Errorf(msg, args...)
}

func Fatalf(msg string, args ...interface{}) {
	logger.Fatalf(msg, args...)
}
