package hqsender

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/iss/confgen"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/log"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/nanny/servicefetcher"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	hqrpc "a.yandex-team.ru/infra/nanny/go/hq"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	hqpb "a.yandex-team.ru/yp/go/proto/hq"
	"context"
	"fmt"
	"strings"
)

func makeAddInstanceRevReq(conf *clusterapi.HostConfigurationInstance) *hqpb.AddInstanceRevRequest {
	rev := conf.GetInstanceRevision()
	return &hqpb.AddInstanceRevRequest{
		Id:       conf.Properties["HQ_INSTANCE_ID"],
		Rev:      rev,
		Hostname: rev.GetHostname(),
		HostnameVersion: &hqpb.HostnameVersion{
			Major: 0,
			Minor: 0,
		},
	}
}

func makeComputeResources(mem, cpu int64, setIO bool) []*hqpb.ComputeResource {
	var rv []*hqpb.ComputeResource
	if setIO {
		rv = make([]*hqpb.ComputeResource, 0, 5)
	} else {
		rv = make([]*hqpb.ComputeResource, 0, 4)
	}
	rv = append(rv,
		&hqpb.ComputeResource{
			Type: hqpb.ComputeResource_SCALAR,
			Name: "mem",
			Scalar: &hqpb.Scalar{
				Value: mem,
			},
		},
		&hqpb.ComputeResource{
			Type: hqpb.ComputeResource_SCALAR,
			Name: "cpu",
			Scalar: &hqpb.Scalar{
				Value: cpu,
			},
		},
		&hqpb.ComputeResource{
			Type: hqpb.ComputeResource_SCALAR,
			Name: "net",
			Scalar: &hqpb.Scalar{
				Value: 0,
			},
		},
		&hqpb.ComputeResource{
			Type: hqpb.ComputeResource_SCALAR,
			Name: "hdd",
			Scalar: &hqpb.Scalar{
				Value: 0,
			},
		},
	)
	if setIO {
		rv = append(rv,
			&hqpb.ComputeResource{
				Type: hqpb.ComputeResource_SCALAR,
				Name: "io",
				Scalar: &hqpb.Scalar{
					Value: 0,
				},
			},
		)
	}
	return rv
}

func makeCreateInstanceRevReq(conf *clusterapi.HostConfigurationInstance, pod *api.Pod) *hqpb.CreateInstanceRequest {
	rev := conf.GetInstanceRevision()
	rr := pod.GetSpec().GetResourceRequests()
	a := &hqpb.ResourceAllocation{
		Port: []*hqpb.Port{
			{
				Name:     "default",
				Port:     confgen.DefaultYPInstancePort,
				Protocol: "TCP",
			},
		},
		Request: makeComputeResources(int64(rr.GetMemoryGuarantee()), int64(rr.GetVcpuGuarantee()), false),
		Limit:   makeComputeResources(int64(rr.GetMemoryLimit()), int64(rr.GetVcpuLimit()), true),
	}
	return &hqpb.CreateInstanceRequest{
		Meta: &hqpb.InstanceMeta{
			Id:        conf.Properties["HQ_INSTANCE_ID"],
			ServiceId: conf.GetId().GetConfiguration().GetGroupId(),
		},
		Spec: &hqpb.InstanceSpec{
			NodeName: pod.GetSpec().GetNodeId(),
			Hostname: rev.GetHostname(),
			HostnameVersion: &hqpb.HostnameVersion{
				Major: 0,
				Minor: 0,
			},
			Allocation: a,
			Revision:   []*hqpb.InstanceRevision{rev},
		},
	}
}

type Sender struct {
	hqFactory *hqrpc.ClientFactory
}

func NewSenderFromConfig(cfg *hqrpc.Config) *Sender {
	return &Sender{
		hqFactory: hqrpc.NewClientFactoryFromConfig(cfg),
	}
}

func isConflict(err error) bool {
	cerr, ok := err.(*rpc.ClientError)
	if !ok {
		return false
	}
	return cerr.Code == 409
}

// Double check before sending to HQ
func validateHQInstanceID(hqInstanceID string, pod *api.Pod, s *servicefetcher.Service) error {
	if !s.ManagedByAllocation() {
		return fmt.Errorf("service %s is not managed by allocation", s.ID)
	}
	p := strings.SplitN(hqInstanceID, "@", 2)
	if !strings.HasPrefix(p[0], pod.GetMeta().GetId()) {
		return fmt.Errorf("HQ instance ID %s does not start with pod ID %s", p[0], pod.GetMeta().GetId())
	}
	if s.ID != p[1] {
		return fmt.Errorf("service ID and HQ instance ID mismatched: %s != %s", s.ID, p[1])
	}
	return nil
}

func validateCreateConf(conf *clusterapi.HostConfigurationInstance, pod *api.Pod, s *servicefetcher.Service) error {
	if !s.ManagedByAllocation() {
		return fmt.Errorf("service %s is not managed by allocation", s.ID)
	}
	if err := validateHQInstanceID(conf.GetProperties()["HQ_INSTANCE_ID"], pod, s); err != nil {
		return err
	}

	found := false
	for _, sn := range s.Snapshots {
		if conf.GetId().GetConfiguration().GetGroupStateFingerprint() == sn.RepoSnapshot.SnapshotMetaInfo.ConfId {
			found = true
			break
		}
	}
	if !found {
		return fmt.Errorf("service %s has no snapshot for conf ID %s", s.ID, conf.GetId().GetConfiguration().GetGroupStateFingerprint())
	}
	return nil
}

func validateDeleteInstance(hqInstanceID string, pod *api.Pod, s *servicefetcher.Service) error {
	if !s.ManagedByAllocation() {
		return fmt.Errorf("service %s is not managed by allocation", s.ID)
	}
	return validateHQInstanceID(hqInstanceID, pod, s)
}

func (s *Sender) SendRevision(ctx context.Context, cluster string, conf *clusterapi.HostConfigurationInstance, pod *api.Pod, service *servicefetcher.Service) error {
	if err := validateCreateConf(conf, pod, service); err != nil {
		return fmt.Errorf("conf %s validation failed: %w", conf.GetId().GetConfiguration().GetGroupStateFingerprint(), err)
	}
	c, found := s.hqFactory.Get(cluster)
	if !found {
		return fmt.Errorf("HQ client for cluster %s not found", cluster)
	}
	addReq := makeAddInstanceRevReq(conf)
	for {
		_, err := c.AddInstanceRev(ctx, addReq)
		if err != nil {
			cerr, ok := err.(*rpc.ClientError)
			if !ok {
				return fmt.Errorf("failed adding instance revision: %w", err)
			}
			if cerr.Code == 404 {
				createReq := makeCreateInstanceRevReq(conf, pod)
				_, err := c.CreateInstance(ctx, createReq)
				// just ignore conflict error on create request and continue retrying add request
				if err != nil && !isConflict(err) {
					return fmt.Errorf("failed creating instance revision: %w", err)
				}

			} else if cerr.Code == 409 {
				// stop retrying on conflict error
				log.Warnf("instance revision %s already exists in HQ", conf.GetInstanceRevision().GetId())
				return nil
			} else {
				return fmt.Errorf("failed adding instance revision: %w", err)
			}
		} else {
			return nil
		}
	}
}

func (s *Sender) DeleteInstance(ctx context.Context, cluster string, hqInstanceID string, pod *api.Pod, service *servicefetcher.Service) error {
	if err := validateDeleteInstance(hqInstanceID, pod, service); err != nil {
		return fmt.Errorf("validation failed: %w", err)
	}
	c, found := s.hqFactory.Get(cluster)
	if !found {
		return fmt.Errorf("HQ client for cluster %s not found", cluster)
	}
	req := &hqpb.DeleteInstanceRequest{Name: hqInstanceID}
	if _, err := c.DeleteInstance(ctx, req); err != nil {
		return err
	}
	return nil
}
