package nannyutil

import (
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	nannyrpc "a.yandex-team.ru/infra/nanny/go/nanny"
	internalpb "a.yandex-team.ru/infra/nanny/go/proto/nanny_internal"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"context"
	"crypto/md5"
	"fmt"
	"github.com/golang/protobuf/proto"
	"strconv"
	"strings"
)

func GetInstanceCtlTaskID(ra *nannyclient.RuntimeAttrs, s *repopb.InstanceCtl) (int, error) {
	t := ""
	for _, r := range ra.Content.Resources.SandboxFiles {
		m := (r.LocalPath == "instancectl" || r.LocalPath == "loop-httpsearch") &&
			r.TaskType == "BUILD_INSTANCE_CTL" &&
			r.ResourceType == "INSTANCECTL"
		if m {
			t = r.TaskID
			break
		}
	}
	if t == "" {
		t = s.FetchableMeta.SandboxResource.TaskId
	}
	if t == "" {
		return 0, fmt.Errorf("failed to find InstanceCtl resource")
	}
	i, err := strconv.Atoi(t)
	if err != nil {
		return i, err
	}
	return i, nil
}

func ComputeInstanceSpecHashMD5(s *repopb.InstanceSpec) ([md5.Size]byte, error) {
	var h [md5.Size]byte
	// TODO: make sure Golang's proto.Marshal is equal to Python's SerializeToString
	b := &proto.Buffer{}
	b.SetDeterministic(true)
	err := b.Marshal(s)
	if err != nil {
		return h, fmt.Errorf("failed marshalling InstanceSpec proto: %w", err)
	}
	h = md5.Sum(b.Bytes())
	return h, nil
}

func GetSnapshotInstanceSpec(ctx context.Context, c *nannyrpc.NannyRPCClient, snapshotID string) (*repopb.InstanceSpec, error) {
	req := &repopb.GetSnapshotInstanceSpecRequest{}
	req.SnapshotId = snapshotID
	rsp, err := c.GetSnapshotInstanceSpec(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.InstanceSpec, nil
}

func GetISSConfTemplate(ctx context.Context, c *nannyrpc.NannyRPCClient, snapshotID string) (*internalpb.IssConfTemplate, error) {
	req := &internalpb.GetIssConfTemplateRequest{}
	req.SnapshotId = snapshotID
	rsp, err := c.GetIssConfTemplate(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetTemplate(), nil
}

func ParseInstanceCtlVersion(v string) (uint64, uint64) {
	if v == "" {
		return 0, 0
	}
	parts := strings.Split(v, ".")
	if len(parts) != 2 {
		return 0, 0
	}
	major, err := strconv.ParseUint(parts[0], 10, 64)
	if err != nil {
		return 0, 0
	}
	minor, err := strconv.ParseUint(parts[1], 10, 64)
	if err != nil {
		return 0, 0
	}
	return major, minor
}

func CompareInstanceCtlVersions(v1 string, v2 string) int {
	maj1, min1 := ParseInstanceCtlVersion(v1)
	maj2, min2 := ParseInstanceCtlVersion(v2)
	if maj1 == maj2 {
		if min1 > min2 {
			return 1
		} else if min1 < min2 {
			return -1
		}
		return 0
	}
	if maj1 > maj2 {
		return 1
	}
	return -1
}
