package servicefetcher

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/instancespec"
	nannyclient "a.yandex-team.ru/infra/nanny/go/client"
	nannyrpc "a.yandex-team.ru/infra/nanny/go/nanny"
	internalpb "a.yandex-team.ru/infra/nanny/go/proto/nanny_internal"
	repopb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"context"
)

type snapshot struct {
	ID           string
	RepoSnapshot *repopb.Snapshot
	RuntimeAttrs *nannyclient.RuntimeAttrs
	InstanceSpec instancespec.Interface
	ISSTemplate  *internalpb.IssConfTemplate
}

type Service struct {
	ID        string
	Snapshots []*snapshot
}

func (s *Service) ManagedByAllocation() bool {
	hasAlloc := false
	for _, sn := range s.Snapshots {
		ct, _ := sn.RuntimeAttrs.Content.Instances.ChosenTypeMapped()
		if sn.RepoSnapshot.Target == repopb.Snapshot_ACTIVE {
			return ct == nannyclient.InstanceTypeAllocationPods
		}
		if ct == nannyclient.InstanceTypeAllocationPods {
			hasAlloc = true
		}
	}
	// Service has no ACTIVE snapshots, so we consider it is managed by Allocation if it has at least one snapshot
	// with ALLOCATION_PODS chosen type.
	return hasAlloc
}

type ServiceFetcher struct {
	nannyClient    *nannyclient.NannyClient
	nannyRPCClient *nannyrpc.NannyRPCClient
}

func NewServiceFetcher(nannyClient *nannyclient.NannyClient, nannyRPCClient *nannyrpc.NannyRPCClient) *ServiceFetcher {
	return &ServiceFetcher{
		nannyClient:    nannyClient,
		nannyRPCClient: nannyRPCClient,
	}
}

func (f *ServiceFetcher) FetchService(ctx context.Context, ID string) (*Service, error) {
	rv := &Service{ID: ID}
	req := &repopb.GetServiceRequest{}
	req.ServiceId = ID
	rsp, err := f.nannyRPCClient.GetService(ctx, req)
	if err != nil {
		return nil, err
	}
	snapshots := make([]*snapshot, 0, len(rsp.Service.GetSpec().GetSnapshot()))
	for _, repoSn := range rsp.Service.GetSpec().GetSnapshot() {
		sn := &snapshot{}
		sn.ID = repoSn.GetId()
		sn.RepoSnapshot = repoSn
		ra, err := f.nannyClient.GetRuntimeAttributeSnapshot(ctx, ID, repoSn.GetId())
		if err != nil {
			return nil, err
		}
		sn.RuntimeAttrs = ra

		repoReq := &repopb.GetSnapshotInstanceSpecRequest{}
		repoReq.SnapshotId = repoSn.GetId()
		repoRsp, err := f.nannyRPCClient.GetSnapshotInstanceSpec(ctx, repoReq)
		if err != nil {
			return nil, err
		}
		sn.InstanceSpec = instancespec.InstanceSpecFactory(repoRsp.GetInstanceSpec())

		intReq := &internalpb.GetIssConfTemplateRequest{}
		intReq.SnapshotId = repoSn.GetId()
		intRsp, err := f.nannyRPCClient.GetIssConfTemplate(ctx, intReq)
		if err != nil {
			return nil, err
		}
		sn.ISSTemplate = intRsp.GetTemplate()
		snapshots = append(snapshots, sn)
	}
	rv.Snapshots = snapshots
	return rv, nil
}
