package cache

import (
	listers "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/listers/allocationctl/v1alpha1"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/ypclient"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/yputil"
	"context"
	"fmt"
	"k8s.io/apimachinery/pkg/labels"
)

// Lister is any object that knows how to perform list.
type Lister interface {
	// List should return a list type object i.e. object with Items field.
	List(opts api.ListOptions) ([]interface{}, error)
}

// ListFunc knows how to list resources
type ListFunc func(opts api.ListOptions) ([]interface{}, error)

// ListHelper knows how to list resources. It satisfies the Lister interface.
// It is a convenience function for users of NewReflector, etc.
// ListFunc must not be nil
type ListHelper struct {
	ListFunc ListFunc
}

func (lh *ListHelper) List(opts api.ListOptions) ([]interface{}, error) {
	return lh.ListFunc(opts)
}

func NewPodListerFilteredByAllocIDs(ctx context.Context, c ypclient.PodsGetter, allocsLister listers.AllocationLister) *ListHelper {
	listFunc := func(opts api.ListOptions) ([]interface{}, error) {
		if opts.Filter == "" {
			allocs, err := allocsLister.List(labels.Everything())
			if err != nil {
				return nil, fmt.Errorf("failed to list allocations for pods filtering: %w", err)
			}
			psIDs := make([]string, 0, len(allocs))
			for _, alloc := range allocs {
				psIDs = append(psIDs, alloc.Name)
			}
			opts.Filter = yputil.MakePodSetIDsFilter(psIDs)
		}
		pods, err := c.Pods().List(ctx, opts)
		if err != nil {
			return []interface{}{}, err
		}
		list := make([]interface{}, 0, len(pods))
		for _, p := range pods {
			list = append(list, p)
		}
		return list, nil
	}
	return &ListHelper{ListFunc: listFunc}
}
