package cache

import (
	"context"
	"fmt"
	"sync"
	"time"

	"k8s.io/apimachinery/pkg/util/wait"
	k8scache "k8s.io/client-go/tools/cache"
	"k8s.io/utils/clock"

	"a.yandex-team.ru/infra/allocation-ctl/pkg/log"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
)

// Reflector repeatedly syncs the specified resource up to the given store.
type Reflector struct {
	lock sync.RWMutex
	// synced is true if the store has been synced at least once.
	synced bool
	// name identifies this reflector.
	name string
	// The name of the type we expect to place in the store.
	expectedTypeName string
	// The destination to sync up with the upstream source.
	store k8scache.Store
	// lister is used to perform lists.
	lister       Lister
	listOpts     api.ListOptions
	resyncPeriod time.Duration
	// backoff manages backoff of ListAndSync
	backoffManager wait.BackoffManager
}

func NewReflector(name string, expectedTypeName string, store k8scache.Store, lister Lister, listOpts api.ListOptions, resyncPeriod time.Duration) *Reflector {
	realClock := &clock.RealClock{}
	return &Reflector{
		name:             name,
		expectedTypeName: expectedTypeName,
		store:            store,
		lister:           lister,
		listOpts:         listOpts,
		resyncPeriod:     resyncPeriod,
		backoffManager:   wait.NewExponentialBackoffManager(800*time.Millisecond, 30*time.Second, 2*time.Minute, 2.0, 1.0, realClock),
	}
}

// ListAndSync repeatedly lists all items and sync them up to store,
func (r *Reflector) ListAndSync(ctx context.Context) error {
	for {
		listCh := make(chan struct{}, 1)
		panicCh := make(chan interface{}, 1)
		var list []interface{}
		var err error
		go func() {
			defer func() {
				if r := recover(); r != nil {
					panicCh <- r
				}
			}()
			list, err = r.lister.List(r.listOpts)
			close(listCh)
		}()
		select {
		case <-ctx.Done():
			return nil
		case r := <-panicCh:
			panic(r)
		case <-listCh:
		}
		if err != nil {
			return fmt.Errorf("failed to list %v: %v", r.expectedTypeName, err)
		}
		log.Infof("%s: objects %s listed: %d", r.name, r.expectedTypeName, len(list))
		if err := r.syncWith(list); err != nil {
			return fmt.Errorf("unable to sync list result: %v", err)
		}
		// Sleep for resyncPeriod
		log.Infof("%s: sleeping for %s", r.name, r.resyncPeriod)
		t := time.NewTimer(r.resyncPeriod)
		select {
		case <-ctx.Done():
			return nil
		case r := <-panicCh:
			panic(r)
		case <-t.C:
		}
	}
}

func (r *Reflector) HasSynced() bool {
	r.lock.Lock()
	defer r.lock.Unlock()
	return r.synced
}

// Run repeatedly uses the reflector's ListAndSync to fetch all the
// objects and sync them up to store.
// Run will exit when stopCh is closed.
func (r *Reflector) Run(ctx context.Context) {
	log.Infof("%s: starting reflector %s", r.name, r.expectedTypeName)
	wait.BackoffUntil(func() {
		if err := r.ListAndSync(ctx); err != nil {
			log.Errorf("%s: ListAndSync failed: %v: %v", r.name, r.expectedTypeName, err)
		}
	}, r.backoffManager, true, ctx.Done())
	log.Infof("%s: stopping reflector %s", r.name, r.expectedTypeName)
}

// syncWith replaces the store's items with the given list.
func (r *Reflector) syncWith(items []interface{}) error {
	found := make([]interface{}, 0, len(items))
	found = append(found, items...)
	r.lock.Lock()
	defer r.lock.Unlock()
	r.synced = true
	return r.store.Replace(found, "")
}
