package yp

import (
	listers "a.yandex-team.ru/infra/allocation-ctl/pkg/generated/listers/allocationctl/v1alpha1"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/cache"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/ypclient"
	"context"
	"fmt"
	k8scache "k8s.io/client-go/tools/cache"
	"time"
)

type ClusterInfo struct {
	Name            string `yaml:"name"`
	DC              string `yaml:"dc"`
	Location        string `yaml:"location"`
	HostnameSegment string `yaml:"hostname_segment"`
	HostnameSuffix  string `yaml:"hostname_suffix"`
}

type Config struct {
	Clusters       []*ClusterInfo `yaml:"clusters"`
	SleepAfterSync time.Duration  `yaml:"sleep_after_sync"`
}

type ypCluster struct {
	Info         *ClusterInfo
	Client       *ypclient.YPClient
	PodIndexer   k8scache.Indexer
	podReflector *cache.Reflector
}

func newYpCluster(ctx context.Context, info *ClusterInfo, sleepAfterSync time.Duration, useCAS bool, allocsLister listers.AllocationLister) (*ypCluster, error) {
	client, err := ypclient.NewYPClientFromCluster(info.Name)
	if err != nil {
		return nil, fmt.Errorf("error creating YP client for cluster %s: %w", info.Name, err)
	}
	l := cache.NewPodListerFilteredByAllocIDs(ctx, client, allocsLister)
	indxr := k8scache.NewIndexer(
		func(obj interface{}) (string, error) {
			return obj.(*api.Pod).Meta.Id, nil
		},
		k8scache.Indexers{"allocID": indexByAllocID})
	opts := api.GetDefaultListOptions(useCAS)
	if useCAS && opts.SpecIdx == -1 {
		return nil, fmt.Errorf("wrong list options for pods: %s is not found in selectors but useCAS is true", api.SpecSelector)
	}
	r := cache.NewReflector(fmt.Sprintf("pod-reflector[%s]", info.Name), "Pod", indxr, l, opts, sleepAfterSync*time.Second)
	rv := &ypCluster{
		Info:         info,
		Client:       client,
		PodIndexer:   indxr,
		podReflector: r,
	}
	return rv, nil
}

func (c *ypCluster) Run(ctx context.Context) {
	go c.podReflector.Run(ctx)
}

func (c *ypCluster) HasSynced() bool {
	return c.podReflector.HasSynced()
}

type YpClusterMap map[string]*ypCluster

func NewYpClusterMap(ctx context.Context, cfg *Config, useCAS bool, allocsLister listers.AllocationLister) (YpClusterMap, error) {
	rv := make(YpClusterMap)
	for _, info := range cfg.Clusters {
		cluster, err := newYpCluster(ctx, info, cfg.SleepAfterSync, useCAS, allocsLister)
		if err != nil {
			return nil, fmt.Errorf("error creating YP cluster %s: %w", info.Name, err)
		}
		rv[info.Name] = cluster
	}
	return rv, nil
}

func (m *YpClusterMap) Run(ctx context.Context) {
	for _, c := range *m {
		c.Run(ctx)
	}
}

func (m *YpClusterMap) HasSynced() bool {
	for _, c := range *m {
		if !c.HasSynced() {
			return false
		}
	}
	return true
}

func indexByAllocID(obj interface{}) ([]string, error) {
	pod, ok := obj.(*api.Pod)
	if !ok {
		return []string{}, nil
	}
	return []string{pod.GetMeta().PodSetId}, nil
}
