package rpc

import (
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
	"a.yandex-team.ru/yp/go/yp"
	"context"
	"fmt"
)

type GetObjectsResponseInterface interface {
	Count() int
	Next() bool
	Fill(dest ...interface{}) error
	Timestamps() ([]uint64, error)
}

type SelectObjectsResponseInterface interface {
	ContinuationToken() string
	Count() int
	Next() bool
	Fill(dest ...interface{}) error
}

type RPCInterface interface {
	GetPods(ctx context.Context, req yp.GetPodsRequest) (GetObjectsResponseInterface, error)
	SelectPods(ctx context.Context, req yp.SelectPodsRequest) (SelectObjectsResponseInterface, error)
	UpdateObjects(ctx context.Context, req yp.UpdateObjectsRequest) error
}

// implements GetObjectsResponseInterface
type GetObjectsResponse struct {
	rsp *yp.GetObjectsResponse
}

func (r *GetObjectsResponse) Count() int {
	return r.rsp.Count()
}

func (r *GetObjectsResponse) Next() bool {
	return r.rsp.Next()
}

func (r *GetObjectsResponse) Fill(dest ...interface{}) error {
	return r.rsp.Fill(dest...)
}

func (r *GetObjectsResponse) Timestamps() ([]uint64, error) {
	return r.rsp.Timestamps()
}

// implements SelectObjectsResponseInterface
type SelectObjectsResponse struct {
	rsp *yp.SelectObjectsResponse
}

func (r *SelectObjectsResponse) ContinuationToken() string {
	return r.rsp.ContinuationToken()
}

func (r *SelectObjectsResponse) Count() int {
	return r.rsp.Count()
}

func (r *SelectObjectsResponse) Next() bool {
	return r.rsp.Next()
}

func (r *SelectObjectsResponse) Fill(dest ...interface{}) error {
	return r.rsp.Fill(dest...)
}

type ypMetrics struct {
	requests *unistat.Numeric
	errors   *unistat.Numeric
}

func newYPMetrics(cluster string) *ypMetrics {
	rv := &ypMetrics{
		requests: unistat.NewNumeric(fmt.Sprintf("yp-total-requests-%s", cluster), 0, aggr.Counter(), unistat.Sum),
		errors:   unistat.NewNumeric(fmt.Sprintf("yp-total-errors-%s", cluster), 0, aggr.Counter(), unistat.Sum),
	}
	unistat.Register(rv.requests)
	unistat.Register(rv.errors)
	return rv
}

// implements RPCInterface
type Client struct {
	base    *yp.Client
	metrics *ypMetrics
}

func NewClient(base *yp.Client, cluster string) *Client {
	return &Client{base: base, metrics: newYPMetrics(cluster)}
}

func (c *Client) GetPods(ctx context.Context, req yp.GetPodsRequest) (GetObjectsResponseInterface, error) {
	if c.metrics != nil {
		defer c.metrics.requests.Update(1)
	}
	rsp, err := c.base.GetPods(ctx, req)
	if err != nil {
		if c.metrics != nil {
			defer c.metrics.errors.Update(1)
		}
		return nil, err
	}
	return &GetObjectsResponse{rsp: rsp}, nil
}

func (c *Client) SelectPods(ctx context.Context, req yp.SelectPodsRequest) (SelectObjectsResponseInterface, error) {
	if c.metrics != nil {
		defer c.metrics.requests.Update(1)
	}
	rsp, err := c.base.SelectPods(ctx, req)
	if err != nil {
		if c.metrics != nil {
			defer c.metrics.errors.Update(1)
		}
		return nil, err
	}
	return &SelectObjectsResponse{rsp: rsp}, nil
}

func (c *Client) UpdateObjects(ctx context.Context, req yp.UpdateObjectsRequest) error {
	if c.metrics != nil {
		defer c.metrics.requests.Update(1)
	}
	if _, err := c.base.UpdateObjects(ctx, req); err != nil {
		if c.metrics != nil {
			defer c.metrics.errors.Update(1)
		}
		return err
	}
	return nil
}
