package ypclient

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/rpc"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
	"context"
	"fmt"
	"github.com/stretchr/testify/assert"
	"sort"
	"testing"
)

type timestampedPod struct {
	pod        *ypapi.TPod
	timestamps []uint64
}

// implements GetObjectsResponseInterface
type fakeGetPodsResponse struct {
	pods    []*timestampedPod
	current int
	last    int
	parsed  bool
}

func (r *fakeGetPodsResponse) Count() int {
	return len(r.pods)
}

func (r *fakeGetPodsResponse) Next() bool {
	if !r.parsed {
		r.parsed = true
		r.last = len(r.pods) - 1
		r.current = -1
	}

	if r.current >= r.last {
		return false
	}

	r.current++
	return true
}

func (r *fakeGetPodsResponse) Fill(dest ...interface{}) error {
	meta := dest[0].(*ypapi.TPodMeta)
	meta.Id = r.pods[r.current].pod.Meta.Id
	return nil
}

func (r *fakeGetPodsResponse) Timestamps() ([]uint64, error) {
	return r.pods[r.current].timestamps, nil
}

// implements SelectObjectsResponseInterface
type fakeSelectPodIDsResponse struct {
	ids               []string
	current           int
	last              int
	parsed            bool
	continuationToken string
}

func (r *fakeSelectPodIDsResponse) ContinuationToken() string {
	return r.continuationToken
}

func (r *fakeSelectPodIDsResponse) Count() int {
	return len(r.ids)
}

func (r *fakeSelectPodIDsResponse) Next() bool {
	if !r.parsed {
		r.parsed = true
		r.last = len(r.ids) - 1
		r.current = -1
	}

	if r.current >= r.last {
		return false
	}

	r.current++
	return true
}

func (r *fakeSelectPodIDsResponse) Fill(dest ...interface{}) error {
	meta := dest[0].(*ypapi.TPodMeta)
	meta.Id = r.ids[r.current]
	return nil
}

// implements RPCInterface
type fakeRPC struct {
	getPodsReqs   []*yp.GetPodsRequest
	getPodsRsps   []*fakeGetPodsResponse
	getPodsErrIdx int

	selectPodsReqs   []*yp.SelectPodsRequest
	selectPodsRsps   []*fakeSelectPodIDsResponse
	selectPodsErrIdx int
}

func newFakeRPC(getPodsRsps []*fakeGetPodsResponse, getPodsErrIdx int, selectPodsRsps []*fakeSelectPodIDsResponse, selectPodsErrIdx int) *fakeRPC {
	return &fakeRPC{
		getPodsReqs:      make([]*yp.GetPodsRequest, 0, len(getPodsRsps)),
		getPodsRsps:      getPodsRsps,
		getPodsErrIdx:    getPodsErrIdx,
		selectPodsReqs:   make([]*yp.SelectPodsRequest, 0, len(selectPodsRsps)),
		selectPodsRsps:   selectPodsRsps,
		selectPodsErrIdx: selectPodsErrIdx,
	}
}

func (c *fakeRPC) GetPods(ctx context.Context, req yp.GetPodsRequest) (rpc.GetObjectsResponseInterface, error) {
	idx := len(c.getPodsReqs)
	if idx == c.getPodsErrIdx {
		return nil, fmt.Errorf("dummy error on GetPods method")
	}
	c.getPodsReqs = append(c.getPodsReqs, &req)
	return c.getPodsRsps[idx], nil
}

func (c *fakeRPC) SelectPods(ctx context.Context, req yp.SelectPodsRequest) (rpc.SelectObjectsResponseInterface, error) {
	idx := len(c.selectPodsReqs)
	if idx == c.selectPodsErrIdx {
		return nil, fmt.Errorf("dummy error on SelectPodIDs method")
	}
	c.selectPodsReqs = append(c.selectPodsReqs, &req)
	return c.selectPodsRsps[idx], nil
}

func (c *fakeRPC) UpdateObjects(ctx context.Context, req yp.UpdateObjectsRequest) error {
	return nil
}

func TestList(t *testing.T) {
	t.Parallel()
	selectPodsSels := []string{"/meta"}
	getPodsSels := []string{"/meta", "/spec", "/status"}
	selRsps := []*fakeSelectPodIDsResponse{
		{
			ids:               []string{"pod-1", "pod-2", "pod-3", "pod-4", "pod-5"},
			continuationToken: "token-1",
		},
		{
			ids:               []string{"pod-6", "pod-7"},
			continuationToken: "token-2",
		},
	}

	emptySpec := &ypapi.TPodSpec{}
	emptyStatus := &ypapi.TPodStatus{}

	getRsps := []*fakeGetPodsResponse{
		{
			pods: []*timestampedPod{
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-1"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{11, 12, 13},
				},
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-2"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{21, 22, 23},
				},
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-3"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{31, 32, 33},
				},
			},
		},
		{
			pods: []*timestampedPod{
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-4"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{41, 42, 43},
				},
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-5"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{51, 52, 53},
				},
			},
		},
		{
			pods: []*timestampedPod{
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-6"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{61, 62, 63},
				},
				{
					pod: &ypapi.TPod{
						Meta:   &ypapi.TPodMeta{Id: "pod-7"},
						Spec:   emptySpec,
						Status: emptyStatus,
					},
					timestamps: []uint64{71, 72, 73},
				},
			},
		},
	}

	expectedGetReqs := []*yp.GetPodsRequest{
		{
			IDs:             []string{"pod-1", "pod-2", "pod-3"},
			Selectors:       getPodsSels,
			FetchTimestamps: true,
			Format:          yp.PayloadFormatProto,
		},
		{
			IDs:             []string{"pod-4", "pod-5"},
			Selectors:       getPodsSels,
			FetchTimestamps: true,
			Format:          yp.PayloadFormatProto,
		},
		{
			IDs:             []string{"pod-6", "pod-7"},
			Selectors:       getPodsSels,
			FetchTimestamps: true,
			Format:          yp.PayloadFormatProto,
		},
	}
	expectedSelReqs := []*yp.SelectPodsRequest{
		{
			Format:    yp.PayloadFormatProto,
			Filter:    "",
			Selectors: selectPodsSels,
			Limit:     5,
		},
		{
			Format:            yp.PayloadFormatProto,
			Filter:            "",
			Selectors:         selectPodsSels,
			Limit:             5,
			ContinuationToken: "token-1",
		},
	}
	expectedPods := []*api.Pod{
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-1"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 12,
		},
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-2"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 22,
		},
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-3"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 32,
		},
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-4"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 42,
		},
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-5"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 52,
		},
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-6"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 62,
		},
		{
			TPod: ypapi.TPod{
				Meta:   &ypapi.TPodMeta{Id: "pod-7"},
				Spec:   emptySpec,
				Status: emptyStatus,
			},
			SpecTimestamp: 72,
		},
	}

	rpcClient := newFakeRPC(getRsps, -1, selRsps, -1)
	podsCli := &podsClient{
		client: rpcClient,
	}

	opts := api.ListOptions{
		Filter:                 "",
		Selectors:              getPodsSels,
		GetObjectsBatchSize:    3,
		SelectIDsBatchSize:     5,
		GetObjectsThreadsCount: 2,
		FetchTimestamps:        true,
		SpecIdx:                1,
	}

	// Case 1: no errors
	ctx := context.Background()
	pods, err := podsCli.List(ctx, opts)
	assert.NoError(t, err)
	assert.Equal(t, expectedSelReqs, rpcClient.selectPodsReqs)
	sort.Slice(rpcClient.getPodsReqs, func(i, j int) bool {
		return rpcClient.getPodsReqs[i].IDs[0] < rpcClient.getPodsReqs[j].IDs[0]
	})
	assert.Equal(t, expectedGetReqs, rpcClient.getPodsReqs)
	sort.Slice(pods, func(i, j int) bool {
		return pods[i].GetMeta().GetId() < pods[j].GetMeta().GetId()
	})
	assert.Equal(t, expectedPods, pods)

	for _, r := range getRsps {
		r.parsed = false
	}
	for _, r := range selRsps {
		r.parsed = false
	}
	// Case 2: error in producer
	ctx = context.Background()
	rpcClient = newFakeRPC(getRsps, -1, selRsps, 1)
	podsCli = &podsClient{
		client: rpcClient,
	}
	_, err = podsCli.List(ctx, opts)
	assert.Errorf(t, err, "dummy error on SelectPodIDs method")
	assert.Equal(t, len(rpcClient.selectPodsReqs), 1)
	assert.LessOrEqual(t, len(rpcClient.getPodsReqs), 2)

	for _, r := range getRsps {
		r.parsed = false
	}
	for _, r := range selRsps {
		r.parsed = false
	}

	// Case 3: error in one of workers
	ctx = context.Background()
	rpcClient = newFakeRPC(getRsps, 1, selRsps, -1)
	podsCli = &podsClient{
		client: rpcClient,
	}
	_, err = podsCli.List(ctx, opts)
	assert.Errorf(t, err, "dummy error on GetPods method")
}
