package yputil

import (
	"a.yandex-team.ru/infra/allocation-ctl/pkg/yp/api"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
	"a.yandex-team.ru/yt/go/yson"
	"fmt"
	"strings"
)

func findLabelYSON(labels *ytree.TAttributeDictionary, key string) ([]byte, bool) {
	for _, a := range labels.GetAttributes() {
		if a.GetKey() == key {
			return a.GetValue(), true
		}
	}
	return nil, false
}

func FindLabelString(labels *ytree.TAttributeDictionary, key string) (string, error) {
	var s string
	v, ok := findLabelYSON(labels, key)
	if !ok {
		return s, fmt.Errorf("key %s is not found", key)
	}
	if err := yson.Unmarshal(v, &s); err != nil {
		return s, fmt.Errorf("failed to unmarshal value for key %s: %w", key, err)
	}
	return s, nil
}

func FindLabelUint64(labels *ytree.TAttributeDictionary, key string) (uint64, error) {
	var i uint64
	v, ok := findLabelYSON(labels, key)
	if !ok {
		return i, fmt.Errorf("key %s is not found", key)
	}
	if err := yson.Unmarshal(v, &i); err != nil {
		return i, fmt.Errorf("failed to unmarshal value for key %s: %w", key, err)
	}
	return i, nil
}

func HasPodISSInstances(pod *api.Pod) bool {
	return len(pod.Spec.Iss.Instances) != 0
}

func MakePodPersistentHostname(podID string, cluster string, hostnameSuffix string) string {
	if hostnameSuffix != "" {
		return fmt.Sprintf("%s.%s", podID, hostnameSuffix)
	}
	return fmt.Sprintf("%s.%s.yp-c.yandex.net", podID, cluster)
}

func GetPodBackboneIP(pod *api.Pod) (string, bool) {
	for _, a := range pod.GetStatus().GetIp6AddressAllocations() {
		if a.GetVlanId() == "backbone" {
			return a.GetAddress(), true
		}
	}
	return "", false
}

func MakePodToUpdateISSSpec(pod *api.Pod, issSpec *clusterapi.HostConfiguration) *api.Pod {
	return &api.Pod{
		TPod: ypapi.TPod{
			Meta: &ypapi.TPodMeta{Id: pod.GetMeta().GetId()},
			Spec: &ypapi.TPodSpec{Iss: issSpec},
		},
		SpecTimestamp: pod.SpecTimestamp,
	}
}

func MakePodSetIDsFilter(psIDs []string) string {
	if len(psIDs) == 0 {
		return ""
	}
	ids := strings.Join(psIDs, "\",\"")
	s := fmt.Sprintf("[/meta/pod_set_id] IN (\"%s\")", ids)
	return s
}

func SetLabelString(labels *ytree.TAttributeDictionary, k string, v string) error {
	ysonV, err := yson.Marshal(v)
	if err != nil {
		return fmt.Errorf("failed marshaling to YSON value %s for key %s", v, k)
	}
	labels.Attributes = append(labels.GetAttributes(), &ytree.TAttribute{Key: &k, Value: ysonV})
	return nil
}

func SetLabelUint64(labels *ytree.TAttributeDictionary, k string, v uint64) error {
	ysonV, err := yson.Marshal(v)
	if err != nil {
		return fmt.Errorf("failed marshaling to YSON value %d for key %s", v, k)
	}
	labels.Attributes = append(labels.GetAttributes(), &ytree.TAttribute{Key: &k, Value: ysonV})
	return nil
}
