import datetime
import logging
import requests
import json
import os
import argparse

import yt.wrapper
from copy import copy
import infra.analytics.io_limits_pipeline.utils as utils


signals = "portoinst-io_read_bytes_/ssd_tmmv||portoinst-io_read_bytes_/place_tmmv||portoinst-io_ops_/place_tmmv||" \
          "portoinst-io_ops_/ssd_tmmv||portoinst-cpu_usage_cores_tmmv||portoinst-io_write_fs_bytes_tmmv||" \
          "portoinst-io_serviced_/place_read_hgram"


def dummy_mapper(input_row):

    for i in range(10):

        output_dict = copy(input_row)
        output_dict["pod_name"] = input_row["pod_name"] + str(i)
        yield output_dict


def generate_time_intervals_week():
    intervals = []

    end = int(datetime.datetime.now().timestamp()) - 60*5

    for i in range(7):
        start = end - 86400
        intervals.append((start, end))
        end = start - 1

    return intervals


def construct_rows_from_signal(pod):

    base_url = "https://solomon.yandex-team.ru/api/v2/projects/yasm_{0}/sensors/data"
    expression_template = '''signal="{0}", host="{1}", cluster="host_*", nanny="{2}", ctype="{3}"'''

    solomon_token = os.environ.get("YT_SECURE_VAULT_SOLOMON_TOKEN")

    hdrs = {
        "Authorization": "OAuth {}".format(solomon_token),
        "Content-Type": "application/json;charset=UTF-8",
        "Accept": "application/json"
    }

    fmt = "%Y-%m-%dT%H:%M:%S.%fZ"

    intervals = generate_time_intervals_week()

    for tmstmp_st, tmstmp_end in intervals:

        start, end = (datetime.datetime.fromtimestamp(tmstmp_st) - datetime.timedelta(hours=3)).strftime(fmt), \
                     datetime.datetime.fromtimestamp(tmstmp_end).strftime(fmt)

        try:
            parsed_tags = parse_yasm_tags(pod["yasm_tag"])

            pod_url = pod["pod_name"] + "." + pod["dc"] + ".yp-c.yandex.net"

            try:
                itype, ctype, nanny = parsed_tags.get("itype"), parsed_tags.get("ctype"), parsed_tags.get("nanny",
                                                                                                          "")
            except AttributeError:
                logging.info("Epty tags detected on pod {0} for service {1}".format(pod_url, pod["service_name"]))
                itype, ctype, nanny = None, None, None

            if itype and ctype:

                if nanny == "":
                    nanny = pod["service_name"]

                expr = "{" + expression_template.format(signals, pod_url, nanny, ctype) + "}"

                payload = {
                    "program": expr,
                    "from": start,
                    "to": end
                }
                try:
                    reslt = requests.post(base_url.format(itype), data=json.dumps(payload), headers=hdrs).json()
                except:
                    logging.info("No data extracted from solomon on pod {0} for service {1}".format(
                        pod_url, pod["service_name"]))
                    reslt = {}

                try:

                    vector = reslt['vector']
                    if not nanny:
                        nanny = pod["service_name"]
                    for rst in parse_vector(vector, pod["host"], pod_url, nanny):
                        yield rst

                except:
                    logging.info("Error while parsing vector on pod {0} for service {1}".format(pod_url,
                                                                                                pod["service_name"]))

        except:
            logging.info("Error while parsing pod for service {0}".format(pod["service_name"]))


def parse_yasm_tags(tagstr):
    try:
        splitted = {i.split("=")[0]: i.split("=")[1] for i in tagstr.split(";")}
        return splitted
    except IndexError:
        return None


def parse_vector(vector, host, pod, nanny):

    final_result = []

    if len(vector) > 0 and vector[0].get("timeseries"):

        try:
            for tmsmpnum, tmstmp in enumerate(vector[0]["timeseries"]["timestamps"]):
                obs = {}
                obs["timestamp"] = int(tmstmp / 1000)

                for signum in range(len(vector)):

                    # проверим, является ли сигнал гистограмный вот таким странным образом
                    try:
                        if 'buckets' not in vector[signum]["timeseries"]["values"][tmsmpnum]:
                            try:
                                obs[vector[signum]["timeseries"]["labels"]["signal"]] = 0 if \
                                    vector[signum]["timeseries"] \
                                ["values"][tmsmpnum]["sum"] is None else \
                                    vector[signum]["timeseries"]["values"][tmsmpnum]["sum"]
                            except IndexError:
                                pass
                        else:
                            if len(vector[signum]["timeseries"]["values"][tmsmpnum]["buckets"]) == 0:
                                obs[vector[signum]['timeseries']['labels']['signal']] = float(0)
                            else:
                                if len(vector[signum]["timeseries"]["values"][tmsmpnum]["buckets"]) == 1 and \
                                        vector[signum]["timeseries"]["values"][tmsmpnum]["buckets"][0] >= 1:

                                    power = vector[signum]["timeseries"]["values"][tmsmpnum]["startPower"] + 1
                                    base = vector[signum]["timeseries"]["values"][tmsmpnum]["base"]

                                    obs[vector[signum]["timeseries"]["labels"]["signal"]] = float(round(base**power, 0))
                    except IndexError:
                        pass

                obs["host"] = host
                obs["pod"] = pod
                obs["nannyservice"] = nanny

                final_result.append(obs)
        except KeyError:
            logging.info("Error while parsing vector for service {0}".format(nanny))

    else:
        pass

    return final_result


if __name__ == '__main__':
    yt.wrapper.config.set_proxy("hahn")
    parser = argparse.ArgumentParser()
    parser.add_argument("--local_run", type=utils.str2bool, default=True)
    parser.add_argument("--source_table", type=str, default=utils.META_DATA)
    parser.add_argument("--output_table", type=str, default=utils.AUTOMATED_GENERATED_RAW_DATA)
    args = parser.parse_args()

    if args.local_run is True:

        utils.read_solomon_token()

        yt.wrapper.run_map(
            construct_rows_from_signal,
            source_table=args.source_table,
            destination_table=args.output_table,
            spec={"secure_vault": {"SOLOMON_TOKEN": utils.get_solomon_token()}}
        )
    else:
        yt.wrapper.run_map(
            construct_rows_from_signal,
            source_table=args.source_table,
            destination_table=args.output_table,
            spec={"secure_vault": {"SOLOMON_TOKEN": os.environ.get("SOLOMON_TOKEN")}}
        )
