import os
import shlex
import base64
import typing
import argparse

import yaml

import library.python.oauth as lpo
import library.python.vault_client as vault_client
from yp.client import YpClient


VAULT_CLIENT = None
VAULT_CACHE = {}

YP_TOKEN = None
YP_TOKEN_ENV = 'DCTL_YP_TOKEN'
YP_CLIENT = None

try:
    YP_TOKEN_PATH = os.path.expanduser('~/.dctl/token')
except Exception:
    YP_TOKEN_PATH = None

DCTL_CLIENT_ID = '688c92864835447b9ed74221f44420b7'
DCTL_CLIENT_SECRET = '17bb18beacef42868f023d93d475e127'

VAULT_HOST = 'https://vault-api.passport.yandex.net'


def get_config(path: str) -> dict:
    with open(path) as f:
        data = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))

    assert isinstance(data, dict)
    data.setdefault('cluster', 'xdc')
    data.setdefault('custom_env', {})
    return data


def get_token() -> str:
    global YP_TOKEN

    if YP_TOKEN is not None:
        return YP_TOKEN

    token = os.getenv(YP_TOKEN_ENV)
    if token is not None:
        YP_TOKEN = token
        return token

    if YP_TOKEN_PATH is not None and os.path.exists(YP_TOKEN_PATH):
        with open(YP_TOKEN_PATH) as f:
            token = f.read().strip()
            if token:
                YP_TOKEN = token
                return token

    token = lpo.get_token(DCTL_CLIENT_ID, DCTL_CLIENT_SECRET, raise_errors=True)
    YP_TOKEN = token
    return token


def get_vault_client() -> vault_client.VaultClient:
    global VAULT_CLIENT

    if VAULT_CLIENT is not None:
        return VAULT_CLIENT

    VAULT_CLIENT = vault_client.VaultClient(
        host='https://vault-api.passport.yandex.net',
        check_status=False,
        authorization=get_token(),
        decode_files=True,
    )
    return VAULT_CLIENT


def get_yp_client(cluster: str) -> YpClient:
    global YP_CLIENT

    if YP_CLIENT is not None:
        return YP_CLIENT

    YP_CLIENT = YpClient(
        address=f'{cluster}.yp.yandex.net:8090',
        config={
            'token': get_token(),
        },
    )
    return YP_CLIENT


def resolve_secret(version: str, field: str) -> str:
    if version not in VAULT_CACHE:
        VAULT_CACHE[version] = get_vault_client().get_version(version)['value']

    return VAULT_CACHE[version][field]


def resolv_env(env_value: dict, secrets: dict) -> typing.Tuple[str, str]:
    if 'literal_env' in env_value['value']:
        return env_value['name'], env_value['value']['literal_env']['value']
    else:
        secret_info = env_value['value']['secret_env']
        secret_version = secrets[secret_info['alias']]['secret_version']
        secret_data = resolve_secret(secret_version, secret_info['id'])
        if secret_info.get('decode_base64', False):
            secret_data = base64.b64decode(secret_data).decode()

        return env_value['name'], secret_data


def make_env_params(cfg: dict) -> typing.Iterator[typing.Tuple[str, str]]:
    deploy_unit = get_yp_client(cfg['cluster']).get_object(
        object_type='stage',
        object_identity=cfg['stage'],
        selectors=[f'/spec/deploy_units/{cfg["deploy_unit"]}'],
    )[0]
    if 'replica_set' in deploy_unit:
        pod = deploy_unit['replica_set']['replica_set_template']['pod_template_spec']['spec']
    else:
        pod = deploy_unit['multi_cluster_replica_set']['replica_set']['pod_template_spec']['spec']

    box = next(iter(filter(
        lambda b: b['id'] == cfg['box'],
        pod['pod_agent_payload']['spec']['boxes'],
    )))
    workload = next(iter(filter(
        lambda w: w['id'] == cfg['workload'] and w['box_ref'] == cfg['box'],
        pod['pod_agent_payload']['spec']['workloads'],
    ))) if 'workload' in cfg else None
    secrets = pod.get('secrets', {})
    secrets.update(pod.get('secret_refs', {}))

    for env in box.get('env', []):
        yield resolv_env(env, secrets)

    if workload is not None:
        for env in workload.get('env', []):
            yield resolv_env(env, secrets)

    for env in cfg['custom_env'].items():
        yield env


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--env', default='env.yaml')
    args = parser.parse_args()
    cfg = get_config(args.env)
    for k, v in make_env_params(cfg):
        v = shlex.quote(str(v))
        print(f'export {k}={v};')
