package ru.yandex.infra.auth;

import java.io.File;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;

import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.eclipse.jetty.server.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.idm.api.IdmApi;
import ru.yandex.infra.auth.idm.api.IdmApiImpl;
import ru.yandex.infra.auth.idm.api.IdmApiReadOnlyImpl;
import ru.yandex.infra.auth.idm.api.IdmApiService;
import ru.yandex.infra.auth.idm.service.IdmService;
import ru.yandex.infra.auth.nanny.NannyApi;
import ru.yandex.infra.auth.nanny.NannyApiImpl;
import ru.yandex.infra.auth.nanny.NannyRolesSynchronizer;
import ru.yandex.infra.auth.nanny.NannyService;
import ru.yandex.infra.auth.nanny.ReadOnlyNannyApiImpl;
import ru.yandex.infra.auth.servlets.AclServlet;
import ru.yandex.infra.auth.servlets.ApiGetProjectAclServlet;
import ru.yandex.infra.auth.servlets.GroupsGCServlet;
import ru.yandex.infra.auth.servlets.IdmAddRemoveMembershipServlet;
import ru.yandex.infra.auth.servlets.IdmAddRemoveRoleServlet;
import ru.yandex.infra.auth.servlets.IdmAuthorization;
import ru.yandex.infra.auth.servlets.IdmCertificateAutorization;
import ru.yandex.infra.auth.servlets.IdmGetMembershipsServlet;
import ru.yandex.infra.auth.servlets.IdmGetRolesServlet;
import ru.yandex.infra.auth.servlets.IdmInfoServlet;
import ru.yandex.infra.auth.servlets.IdmTvmAuthorization;
import ru.yandex.infra.auth.servlets.PrivateApiServlet;
import ru.yandex.infra.auth.staff.StaffApi;
import ru.yandex.infra.auth.staff.StaffApiImpl;
import ru.yandex.infra.auth.tasks.RoleSubjectCacheUpdater;
import ru.yandex.infra.auth.tasks.RolesUpdater;
import ru.yandex.infra.auth.tasks.TvmTicketUpdater;
import ru.yandex.infra.auth.tasks.Watchdog;
import ru.yandex.infra.auth.utils.ConfigUtils;
import ru.yandex.infra.auth.yp.YpClients;
import ru.yandex.infra.auth.yp.YpGroupsClient;
import ru.yandex.infra.auth.yp.YpGroupsClientImpl;
import ru.yandex.infra.auth.yp.YpObjectPermissionsUpdater;
import ru.yandex.infra.auth.yp.YpObjectPermissionsUpdaterImpl;
import ru.yandex.infra.auth.yp.YpObjectsTreeGetter;
import ru.yandex.infra.auth.yp.YpObjectsTreeGetterImpl;
import ru.yandex.infra.auth.yp.YpService;
import ru.yandex.infra.auth.yp.YpServiceImpl;
import ru.yandex.infra.controller.YtSettings;
import ru.yandex.infra.controller.concurrent.DummyLeaderService;
import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.dto.NannyServiceMeta;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.metrics.MetricAdapterRegistry;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.servlets.UnistatServlet;
import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.controller.util.YpUtils;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.YpObjectSettings;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.infra.controller.yp.YpTransactionClientImpl;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TNannyServiceSpec;
import ru.yandex.yp.client.api.TNannyServiceStatus;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.auth.servlets.PrivateApiServlet.PrivateOperation.CLEANUP_NODES;
import static ru.yandex.infra.auth.servlets.PrivateApiServlet.PrivateOperation.LIST_PROJECTS_WITHOUT_OWNER;
import static ru.yandex.infra.auth.servlets.PrivateApiServlet.PrivateOperation.LIST_ROLE_NODES;
import static ru.yandex.infra.auth.servlets.PrivateApiServlet.PrivateOperation.LIST_ROLE_SUBJECTS;
import static ru.yandex.infra.auth.servlets.PrivateApiServlet.PrivateOperation.REMOVE_ROLE_SUBJECT;
import static ru.yandex.infra.auth.servlets.PrivateApiServlet.PrivateOperation.UPDATE_ROLE_ACLS;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.SYSTEM_LABEL_KEY;
import static ru.yandex.infra.controller.metrics.MetricUtils.buildMetricRegistry;
import static ru.yandex.infra.controller.servlets.ServerBuilder.buildServer;
import static ru.yandex.infra.controller.util.ConfigUtils.token;

public final class Main {
    private static final Logger LOG = LoggerFactory.getLogger(Main.class);
    private static final String AUTHCTL_SERVICE_NAME = "authctl";
    private static final String METRIC_FAILED_LEADER_LOCK_ACQUIRE_ATTEMPTS = "failed_leader_lock_acquire_attempts";

    private static final AtomicLong metricFailedLeaderLockAcquireAttempts = new AtomicLong();

    private Main() {
    }

    public static void main(String[] args) {
        // To exit app if main thread throws
        try {
            doMain(args);
        } catch (Exception e) {
            LOG.error("Exception in main:", e);
            ExitUtils.gracefulExit(ExitUtils.EXCEPTION_MAIN);
        }
    }

    private static void doMain(String[] args) {
        LOG.info("------------------------------------------------------------------------------------------------------");
        LOG.info("Starting new instance of Auth Controller...");

        Config config = ConfigFactory.parseFile(new File(args[0]), ConfigParseOptions.defaults().setAllowMissing(false))
                .withFallback(ConfigFactory.load("application_defaults.conf"));
        Config rolesConfig = ConfigFactory.load("roles_to_permissions.conf");
        Config mainConfig = config.getConfig("main");
        boolean isReadonlyMode = (mainConfig.hasPath("dry_run") && mainConfig.getBoolean("dry_run")) || //for compatibility with old configs
                (config.hasPath("readonly_mode") && config.getBoolean("readonly_mode"));
        if (isReadonlyMode) {
            LOG.warn("!!! STARTING IN READONLY RUN MODE -> ONLY READ OPERATIONS, NO CHANGES TO YP/YT/IDM !!!");
        }

        Config metricsConfig = config.getConfig("metrics");
        MetricRegistry metricRegistry = buildMetricRegistry(metricsConfig);
        Metrics metrics = new Metrics(metricRegistry, mainConfig.getDuration("roles_update_rate").toMillis());

        YtSettings ytSettings = ru.yandex.infra.controller.util.ConfigUtils.ytSettings(config.getConfig("yt"));
        LeaderService leaderService = isReadonlyMode ?
                new DummyLeaderService(metricRegistry) :
                ru.yandex.infra.controller.util.ConfigUtils.leaderService(AUTHCTL_SERVICE_NAME,
                        config.getConfig("leader_lock"), metricRegistry, ytSettings);

        GaugeRegistry gaugeRegistry = new MetricAdapterRegistry(metricRegistry);

        GaugeRegistry authCtlMetricsRegistry = new NamespacedGaugeRegistry(gaugeRegistry, AUTHCTL_SERVICE_NAME);
        authCtlMetricsRegistry.add(METRIC_FAILED_LEADER_LOCK_ACQUIRE_ATTEMPTS, new GolovanableGauge<>(metricFailedLeaderLockAcquireAttempts::get, "dmmm"));

        RolesInfo rolesInfo = RolesInfo.fromConfig(rolesConfig);
        YpService ypService = createYpService(config, rolesInfo, rolesConfig, isReadonlyMode, metricRegistry, metrics, gaugeRegistry);

        StaffApi staffApi = createStaffApiService(config, gaugeRegistry);
        IdmApiService idmApiService = createIdmApiService(config, rolesInfo, isReadonlyMode, metrics, staffApi, gaugeRegistry);

        Watchdog watchdog = new Watchdog(config.getConfig("watchdog"));

        IdmAuthorization idmAuthorizationStrategy;
        if (mainConfig.getBoolean("use_tvm")) {
            TvmTicketUpdater tvmTicketUpdater = createAndStartTvmTicketUpdater(config.getConfig("tvm"), watchdog);
            idmAuthorizationStrategy = new IdmTvmAuthorization(tvmTicketUpdater);
        } else {
            idmAuthorizationStrategy = new IdmCertificateAutorization(mainConfig.getString("valid_cns"));
        }

        GroupsAndUsersCacheImpl groupsAndUsersCache = new GroupsAndUsersCacheImpl(staffApi,
                ypService.getMasterClusterClients(),
                mainConfig.getDuration("groups_and_users_cache_rate"),
                mainConfig.getDuration("main_loop_timeout"),
                gaugeRegistry);
        groupsAndUsersCache.start();

        NannyService nannyService = createNannyService(config, gaugeRegistry, ypService, leaderService,
                groupsAndUsersCache, rolesInfo, isReadonlyMode);

        IdmService idmService = new IdmService(ypService, rolesInfo, nannyService, gaugeRegistry);
        RolesUpdater updater = createRolesUpdater(
                config, idmApiService, ypService, watchdog, leaderService, nannyService, gaugeRegistry
        );

        RoleSubjectCacheUpdater roleSubjectsProvider = new RoleSubjectCacheUpdater(
                ypService,
                mainConfig.getDuration("role_subject_cache_rate"),
                mainConfig.getDuration("main_loop_timeout"),
                gaugeRegistry
        );
        roleSubjectsProvider.start();

        createAndStartHttpServers(config, ypService, rolesInfo, metricRegistry,
                idmAuthorizationStrategy, idmService, updater, roleSubjectsProvider);

        while (!leaderService.isLeader()) {
            try {
                leaderService.ensureLeadership();
            } catch (Exception exception) {
                metricFailedLeaderLockAcquireAttempts.incrementAndGet();
                LOG.warn("Failed to acquire lock: {}", exception.getMessage());
            }
        }

        updater.start();
    }

    private static YpService createYpService(
            Config config,
            RolesInfo rolesInfo,
            Config rolesConfig,
            boolean dryRun,
            MetricRegistry metricRegistry,
            Metrics metrics,
            GaugeRegistry gaugeRegistry) {
        int selectPageSize = config.getInt("yp.select_request_page_size");
        int watchPageSize = config.getInt("yp.watch_request_page_size");

        YpRawObjectService ypMasterClient = YpUtils.ypRawClient(config.getConfig("yp.master"), config.getConfig("yp"), gaugeRegistry, dryRun);
        LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository =
                ConfigUtils.ypStageRepository(ypMasterClient, selectPageSize, watchPageSize,
                        config.getConfig("stage_repository.labels"), gaugeRegistry);
        LabelBasedRepository<NannyServiceMeta, TNannyServiceSpec, TNannyServiceStatus> nannyServiceRepository =
                ConfigUtils.ypNannyServiceRepository(ypMasterClient, selectPageSize, watchPageSize, gaugeRegistry);
        LabelBasedRepository<ProjectMeta, TProjectSpec, TProjectStatus> projectRepository =
                ConfigUtils.ypProjectRepository(ypMasterClient, selectPageSize, watchPageSize, gaugeRegistry);
        LabelBasedRepository<SchemaMeta, DataModel.TUserSpec, DataModel.TUserStatus> userRepository =
                ConfigUtils.ypUserRepository(ypMasterClient, selectPageSize, gaugeRegistry);
        LabelBasedRepository<SchemaMeta, DataModel.TGroupSpec, DataModel.TGroupStatus> groupsRepository =
                ConfigUtils.ypGroupRepository(ypMasterClient, config.getInt("yp.request_group_page_size"), gaugeRegistry);

        Map<YpObjectType, YpObjectSettings> ypObjectsCacheSettings = YpObjectSettings.loadFromConfig(config.getConfig("yp.settings_per_object_type"));

        var ypGroupsRepositoryForSystemDeploy = ConfigUtils.ypGroupRepository(ypMasterClient,
                config.getInt("yp.request_group_page_size"),
                gaugeRegistry,
                ImmutableMap.of(SYSTEM_LABEL_KEY, config.getString("main.system_name")));
        YpObjectsTreeGetter ypObjectsTreeGetter = new YpObjectsTreeGetterImpl(ypStageRepository,
                nannyServiceRepository,
                projectRepository,
                ypGroupsRepositoryForSystemDeploy,
                metrics,
                config.getString("main.use_uuid_since"),
                ypObjectsCacheSettings,
                gaugeRegistry,
                config.getBoolean("nanny.process_services_without_project"));

        YpTransactionClient ypTransactionMasterClient = new YpTransactionClientImpl(ypMasterClient);
        YpObjectPermissionsUpdater ypStageAclUpdater =
                new YpObjectPermissionsUpdaterImpl<>(ypTransactionMasterClient, ypStageRepository);
        YpObjectPermissionsUpdater ypNannyServiceAclUpdater =
                new YpObjectPermissionsUpdaterImpl<>(ypTransactionMasterClient, nannyServiceRepository);

        YpGroupsClient ypGroupsMasterClient = new YpGroupsClientImpl(groupsRepository, config.getString("main.robot"), config.getInt("yp.select_abc_group_page_size"));
        YpObjectPermissionsUpdater ypProjectAclUpdater =
                new YpObjectPermissionsUpdaterImpl<>(ypTransactionMasterClient, projectRepository);

        Map<String, YpRawObjectService> ypSlaveClientsPerCluster =
                ConfigUtils.ypClientsPerCluster(config.getConfig("yp"), config.getConfig("yp.slaves"), gaugeRegistry, dryRun);

        YpClients master = new YpClients(ypTransactionMasterClient, ypGroupsMasterClient,
                projectRepository, ypStageRepository, userRepository, nannyServiceRepository, groupsRepository);

        Map<String, YpClients> slaves = StreamEx.of(ypSlaveClientsPerCluster.keySet()).toMap(cluster -> {
            var rawClient = ypSlaveClientsPerCluster.get(cluster);
            var transactionalClient = new YpTransactionClientImpl(rawClient);
            var groupRepository = ConfigUtils.ypGroupRepository(rawClient, config.getInt("yp.request_group_page_size"), gaugeRegistry, cluster);
            var groupClient = new YpGroupsClientImpl(groupRepository, config.getString("main.robot"), config.getInt("yp.select_abc_group_page_size"));
            return new YpClients(transactionalClient, groupClient);
        });

        return new YpServiceImpl(
                ypObjectsTreeGetter, ypStageAclUpdater, ypProjectAclUpdater, ypNannyServiceAclUpdater,
                master,
                slaves,
                rolesInfo, rolesConfig.getConfig("roles_tree.root"), config.getDuration("yp.groups_requests_rate"),
                config.getInt("yp.groups_requests_limit"),
                metricRegistry,
                metrics,
                config.getString("main.system_name")
        );
    }

    private static AsyncHttpClient createHttpClient(Config config) {
        int connectTimeout = (int) config.getDuration("connect_timeout").toMillis();
        int requestTimeout = (int) config.getDuration("request_timeout").toMillis();
        return new DefaultAsyncHttpClient(
                new DefaultAsyncHttpClientConfig.Builder()
                        .setRequestTimeout(requestTimeout)
                        .setConnectTimeout(connectTimeout)
                        .build()
        );
    }

    private static IdmApiService createIdmApiService(Config config,
                                                     RolesInfo rolesInfo,
                                                     boolean isReadonlyMode,
                                                     Metrics metrics,
                                                     StaffApi staffApi,
                                                     GaugeRegistry gaugeRegistry) {
        final Config idmConfig = config.getConfig("idm");

        final Config httpConfig = idmConfig.getConfig("http");
        AsyncHttpClient httpClient = createHttpClient(httpConfig);
        var idmMetricsRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "idm");

        IdmApi idmApi = isReadonlyMode ? IdmApiReadOnlyImpl.configure(idmConfig, httpClient, idmMetricsRegistry)
                : IdmApiImpl.configure(idmConfig, httpClient, idmMetricsRegistry);

        return IdmApiService.configure(
                httpConfig,
                idmApi,
                staffApi,
                rolesInfo,
                idmConfig.getString("system"),
                idmMetricsRegistry
        );
    }

    private static StaffApi createStaffApiService(Config config, GaugeRegistry gaugeRegistry) {
        final Config staffConfig = config.getConfig("staff");
        AsyncHttpClient httpClient = createHttpClient(staffConfig.getConfig("http"));
        return new StaffApiImpl(staffConfig, httpClient, new NamespacedGaugeRegistry(gaugeRegistry, "staff"));
    }

    private static NannyRolesSynchronizer createNannyRolesSynchronizer(Config nannyConfig,
                                                GaugeRegistry gaugeRegistry,
                                                YpService ypService,
                                                LeaderService leaderService,
                                                GroupsAndUsersCache groupsAndUsersCache,
                                                RolesInfo rolesInfo,
                                                boolean isReadonlyMode) {
        AsyncHttpClient httpClient = createHttpClient(nannyConfig.getConfig("http"));
        NamespacedGaugeRegistry nannyRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "nanny");
        NannyApi nannyApi = isReadonlyMode ? new ReadOnlyNannyApiImpl(nannyConfig, httpClient, nannyRegistry) :
                new NannyApiImpl(nannyConfig, httpClient, nannyRegistry);
        NannyRolesSynchronizer synchronizer = new NannyRolesSynchronizer(nannyConfig,
                nannyApi,
                ypService,
                groupsAndUsersCache,
                leaderService,
                gaugeRegistry,
                rolesInfo);
        synchronizer.start();
        return synchronizer;
    }

    private static NannyService createNannyService(Config config,
                                                   GaugeRegistry gaugeRegistry,
                                                   YpService ypService,
                                                   LeaderService leaderService,
                                                   GroupsAndUsersCache groupsAndUsersCache,
                                                   RolesInfo rolesInfo,
                                                   boolean isReadonlyMode) {
        Config nannyConfig = config.getConfig("nanny");

        if (!nannyConfig.getBoolean("enabled")) {
            LOG.warn("Nanny IDM nodes synchronization is disabled by config option nanny.enabled = false");
            return NannyService.DISABLED;
        }

        return createNannyRolesSynchronizer(nannyConfig, gaugeRegistry, ypService, leaderService, groupsAndUsersCache, rolesInfo, isReadonlyMode);
    }

    private static RolesUpdater createRolesUpdater(
            Config config,
            IdmApiService idmApiService,
            YpService ypService,
            Watchdog watchdog,
            LeaderService leaderService,
            NannyService nannyRolesProvider,
            GaugeRegistry gaugeRegistry) {

        Config mainConfig = config.getConfig("main");

        List<RoleSubject> superUserSubjects = mainConfig.getStringList("super_users").stream()
                .map(subject -> {
                    RoleSubject roleSubject;
                    final Role superUserRole = Role.superUser();
                    try {
                        roleSubject = new RoleSubject(
                                "",
                                Long.parseLong(subject),
                                superUserRole
                        );
                    } catch (NumberFormatException e) {
                        roleSubject = new RoleSubject(
                                subject,
                                0L,
                                superUserRole
                        );
                    }
                    return roleSubject;
                })
                .collect(Collectors.toList());

        Duration roleUpdateRate = mainConfig.getDuration("roles_update_rate");
        Duration watchdogTimeout = mainConfig.getDuration("watchdog_timeout");

        return new RolesUpdater(
                idmApiService,
                ypService,
                nannyRolesProvider,
                leaderService,
                roleUpdateRate,
                superUserSubjects,
                watchdogTimeout,
                watchdog,
                mainConfig.getBoolean("yp_groups_gc"),
                mainConfig.getBoolean("yp_stage_acl_cleanup"),
                config.getBoolean("nanny.process_services_without_project"),
                mainConfig.getString("system_name"),
                gaugeRegistry
        );
    }

    private static void createAndStartHttpServers(
            Config config,
            YpService ypService,
            RolesInfo rolesInfo,
            MetricRegistry metricRegistry,
            IdmAuthorization idmAuthorizationStrategy,
            IdmService idmService,
            RolesUpdater updater,
            RoleSubjectsProvider roleSubjectsProvider) {
        List<Server> servers = getServers(
                config,
                metricRegistry,
                idmService,
                ypService,
                rolesInfo,
                idmAuthorizationStrategy,
                updater,
                roleSubjectsProvider
        );

        try {
            for (Server server : servers) {
                server.start();
            }
        } catch (Exception e) {
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }

    private static TvmTicketUpdater createAndStartTvmTicketUpdater(Config config, Watchdog watchdog) {
        int clientId = config.getInt("client_id");
        int sourceId = config.getInt("source_id");
        String secret = token(config.getString("secret_file"));
        Duration tvmTicketUpdateRate = config.getDuration("ticket_update_rate");
        Duration tvmWatchdogTimeout = config.getDuration("watchdog_timeout");

        ScheduledExecutorService executor = new ScheduledThreadPoolExecutor(1);
        return new TvmTicketUpdater(
                clientId, sourceId, secret, executor, tvmTicketUpdateRate, tvmWatchdogTimeout, watchdog
        );
    }

    private static List<Server> getServers(
            Config config,
            MetricRegistry metricRegistry,
            IdmService idmService,
            YpService ypService,
            RolesInfo rolesInfo,
            IdmAuthorization authorizationStrategy,
            RolesUpdater updater,
            RoleSubjectsProvider roleSubjectsProvider) {

        Map<String, Map<String, HttpServlet>> servlets = ImmutableMap.of(
                "private_api", Map.of(
                        "get-nodes", new PrivateApiServlet(ypService, LIST_ROLE_NODES),
                        "get-subjects", new PrivateApiServlet(ypService, LIST_ROLE_SUBJECTS, roleSubjectsProvider),
                        "get-projects-without-owner", new PrivateApiServlet(ypService, LIST_PROJECTS_WITHOUT_OWNER, roleSubjectsProvider),
                        "remove-subject", new PrivateApiServlet(ypService, REMOVE_ROLE_SUBJECT),
                        "cleanup", new PrivateApiServlet(ypService, CLEANUP_NODES),
                        "update-acls", new PrivateApiServlet(ypService, UPDATE_ROLE_ACLS, roleSubjectsProvider),
                        "groups-gc", new GroupsGCServlet(ypService),
                        "check-acls", new AclServlet(ypService)),
                "public_api", Map.of(
                        "idm/info/", new IdmInfoServlet(idmService, authorizationStrategy),
                        "idm/add-role/", new IdmAddRemoveRoleServlet(idmService, authorizationStrategy, IdmService.RequestType.ADD_ROLE),
                        "idm/remove-role/", new IdmAddRemoveRoleServlet(idmService, authorizationStrategy, IdmService.RequestType.REMOVE_ROLE),
                        "idm/get-all-roles/", new IdmGetRolesServlet(idmService, authorizationStrategy),
                        "idm/add-batch-memberships/", new IdmAddRemoveMembershipServlet(
                                idmService, authorizationStrategy, IdmService.RequestType.ADD_BATCH_MEMBERSHIPS),
                        "idm/remove-batch-memberships/", new IdmAddRemoveMembershipServlet(
                                idmService, authorizationStrategy, IdmService.RequestType.REMOVE_BATCH_MEMBERSHIPS),
                        "idm/get-memberships/", new IdmGetMembershipsServlet(idmService, authorizationStrategy),
                        "get-project-acl", new ApiGetProjectAclServlet(idmService, rolesInfo, roleSubjectsProvider, config.getBoolean("main.reply_404_for_missed_project"))),
                "unistat", Map.of(
                        "unistat", new UnistatServlet(metricRegistry))
        );

        var httpConfig = config.getConfig("http_server");

        return httpConfig.root().keySet().stream()
                .map(entry -> buildServer(httpConfig.getConfig(entry), servlets.get(entry)))
                .collect(Collectors.toList());
    }
}
