package ru.yandex.infra.auth;

import java.util.Deque;
import java.util.concurrent.ConcurrentLinkedDeque;

import com.codahale.metrics.MetricRegistry;

import ru.yandex.qe.telemetry.metrics.Gauges;

public class Metrics {
    private static final String TMP_QUOTA_USAGE = "tmp_quota_usage";
    private static final String QUOTA_MISMATCH = "quota_mismatch";
    private static final String YP_ERRORS = "yp_errors_count";
    private static final String YP_IDM_GROUPS_COUNT = "yp.idm_groups_count";

    private final Deque<Long> ypErrorsTimestamps;
    private final long errorsTTL;
    private int tmpQuotaUsageCount;
    private int quotaMismatchCount;
    private Integer ypIDMGroupsCount;

    public Metrics(MetricRegistry metricRegistry, long errorsTTL) {
        this.ypErrorsTimestamps = new ConcurrentLinkedDeque<>();
        this.errorsTTL = errorsTTL;
        reset();
        Gauges.forSupplier(
                metricRegistry,
                TMP_QUOTA_USAGE,
                this::getTmpQuotaUsageCount
        );
        Gauges.forSupplier(
                metricRegistry,
                QUOTA_MISMATCH,
                this::getQuotaMismatchCount
        );
        Gauges.forSupplier(
                metricRegistry,
                YP_ERRORS,
                this::getYpErrorsCount
        );
        Gauges.forSupplier(
                metricRegistry,
                YP_IDM_GROUPS_COUNT,
                this::getYpIDMGroupsCount
        );
    }

    public void reset() {
        tmpQuotaUsageCount = 0;
        quotaMismatchCount = 0;
        ypErrorsTimestamps.clear();
    }

    public int getTmpQuotaUsageCount() {
        return tmpQuotaUsageCount;
    }

    public void setTmpQuotaUsageCount(int tmpQuotaUsageCount) {
        this.tmpQuotaUsageCount = tmpQuotaUsageCount;
    }

    public int getQuotaMismatchCount() {
        return quotaMismatchCount;
    }

    public void setQuotaMismatchCount(int quotaMismatchCount) {
        this.quotaMismatchCount = quotaMismatchCount;
    }

    public void addYpError() {
        ypErrorsTimestamps.addLast(System.currentTimeMillis());
    }

    public int getYpErrorsCount() {
        return getErrorsCountTemplate(ypErrorsTimestamps, errorsTTL);
    }

    private int getErrorsCountTemplate(Deque<Long> errorsTimestamps, Long errorsTTL) {
        long currentTime = System.currentTimeMillis();
        while (!errorsTimestamps.isEmpty() && currentTime - errorsTimestamps.getFirst() > errorsTTL) {
            errorsTimestamps.removeFirst();
        }
        return errorsTimestamps.size();
    }

    public Integer getYpIDMGroupsCount() {
        return ypIDMGroupsCount;
    }

    public void setYpIDMGroupsCount(Integer ypIDMGroupsCount) {
        this.ypIDMGroupsCount = ypIDMGroupsCount;
    }
}
