package ru.yandex.infra.auth;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;

import ru.yandex.misc.lang.StringUtils;

import static org.apache.commons.lang3.math.NumberUtils.min;

public class Role implements Comparable<Role> {
    public static final String ROLE_NAME_DELIMITER = ".";
    public static final String NANNY_ROLES_PARENT_NODE = "<Nanny>";
    public static final String ROLE_SUPER_USER = "SUPER_USER";

    public static final int PROJECT_ID_LEVEL_DEPTH_IN_ROLE = 0;
    public static final int STAGE_ID_LEVEL_DEPTH_IN_ROLE = 1;
    public static final int BOX_ID_LEVEL_DEPTH_IN_ROLE = 2;
    public static final int NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE = 2;
    public static final int PROJECTLESS_NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE = 1;
    public static final int MAX_DEPLOY_ROLE_DEPTH = 4;

    private final List<String> levels;
    private final String leaf;
    private final String uniqueId;

    //Used only for Project nodes, to grant basic Owner role membership to Project's creator (user from Staff)
    private final String creator;

    public static Role empty() {
        return new Role("", "");
    }

    public static Role superUser() {
        return new Role("", ROLE_SUPER_USER);
    }

    public Role(String rolePath, String leaf, String creator, String uniqueId) {
        this.levels = Lists.newArrayList(rolePath.split("\\" + ROLE_NAME_DELIMITER));
        this.leaf = replaceNullWithEmpty(leaf);
        this.creator = replaceNullWithEmpty(creator);
        this.uniqueId = replaceNullWithEmpty(uniqueId);

        if (!leaf.isEmpty()) {
            levels.add(leaf);
        }
    }

    @VisibleForTesting
    /**
     * Creates role from path/leaf. Constructor for Unit tests only.
     * @deprecated
     * Should not be used in production code. Leaved only because of 282 usages in unit/integration tests...
     */
    public Role(String rolePath, String leaf) {
        this(rolePath, leaf, "", "");
    }

    public boolean isRoot() {
        return levels.size() == 1 && levels.get(0).isEmpty();
    }

    public boolean isNannyRole() {
        return levels.size() > 0 && levels.get(0).equals(NANNY_ROLES_PARENT_NODE) || //Service without Project
                levels.size() > 1 && levels.get(1).equals(NANNY_ROLES_PARENT_NODE);  //Service with Project
    }

    public boolean isNannyParentNode() {
        return (levels.size() == 1 && levels.get(0).equals(NANNY_ROLES_PARENT_NODE)) ||
                (levels.size() == 2 && levels.get(1).equals(NANNY_ROLES_PARENT_NODE));
    }


    public boolean isNannyServiceRootNode() {
        return (levels.size() == (PROJECTLESS_NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE + 1) && //Service without Project
                    levels.get(0).equals(NANNY_ROLES_PARENT_NODE) && StringUtils.isBlank(leaf)) ||
               (levels.size() == (NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE + 1) && //Service with Project
                    levels.get(1).equals(NANNY_ROLES_PARENT_NODE));
    }

    public Optional<String> getProjectId() {
        if (levels.size() != 0) {
            String projectId = levels.get(0);
            if (!projectId.isEmpty() && !projectId.equals(NANNY_ROLES_PARENT_NODE)) {
                return Optional.of(projectId);
            }
        }

        return Optional.empty();
    }

    public Optional<String> getStageId() {
        int stageLevelPlus1 = STAGE_ID_LEVEL_DEPTH_IN_ROLE + 1;
        if (!isNannyRole() && levels.size() == stageLevelPlus1 && leaf.isEmpty() ||
                levels.size() > stageLevelPlus1) {
            return Optional.of(levels.get(STAGE_ID_LEVEL_DEPTH_IN_ROLE));
        }
        return Optional.empty();
    }

    public Optional<String> getNannyServiceId() {
        if (isNannyRole()) {
            if (levels.get(0).equals(NANNY_ROLES_PARENT_NODE)) {
                if (levels.size() > PROJECTLESS_NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE) {
                    return Optional.of(levels.get(PROJECTLESS_NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE));
                }
            } else if (levels.size() > NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE) {
                return Optional.of(levels.get(NANNY_SERVICE_ID_LEVEL_DEPTH_IN_ROLE));
            }
        }
        return Optional.empty();
    }

    public Optional<String> getLevelName(int level) {
        try {
            return Optional.of(levels.get(level));
        } catch (IndexOutOfBoundsException ex) {
            return Optional.empty();
        }
    }

    public String getLeaf() {
        return leaf;
    }

    public String getCreator() {
        return creator;
    }

    public String getUniqueId() {
        return uniqueId;
    }

    public Optional<String> getYpObjectUuid() {
        if (uniqueId.isEmpty()) {
            return Optional.empty();
        }
        var splitted = uniqueId.split("\\" + ROLE_NAME_DELIMITER);
        return Optional.of(splitted[splitted.length - 1]);
    }

    public int size() {
        return levels.size();
    }

    public boolean contains(Role role) {
        for (int index = 0; index < role.size(); ++index) {
            if (!role.getLevelName(index).equals(getLevelName(index))) {
                return false;
            }
        }
        return true;
    }

    public String getLevelsJoinedWithDelimiter() {
        return levels.stream()
                .filter(level -> !level.isEmpty())
                .collect(Collectors.joining(ROLE_NAME_DELIMITER));
    }

    public String getExtendedDescription() {
        return getLevelsJoinedWithDelimiter() + (uniqueId.isEmpty() ? "" : ", uniqueId = " + uniqueId);
    }

    public static String getLeafUniqueId(String parentUniqueId, String role) {
        if (StringUtils.isBlank(parentUniqueId)) {
            return "";
        }
        return role + ROLE_NAME_DELIMITER + parentUniqueId;
    }

    public static String getRolePathForNannyService(String projectId, String serviceName) {
        return (StringUtils.isBlank(projectId) ? "" : projectId + Role.ROLE_NAME_DELIMITER)
                + Role.NANNY_ROLES_PARENT_NODE + Role.ROLE_NAME_DELIMITER
                + serviceName;
    }


    public static String getIDMKeyNodeName(boolean isNannyRole, boolean isProjectLessNannyRole, int level, String lastLevelName) {
        switch (level) {
            case 0: return "Project";
            case 1: return isNannyRole ? "Service name" : "Project role or Stage name";
            case 2: return isNannyRole ? (isProjectLessNannyRole ? "role" : "Service name") : "Stage role or box name";
            case 3: return "role";
            default: return lastLevelName;
        }
    }

    public String getIDMKeyNodeName() {
        return getIDMKeyNodeName(isNannyRole(), levels.get(0).equals(NANNY_ROLES_PARENT_NODE), levels.size(), getLastLevelName());
    }

    @VisibleForTesting
    String getLastLevelName() {
        return leaf.isEmpty() && levels.size() > 0 ? levels.get(levels.size()-1) : leaf;
    }

    /**
     * toString implementation
     * @deprecated
     * This method is no longer acceptable to avoid confusion.
     * <p> Use {@link Role#getLevelsJoinedWithDelimiter()} for identification and
     * {@link Role#getExtendedDescription()} for logging.
     * @return string representation of role
     */
    @Override
    @Deprecated
    public String toString() {
        //Old usages of Role.toString() were replaced with getLevelsJoinedWithDelimiter() / getExtendedDescription() / getYpGroupName()
        //I've left old behaviour here in case of missed implicit usages of Role.toString()
        return getLevelsJoinedWithDelimiter();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Role)) {
            return false;
        }
        Role role = (Role) o;
        return Objects.equals(levels, role.levels) &&
                Objects.equals(leaf, role.leaf) &&
                Objects.equals(uniqueId, role.uniqueId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(levels, leaf, uniqueId);
    }

    @Override
    public int compareTo(Role o) {
        for (int level = 0; level < min(size(), o.size()); ++level) {
            int result = getLevelName(level).orElseThrow().compareTo(o.getLevelName(level).orElseThrow());
            if (result != 0) {
                return result;
            }
        }

        int diff = size() - o.size();
        if (diff != 0) {
            return diff;
        }

        return uniqueId.compareTo(o.uniqueId);
    }

    private static String replaceNullWithEmpty(String str) {
        return str == null ? "" : str;
    }

    public RolesInfo.LevelName getLeafLevel() {
        if(leaf.isEmpty()) {
            return null;
        }

        if (levels.size() == 1 || (levels.size() == 2 && levels.get(0).isEmpty())) {
            return RolesInfo.LevelName.SYSTEM;
        }
        if (levels.size() == 2) {
            return RolesInfo.LevelName.PROJECT;
        }
        if (levels.size() == 3) {
            return isNannyRole() ? RolesInfo.LevelName.NANNY_SERVICE : RolesInfo.LevelName.STAGE;
        }

        if (levels.size() == 4) {
            return isNannyRole() ? RolesInfo.LevelName.NANNY_SERVICE : RolesInfo.LevelName.BOX;
        }

        return null;
    }

    public int getLevelIgnoringLeaf() {
        if (isRoot()) {
            return 0;
        }
        return levels.size() - (StringUtils.isBlank(leaf) ? 0 : 1);
    }
}
