package ru.yandex.infra.auth;

import java.util.Objects;

import com.google.common.base.MoreObjects;

public class RoleSubject implements Comparable<RoleSubject> {
    private final String login;
    private final Long groupId;
    private final Role role;

    public RoleSubject(String login, Long groupId, Role role) {
        if (login == null || groupId == null) {
            throw new IllegalArgumentException("Login and GroupId cannot be null");
        }

        if (!login.isEmpty() && groupId != 0L) {
            throw new IllegalArgumentException("Login and GroupId cannot be specified for one role subject");
        }

        if (login.isEmpty() && groupId == 0L) {
            throw new IllegalArgumentException("Login and GroupId cannot be empty simultaneously");
        }

        this.login = login;
        this.groupId = groupId;
        this.role = role;
    }

    public String getLogin() {
        return login;
    }

    public Long getGroupId() {
        return groupId;
    }

    public Role getRole() {
        return role;
    }

    public boolean isPersonal() {
        return !login.isEmpty();
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("login", login)
                .add("groupId", groupId)
                .add("role", role.getExtendedDescription())
                .toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RoleSubject that = (RoleSubject) o;
        return Objects.equals(login, that.login) &&
                Objects.equals(groupId, that.groupId) &&
                role.equals(that.role);
    }

    @Override
    public int hashCode() {
        return Objects.hash(login, groupId, role);
    }

    @Override
    public int compareTo(RoleSubject o) {
        int roleCompareResult = role.compareTo(o.getRole());
        if (roleCompareResult != 0) {
            return roleCompareResult;
        }
        // difference is in subject
        int loginCompareResult = login.compareTo(o.getLogin());
        if (loginCompareResult != 0) {
            return loginCompareResult;
        }

        return groupId.compareTo(o.getGroupId());
    }
}
