package ru.yandex.infra.auth;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import com.typesafe.config.Config;

import ru.yandex.infra.auth.idm.service.IdmLeaf;
import ru.yandex.infra.controller.util.AclUtils;
import ru.yandex.yp.client.api.AccessControl;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;

public final class RolesInfo {
    private static final String CONFIG_KEY_DELIMITER = ".";
    private static final String ROLES_INFO_KEY = "roles_info";
    private static final String DESCRIPTION_KEY = "description";
    private static final String ACES_KEY = "aces";
    private static final String LEVELS_KEY = "levels";
    private static final String MAPPED_ROLES_KEY = "mapped_roles";
    private static final String PERMISSIONS_KEY = "permissions";
    private static final String ATTRIBUTE_PATH_KEY = "attribute_path";
    private static final String BASE_ATTRIBUTE_PATH_KEY = "base_attribute_path";

    private static final String ROLES_TREE_ROOT_KEY = "roles_tree.root";
    private static final String ROLES_KEY = "roles";
    private static final String NOT_VISIBLE_ROLES_KEY = "not_visible_roles";
    private static final String DEFAULT_CHILDREN_ROLES_KEY = "children.__default__";

    private final Map<String, IdmLeaf> roleDescriptions;
    private final Map<String, List<RoleAce>> roleAces;
    private final Map<String, Set<String>> mappedRoles;
    private final Map<String, Set<String>> allRolesPerLevel;
    private final List<List<String>> visibleRolesPerLevel;

    public final static class RoleAce {
        private final Set<AccessControl.EAccessControlPermission> rolePermissions;
        private final List<String> roleAttributePaths;
        private final String baseAttributePath;

        public RoleAce(Set<AccessControl.EAccessControlPermission> rolePermissions,
                List<String> roleAttributePaths) {
            this(rolePermissions, roleAttributePaths, "");
        }

        public RoleAce(Set<AccessControl.EAccessControlPermission> rolePermissions,
                List<String> roleAttributePaths,
                String baseAttributePath) {
            this.rolePermissions = rolePermissions;
            this.roleAttributePaths = roleAttributePaths;
            this.baseAttributePath = baseAttributePath;
        }

        public Set<AccessControl.EAccessControlPermission> getRolePermissions() {
            return rolePermissions;
        }

        public List<String> getRoleAttributePaths() {
            return roleAttributePaths;
        }

        public String getBaseAttributePath() {
            return baseAttributePath;
        }
    }

    public enum LevelName {
        SYSTEM("system"),
        PROJECT("project"),
        STAGE("stage"),
        BOX("box"),
        NANNY_SERVICE("nanny_service");

        private final String name;

        LevelName(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    public RolesInfo(Map<String, IdmLeaf> roleDescriptions,
            Map<String, List<RoleAce>> roleAces,
            Map<String, Set<String>> mappedRoles,
            List<List<String>> rolesPerLevel,
            Map<String, Set<String>> allRolesPerLevel) {
        this.roleDescriptions = roleDescriptions;
        this.roleAces = roleAces;
        this.mappedRoles = mappedRoles;
        this.visibleRolesPerLevel = rolesPerLevel;
        this.allRolesPerLevel = allRolesPerLevel;
    }

    public static RolesInfo fromConfig(Config config) {
        Config rolesInfoConfig = config.getConfig(ROLES_INFO_KEY);
        Map<String, IdmLeaf> roleDescriptions = new HashMap<>();
        Map<String, List<RoleAce>> roleAces = new HashMap<>();
        Map<String, Set<String>> inheritedRoles = new HashMap<>();
        Map<String, Set<String>> allRolesPerLevel = new HashMap<>();

        rolesInfoConfig.root().keySet()
                .forEach(key -> {
                    roleDescriptions.put(
                            key,
                            IdmLeaf.configure(rolesInfoConfig.getConfig(key + CONFIG_KEY_DELIMITER + DESCRIPTION_KEY))
                    );
                    rolesInfoConfig.getStringList(key + CONFIG_KEY_DELIMITER + LEVELS_KEY).forEach(levelName ->
                            allRolesPerLevel.computeIfAbsent(levelName, k -> new HashSet<>())
                                    .add(key));

                    inheritedRoles.put(key, Set.copyOf(rolesInfoConfig.getStringList(key + CONFIG_KEY_DELIMITER + MAPPED_ROLES_KEY)));

                    List<RoleAce> roleAcesList = new ArrayList<>();
                    rolesInfoConfig.getConfigList(key + CONFIG_KEY_DELIMITER + ACES_KEY).forEach(
                            aceConfig -> {
                                Set<AccessControl.EAccessControlPermission> rolePermissions =
                                        aceConfig.getStringList(PERMISSIONS_KEY)
                                                .stream()
                                                .map(AclUtils::permissionToProto)
                                                .map(Optional::get)
                                                .collect(Collectors.toCollection(TreeSet::new));

                                List<String> roleAttributePaths = aceConfig.hasPath(ATTRIBUTE_PATH_KEY) ?
                                        aceConfig.getStringList(ATTRIBUTE_PATH_KEY) : emptyList();
                                String baseAttributePath = aceConfig.hasPath(BASE_ATTRIBUTE_PATH_KEY) ?
                                        aceConfig.getString(BASE_ATTRIBUTE_PATH_KEY) : "";
                                roleAcesList.add(new RoleAce(rolePermissions, roleAttributePaths, baseAttributePath));
                            }
                    );
                    roleAces.put(key, roleAcesList);
                });

        Config rolesTreeConfig = config.getConfig(ROLES_TREE_ROOT_KEY);
        List<List<String>> rolesPerLevel = new ArrayList<>();
        while (rolesTreeConfig.hasPath(ROLES_KEY)) {
            List<String> roles = rolesTreeConfig.getStringList(ROLES_KEY);

            if (rolesTreeConfig.hasPath(NOT_VISIBLE_ROLES_KEY)) {
                roles.removeAll(rolesTreeConfig.getStringList(NOT_VISIBLE_ROLES_KEY));
            }

            rolesPerLevel.add(roles);

            if (!rolesTreeConfig.hasPath(DEFAULT_CHILDREN_ROLES_KEY)) {
                break;
            }
            rolesTreeConfig = rolesTreeConfig.getConfig(DEFAULT_CHILDREN_ROLES_KEY);
        }

        return new RolesInfo(roleDescriptions, roleAces, inheritedRoles, rolesPerLevel, allRolesPerLevel);
    }

    public Optional<IdmLeaf> getRoleDescription(String role) {
        return Optional.ofNullable(roleDescriptions.get(role));
    }

    public List<RoleAce> getRoleAces(String role) {
        return roleAces.getOrDefault(role, emptyList());
    }

    public Set<String> getMappedRoles(String role) {
        return mappedRoles.getOrDefault(role, emptySet());
    }

    public List<String> getVisibleRolesPerLevel(int level) {
        return visibleRolesPerLevel.get(level);
    }

    public Set<String> getRolesPerLevel(LevelName levelName) {
        return allRolesPerLevel.getOrDefault(levelName.getName(), emptySet());
    }

    public static class Builder {
        private final Map<String, IdmLeaf> roleDescriptions = new HashMap<>();
        private final Map<String, List<RoleAce>> roleAces = new HashMap<>();

        public RolesInfo.Builder addRole(
                String name,
                IdmLeaf description,
                List<RoleAce> aces) {
            roleDescriptions.put(name, description);
            roleAces.put(name, aces);
            return this;
        }

        public RolesInfo build() {
            return new RolesInfo(roleDescriptions, roleAces, null, null, null);
        }
    }
}
