package ru.yandex.infra.auth;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;

public final class TreeNode {
    private final String name;
    private final List<TreeNode> children;
    private final String owner;
    private final boolean convertibleToRole;
    private final String uniqueId;

    private TreeNode(String name, List<TreeNode> children, String owner, boolean convertibleToRole, String uniqueId) {
        this.name = name;
        this.children = children;
        this.owner = owner;
        this.convertibleToRole = convertibleToRole;
        this.uniqueId = uniqueId;
    }

    public List<TreeNode> getChildren() {
        return children;
    }

    public String getName() {
        return name;
    }

    public String getOwner() {
        return owner;
    }

    public boolean isConvertibleToRole() {
        return convertibleToRole;
    }

    public String getUniqueId() {
        return uniqueId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof TreeNode)) {
            return false;
        }
        TreeNode that = (TreeNode) o;
        return Objects.equals(name, that.name) &&
                Objects.equals(children, that.children) &&
                Objects.equals(owner, that.owner) &&
                Objects.equals(convertibleToRole, that.convertibleToRole) &&
                Objects.equals(uniqueId, that.uniqueId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, children, owner, convertibleToRole, uniqueId);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("name", name)
                .add("children", children)
                .add("owner", owner)
                .add("convertableToRole", convertibleToRole)
                .add("uniqueId", uniqueId)
                .toString();
    }

    public static class Builder {
        private String uniqueId = "";
        private String name = "";
        private List<Builder> children = new ArrayList<>();
        private String owner = "";
        private boolean convertibleToRole = false;

        public Builder withName(String name) {
            this.name = name;
            return this;
        }

        public Builder withUniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
            return this;
        }

        public String getName() {
            return name;
        }

        public Builder withChild(Builder node) {
            children.stream()
                    .filter(child -> node.getName().equals(child.getName()))
                    .findFirst()
                    .ifPresentOrElse(nodeToMerge -> {
                        node.getChildren().forEach(nodeToMerge::withChild);
                        if (node.isConvertable()) {
                            nodeToMerge.makeConvertibleToRole();
                        }
                    }, () -> children.add(node));

            return this;
        }

        public List<Builder> getChildren() {
            return children;
        }

        public Builder withOwner(String owner) {
            this.owner = owner;
            return this;
        }

        public String getOwner() {
            return owner;
        }

        public String getUniqueId() {
            return uniqueId;
        }

        public Builder makeConvertibleToRole() {
            this.convertibleToRole = true;
            return this;
        }

        public boolean isConvertable() {
            return convertibleToRole;
        }

        public TreeNode build() {
            List<TreeNode> childrenList = this.children.stream().map(Builder::build).collect(Collectors.toList());

            return new TreeNode(name, childrenList, owner, convertibleToRole, uniqueId);
        }
    }
}
