package ru.yandex.infra.auth.idm.api;

import java.util.Objects;
import java.util.UUID;
import java.util.function.Function;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;

abstract class BatchRequest {
    abstract static class BatchBody { }

    private final Method method;
    private final String path;
    private final String id;
    private final BatchBody body;

    public final static Function<BatchBody, String> UUID_REQUEST_ID_GENERATOR = body -> UUID.randomUUID().toString();

    @VisibleForTesting
    static Function<BatchBody, String> requestIdGenerator = UUID_REQUEST_ID_GENERATOR;

    protected BatchRequest(Method method, String path, BatchBody body) {
        this.method = method;
        this.path = path;
        this.id = requestIdGenerator.apply(body);
        this.body = body;
    }

    @JsonProperty("path")
    public String getPath() {
        return path;
    }

    @JsonProperty("id")
    public String getId() {
        return id;
    }

    @JsonProperty("body")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public BatchBody getBody() {
        return body;
    }

    @JsonProperty("method")
    public Method getMethod() {
        return method;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof BatchRequest)) {
            return false;
        }
        BatchRequest that = (BatchRequest) o;
        return Objects.equals(method, that.method) &&
                Objects.equals(path, that.path) &&
                Objects.equals(id, that.id) &&
                Objects.equals(body, that.body);
    }

    @Override
    public int hashCode() {
        return Objects.hash(method, path, id, body);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("method", method)
                .add("path", path)
                .add("id", id)
                .add("body", body)
                .toString();
    }

    enum Method {
        GET("get"),
        POST("post"),
        PUT("put"),
        DELETE("delete");

        private final String value;

        Method(String value) {
            this.value = value;
        }

        @JsonValue
        public String getValue() {
            return value;
        }
    }
}
