package ru.yandex.infra.auth.idm.api;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeType;
import com.typesafe.config.Config;
import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;

import ru.yandex.infra.controller.metrics.GaugeRegistry;

import static java.lang.String.format;
import static org.slf4j.LoggerFactory.getLogger;
import static ru.yandex.infra.controller.util.ConfigUtils.token;

public final class IdmApiImpl extends IdmApiReadOnlyImpl {
    private static final Logger LOG = getLogger(IdmApiImpl.class);

    private static final String ROLE_REQUESTS_URL_PATH = "rolerequests";
    private static final String BATCH_URL_PATH = "batch";

    IdmApiImpl(AsyncHttpClient httpClient, String host, String token, String system, GaugeRegistry gaugeRegistry) {
        super(httpClient, host, token, system, gaugeRegistry);
    }

    public static IdmApi configure(Config config, AsyncHttpClient httpClient, GaugeRegistry gaugeRegistry) {
        return new IdmApiImpl(httpClient, config.getString(HOST_CONFIG_PATH),
                token(config.getString(TOKEN_FILE_CONFIG_PATH)),
                config.getString(SYSTEM_SLUG_CONFIG_PATH),
                gaugeRegistry);
    }

    @Override
    public CompletableFuture<HttpResponse> removeRoleNode(String path) {
        String url = format("%s/%s/%s/%s%s", host, IDM_API_URL_PATH, ROLE_NODES_URL_PATH, system, path);

        return runWithAuthorization(httpClient.prepareDelete(url),
                response -> {
                    LOG.info("removeRoleNode: {} {}", response.getStatusCode(), response.getResponseBody());
                    return new HttpResponse(response.getStatusCode(), response.getStatusText());
                });
    }

    @Override
    public CompletableFuture<BatchResponse> runBatch(List<BatchRequest> requests) throws RuntimeException {
        String url = format("%s/%s/%s/", host, IDM_API_URL_PATH, BATCH_URL_PATH);

        String json;
        try {
            json = mapper.writeValueAsString(requests);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error while convert requests to json", e);
        }
        return runWithAuthorization(httpClient.preparePost(url).setBody(json),
                response -> {
                    try {
                        LOG.debug("runBatch: {} {}", response.getStatusCode(), response.getResponseBody());
                        JsonNode root = mapper.readTree(response.getResponseBodyAsBytes());
                        BatchResponse.Builder batchBuilder = new BatchResponse.Builder();
                        batchBuilder.addStatusCode(response.getStatusCode());
                        root.get("responses").forEach(node -> {
                            JsonNode body = node.get("body");
                            batchBuilder.addResponse(BatchResponse.Response.fromJsonNode(
                                    node,
                                    body.getNodeType() == JsonNodeType.NULL || !body.has("error_code")
                                            ? "OK"
                                            : format("%s: %s", body.get("error_code"), body.get("message"))
                                    )
                            );
                        });
                        return batchBuilder.build();
                    } catch (IOException e) {
                        throw new RuntimeException("Error while parsing response from IDM: ", e);
                    }
                });
    }

    @Override
    public CompletableFuture<HttpResponse> addRoleSubject(AddSubjectRequest request) {
        String url = format("%s/%s/%s/", host, IDM_API_URL_PATH, ROLE_REQUESTS_URL_PATH);

        String json;
        try {
            json = mapper.writeValueAsString(request);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error while convert requests to json", e);
        }
        return runWithAuthorization(httpClient.preparePost(url).setBody(json),
                response -> {
                    if (response.getStatusCode() >= 400) {
                        try {
                            LOG.warn("addRoleSubject: {} {}", response.getStatusCode(), response.getResponseBody());
                            JsonNode root = mapper.readTree(response.getResponseBodyAsBytes());
                            return new HttpResponse(
                                    response.getStatusCode(),
                                    format("%s: %s", root.get("error_code"), root.get("message"))
                            );
                        } catch (IOException e) {
                            throw new RuntimeException("Error while parsing response from IDM: ", e);
                        }
                    }

                    return new HttpResponse(response.getStatusCode(), response.getStatusText());
                });
    }
}
