package ru.yandex.infra.auth.idm.api;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.JsonNode;
import com.typesafe.config.Config;
import org.asynchttpclient.AsyncHttpClient;

import ru.yandex.infra.auth.utils.ApiClientWithAuthorization;
import ru.yandex.infra.controller.metrics.GaugeRegistry;

import static java.lang.String.format;
import static ru.yandex.infra.controller.util.ConfigUtils.token;

public class IdmApiReadOnlyImpl extends ApiClientWithAuthorization implements IdmApi {

    protected static final String HOST_CONFIG_PATH = "host";
    protected static final String SYSTEM_SLUG_CONFIG_PATH = "system";
    protected static final String TOKEN_FILE_CONFIG_PATH = "token_file";
    protected static final String IDM_API_URL_PATH = "api/v1";
    protected static final String ROLE_NODES_URL_PATH = "rolenodes";

    protected final String system;

    private final CompletableFuture<HttpResponse> successResponse = new CompletableFuture<>();

    IdmApiReadOnlyImpl(AsyncHttpClient httpClient, String host, String token, String system, GaugeRegistry gaugeRegistry) {
        super(httpClient, host, token, gaugeRegistry);

        this.system = system;
        this.successResponse.complete(new HttpResponse(200, "Success (fake response due to readonly mode)"));
    }

    public static IdmApi configure(Config config, AsyncHttpClient httpClient, GaugeRegistry gaugeRegistry) {
        return new IdmApiReadOnlyImpl(httpClient, config.getString(HOST_CONFIG_PATH),
                token(config.getString(TOKEN_FILE_CONFIG_PATH)),
                config.getString(SYSTEM_SLUG_CONFIG_PATH),
                gaugeRegistry);
    }

    @Override
    public CompletableFuture<RoleNodesResponse> getRoleNodesByOffset(int offset, int limit) {
        return getRoleNodes(Optional.of(offset), limit, Optional.empty());
    }

    @Override
    public CompletableFuture<RoleNodesResponse> getRoleNodesAfterNodeWithId(long lastKey, int limit) {
        return getRoleNodes(Optional.empty(), limit, Optional.of(lastKey));
    }

    public CompletableFuture<RoleNodesResponse> getRoleNodes(Optional<Integer> offset, int limit, Optional<Long> lastKey) {
        String url = format("%s/%s/%s/?system=%s&limit=%s&fields=%s", host, IDM_API_URL_PATH, ROLE_NODES_URL_PATH,
                system, limit, RoleNodeInfo.LIST_OF_FIELDS_FOR_IDM_NODES_API_REQUEST);

        if (offset.isPresent()) {
            url += "&offset=" + offset.get();
        } else {
            if (lastKey.isEmpty()) {
                throw new RuntimeException("Either offset or lastKey should be specified to perform getRoleNodes request");
            }
            url += "&last_key=" + lastKey.get();
        }

        return runWithAuthorization(httpClient.prepareGet(url), response -> {
            try {
                if (response.getStatusCode() != 200) {
                    throw new RuntimeException(format("Error response from idm, code - %s, status message - %s",
                            response.getStatusCode(),
                            response.getStatusText()));
                }
                JsonNode root = mapper.readTree(response.getResponseBodyAsBytes());

                return RoleNodesResponse.fromJsonNode(root);
            } catch (IOException e) {
                throw new RuntimeException("Error while parsing response from idm", e);
            }
        });
    }

    @Override
    public CompletableFuture<HttpResponse> removeRoleNode(String path) {
        return successResponse;
    }

    @Override
    public CompletableFuture<BatchResponse> runBatch(List<BatchRequest> requests) throws RuntimeException {
        CompletableFuture<BatchResponse> successBatchResponse = new CompletableFuture<>();
        BatchResponse.Builder builder = new BatchResponse.Builder();
        builder.addStatusCode(200);
        requests.forEach(
                request -> builder.addResponse(
                        new BatchResponse.Response(request.getId(), 200, "OK")
                )
        );

        successBatchResponse.complete(builder.build());
        return successBatchResponse;
    }

    @Override
    public CompletableFuture<HttpResponse> addRoleSubject(AddSubjectRequest request) {
        return successResponse;
    }

}
