package ru.yandex.infra.auth.idm.api;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.Stack;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.function.Function;
import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubject;
import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.auth.idm.service.IdmLeaf;
import ru.yandex.infra.auth.idm.service.IdmName;
import ru.yandex.infra.auth.idm.service.IdmRole;
import ru.yandex.infra.auth.nanny.NannyRole;
import ru.yandex.infra.auth.staff.StaffApi;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.misc.lang.StringUtils;

import static java.lang.String.format;
import static java.lang.Thread.sleep;
import static org.apache.http.HttpStatus.SC_BAD_GATEWAY;
import static org.apache.http.HttpStatus.SC_BAD_REQUEST;
import static org.apache.http.HttpStatus.SC_CREATED;
import static org.apache.http.HttpStatus.SC_GATEWAY_TIMEOUT;
import static org.apache.http.HttpStatus.SC_INTERNAL_SERVER_ERROR;
import static org.apache.http.HttpStatus.SC_NOT_FOUND;
import static org.apache.http.HttpStatus.SC_NO_CONTENT;
import static org.apache.http.HttpStatus.SC_OK;
import static org.apache.http.HttpStatus.SC_SERVICE_UNAVAILABLE;
import static ru.yandex.infra.auth.Role.ROLE_NAME_DELIMITER;
import static ru.yandex.infra.auth.idm.service.IdmName.EMPTY_NAME;
import static ru.yandex.infra.auth.idm.service.IdmRole.IDM_ROLE_PATH_DELIMITER;

public final class IdmApiService {
    private static final Logger LOG = LoggerFactory.getLogger(IdmApiService.class);

    static final String METRIC_ERRORS_COUNT = "errors_count";
    static final String METRIC_LOADED_NODES_COUNT = "loaded_nodes_count";
    static final String METRIC_TOTAL_NODES_COUNT = "total_nodes_count";

    private final AtomicLong metricErrorsCount = new AtomicLong();
    private volatile Integer metricLoadedNodesCount = null;
    private volatile Integer metricTotalNodesCount = null;

    private static final String ROLE_NODES_LIMIT_CONFIG_PATH = "role_nodes_limit";
    private static final String MAX_RETRIES_CONFIG_PATH = "max_retries";
    private static final String RETRY_DELAY_CONFIG_PATH = "retry_delay";
    private static final String GROUP_ROLE_DELIMITER = ":";
    private static final String ABC_GROUP_PREFIX = "abc";

    private final IdmApi idmApi;
    private final StaffApi staffApi;
    private final RolesInfo rolesInfo;
    private final String system;
    private final int maxRoleNodesRequestLimit;
    private final int maxRetries;
    private final Duration retryDelayInMilliseconds;
    private final long apiRequestsTimeoutMilliseconds;

    IdmApiService(IdmApi idmApi, StaffApi staffApi, RolesInfo rolesInfo, String system, int maxRoleNodesRequestLimit,
            int maxRetries, Duration retryDelayInMilliseconds, Duration apiRequestsTimeout, GaugeRegistry gaugeRegistry) {
        this.idmApi = idmApi;
        this.staffApi = staffApi;
        this.rolesInfo = rolesInfo;
        this.system = system;
        this.maxRoleNodesRequestLimit = maxRoleNodesRequestLimit;
        this.retryDelayInMilliseconds = retryDelayInMilliseconds;
        this.maxRetries = maxRetries;
        this.apiRequestsTimeoutMilliseconds = apiRequestsTimeout.toMillis();

        gaugeRegistry.add(METRIC_ERRORS_COUNT, new GolovanableGauge<>(metricErrorsCount::get, "dmmm"));
        gaugeRegistry.add(METRIC_LOADED_NODES_COUNT, new GolovanableGauge<>(() -> metricLoadedNodesCount, "axxx"));
        gaugeRegistry.add(METRIC_TOTAL_NODES_COUNT, new GolovanableGauge<>(() -> metricTotalNodesCount, "axxx"));
    }

    public static IdmApiService configure(Config config, IdmApi idmApi, StaffApi staffApi, RolesInfo rolesInfo,
            String system, GaugeRegistry gaugeRegistry) {
        return new IdmApiService(
                idmApi,
                staffApi,
                rolesInfo,
                system,
                config.getInt(ROLE_NODES_LIMIT_CONFIG_PATH),
                config.getInt(MAX_RETRIES_CONFIG_PATH),
                config.getDuration(RETRY_DELAY_CONFIG_PATH),
                config.getDuration("request_timeout"),
                gaugeRegistry);
    }

    private static Role convertRoleInfoToRole(RoleNodeInfo roleInfo) {
        String role = roleInfo.getValuePath();
        if (role.startsWith(IDM_ROLE_PATH_DELIMITER)) {
            role = role.substring(1);
        }
        if (role.endsWith(IDM_ROLE_PATH_DELIMITER)) {
            role = role.substring(0, role.length() - 1);
        }
        String leaf = "";
        if (!roleInfo.getHelp().isEmpty()) {
            String[] path = role.split(IDM_ROLE_PATH_DELIMITER);
            leaf = path[path.length - 1];
            role = role.substring(0, role.length() - leaf.length());
        }

        return new Role(role.replace(IDM_ROLE_PATH_DELIMITER, ROLE_NAME_DELIMITER), leaf, "", roleInfo.getUniqueId());
    }

    public Set<Role> getRoleNodes() throws IdmApiServiceError {
        Set<Role> roleList = new HashSet<>();
        int tryCounter = 0;
        Stack<Long> idmNodeIdsUsedAsLastKeyRequestParameter = new Stack<>();
        Set<Long> receivedIds = new HashSet<>();

        while (true) {
            try {
                if (tryCounter > 0) {
                    sleep(retryDelayInMilliseconds.toMillis());
                }

                RoleNodesResponse response = (idmNodeIdsUsedAsLastKeyRequestParameter.isEmpty() ?
                        idmApi.getRoleNodesByOffset(0, maxRoleNodesRequestLimit) :
                        idmApi.getRoleNodesAfterNodeWithId(idmNodeIdsUsedAsLastKeyRequestParameter.peek(), maxRoleNodesRequestLimit)).get();
                List<RoleNodeInfo> currentRoleNodes = response.getNodes();
                final int receivedNodesCount = currentRoleNodes.size();
                if (receivedNodesCount > maxRoleNodesRequestLimit) {
                    throw new IdmApiServiceError(
                            format("Unexpected number of nodes in IDM response: %d but expected %d",
                                    receivedNodesCount,
                                    maxRoleNodesRequestLimit));
                }

                if (receivedNodesCount == 0) {
                    LOG.error("Received empty list of nodes");
                    metricErrorsCount.incrementAndGet();
                    break;
                }

                roleList.addAll(currentRoleNodes.stream()
                        .filter(node -> node.getParentPath() != null && !node.isKey())
                        .map(IdmApiService::convertRoleInfoToRole)
                        .collect(Collectors.toSet()));

                List<Long> nodeIdentifiers = currentRoleNodes.stream()
                        .map(RoleNodeInfo::getId)
                        .sorted()
                        .distinct()
                        .collect(Collectors.toList());
                if (nodeIdentifiers.size() != receivedNodesCount) {
                    LOG.warn("Received {} nodes with {} unique ids", receivedNodesCount, nodeIdentifiers.size());
                }

                idmNodeIdsUsedAsLastKeyRequestParameter.push(nodeIdentifiers.get(nodeIdentifiers.size()-1));

                receivedIds.addAll(nodeIdentifiers);
                LOG.info("Got next {} nodes. Node identifiers range: [{};{}]. Received {} of {} nodes",
                        receivedNodesCount,
                        nodeIdentifiers.get(0),
                        nodeIdentifiers.get(nodeIdentifiers.size()-1),
                        receivedIds.size(), response.getTotalCount());
                metricLoadedNodesCount = receivedIds.size();
                if (metricTotalNodesCount == null || receivedIds.size() < response.getTotalCount()) {
                    metricTotalNodesCount = response.getTotalCount();
                }

                if (response.getNext() == null) {
                    break;//No more nodes left
                }
                tryCounter = 0;
            } catch (Exception e) {
                metricErrorsCount.incrementAndGet();
                if (++tryCounter >= maxRetries) {
                    throw new IdmApiServiceError("Max IDM retries exceeded", e);
                }
                if (!idmNodeIdsUsedAsLastKeyRequestParameter.isEmpty()) {
                    //trying to download previous batch in case of removed role node
                    //IDM replies with {"error_code": "BAD_REQUEST", "message": "Указанно некорректное значение ключа сортировки."}
                    idmNodeIdsUsedAsLastKeyRequestParameter.pop();
                }
                LOG.error("IDM error, try {}: ", tryCounter, e);
            }
        }

        LOG.info("Finally downloaded {} nodes ({} role nodes) from IDM", receivedIds.size(), roleList.size());

        if (!roleList.isEmpty()) {
            // Always add ROOT node if we have some nodes in IDM
            roleList.add(Role.empty());
        }
        return roleList;
    }

    public void removeRoleNode(Role role) throws IdmApiServiceError {
        IdmRole idmRole = IdmRole.createFromRole(role);
        runWithRetries(
                "Remove node",
                idmApi::removeRoleNode,
                (response, request) -> {
                    LOG.info("Remove role node {} response: {}", role.getExtendedDescription(), response);

                    if (response.isSuccess()) {
                        return RequestResult.OK;
                    }

                    if (response.getStatusCode() == SC_NOT_FOUND) {
                        LOG.warn("Role {} not found", role.getExtendedDescription());
                        return RequestResult.OK;
                    }

                    String errorMsg = format("error in response: %s while try delete node %s",
                            response.getStatusMessage(),
                            role.getExtendedDescription());

                    if (response.isServerError()) {
                        LOG.warn(errorMsg);
                        return RequestResult.NEED_RETRY;
                    }

                    // client error
                    LOG.error(errorMsg);
                    return RequestResult.BAD_REQUEST;
                },
                idmRole.getFullIdmPath()
        );
    }

    public void addRoleNodes(Collection<Role> roleNodes, Map<String, Role> rolesWithChangedProjectByUuid) throws IdmApiServiceError {
        Map<String, BatchRequest> allBatches = new LinkedHashMap<>();
        List<BatchRequest> ownerSubjectBatches = new ArrayList<>();
        for (Role role : roleNodes) {
            Role roleToRemove = rolesWithChangedProjectByUuid.get(role.getUniqueId());
            if (roleToRemove != null && StringUtils.isBlank(roleToRemove.getLeaf())) {
                LOG.info("Remove role node: {}", roleToRemove.getExtendedDescription());
                IdmRole idmRole = IdmRole.createFromRole(roleToRemove);
                var removeNodeRequest = new RemoveNodeRequest(system, idmRole.getFullIdmPath());
                allBatches.put(removeNodeRequest.getId(), removeNodeRequest);
            }

            List<BatchRequest> batches = createAddNodeRequest(role);
            batches.forEach(batch -> allBatches.put(batch.getId(), batch));

            if (!role.getCreator().isEmpty()) {
                String roleName = role.size() == 1 ? "OWNER" : "MAINTAINER";
                Role newRole = new Role(role.getLevelsJoinedWithDelimiter(), roleName, role.getCreator(), Role.getLeafUniqueId(role.getUniqueId(), roleName));
                RoleSubject subject = createRoleSubject(newRole);
                if (subject != null) {
                    ownerSubjectBatches.add(new BatchedAddSubjectRequest(createAddRoleSubjectRequest(subject)));
                }
            }

            if (role instanceof NannyRole) {
                NannyRole nannyRole = (NannyRole)role;
                nannyRole.getRoleSubjects().forEach(subject -> ownerSubjectBatches.add(new BatchedAddSubjectRequest(createAddRoleSubjectRequest(subject))));
            }
        }
        // Requests for role subjects have to go last in the batch
        ownerSubjectBatches.forEach(batch -> allBatches.put(batch.getId(), batch));

        runWithRetries(
                "Add nodes batch",
                idmApi::runBatch,
                (batchResponse, batchRequest) -> {
                    LOG.info("IDM response: {}", batchResponse);
                    switch (batchResponse.getStatusCode()) {
                        case SC_INTERNAL_SERVER_ERROR:
                        case SC_BAD_GATEWAY:
                        case SC_SERVICE_UNAVAILABLE:
                        case SC_GATEWAY_TIMEOUT:
                            LOG.error("Server error in response while try to run batch. Will retry.");
                            return RequestResult.NEED_RETRY;
                        case SC_BAD_REQUEST:
                            // filter out requests which cause client error
                            batchResponse.getResponses()
                                    .stream()
                                    .filter(HttpResponse::isClientError)
                                    .peek(response -> LOG.info("Client error for request id {}, status message: {}",
                                            response.getId(), response.getStatusMessage()))
                                    .map(BatchResponse.Response::getId)
                                    .forEach(allBatches::remove);

                            batchRequest.clear();
                            batchRequest.addAll(allBatches.values());
                            return batchRequest.isEmpty() ? RequestResult.OK : RequestResult.NEED_RESEND;
                        case SC_OK:
                        case SC_CREATED:
                        case SC_NO_CONTENT:
                            return RequestResult.OK;
                        default:
                            LOG.error("Unexpected error code from IDM server: {}", batchResponse.getStatusCode());
                            return RequestResult.BAD_REQUEST;
                    }
                },
                new ArrayList<>(allBatches.values())
        );
    }

    public void addRoleSubject(RoleSubject subject) throws IdmApiServiceError {
        AddSubjectRequest addSubjectRequest = createAddRoleSubjectRequest(subject);

        runWithRetries(
                "Add role subject",
                idmApi::addRoleSubject,
                (response, request) -> {
                    if (response.isSuccess()) {
                        LOG.info("Subject for node {} added", subject.getRole().getExtendedDescription());
                        return RequestResult.OK;
                    }

                    String errorMsg = format("Error while try add subject for node %s: %s",
                            subject.getRole().getExtendedDescription(),
                            response.getStatusMessage()
                    );

                    if (response.isServerError()) {
                        LOG.warn(errorMsg);
                        return RequestResult.NEED_RETRY;
                    }

                    // client error
                    LOG.error(errorMsg);
                    return RequestResult.BAD_REQUEST;
                },
                addSubjectRequest
        );
    }

    private <ReqT, RespT>
    void runWithRetries(String requestDescription,
                        Function<ReqT, CompletableFuture<RespT>> getResp,
                        BiFunction<RespT, ReqT, RequestResult> processingResponse,
                        ReqT request) throws IdmApiServiceError {
        for (int tryCounter = 0; tryCounter < maxRetries; ) {
            try {
                LOG.info("Sending request to idm, try {}: {} {}", tryCounter, requestDescription, request);
                RespT response = getResp.apply(request).get();
                RequestResult result = processingResponse.apply(response, request);

                switch (result) {
                    case OK:
                        return;

                    case NEED_RETRY:
                        metricErrorsCount.incrementAndGet();
                        Thread.sleep(retryDelayInMilliseconds.toSeconds());
                        ++tryCounter;
                        break;

                    case BAD_REQUEST:
                        metricErrorsCount.incrementAndGet();
                        throw new IdmApiServiceError(format("Bad request %s", request));

                    case NEED_RESEND:
                    default:
                        break;
                }

            } catch (InterruptedException | ExecutionException e) {
                metricErrorsCount.incrementAndGet();
                ++tryCounter;
                LOG.error("Error while send request, do retry", e);
            }
        }
        throw new IdmApiServiceError(format("Max retries exceeded for %s", request));
    }

    private List<BatchRequest> createAddNodeRequest(Role role) {
        List<BatchRequest> batchRequests = new ArrayList<>();
        IdmRole idmRole = IdmRole.createFromRole(role);
        String shortName = idmRole.getName();
        String pathToParent = idmRole.getIdmPath();

        LOG.info("Add node [{}]: slug = {}, parent = {}", role.getExtendedDescription(), shortName, pathToParent);
        if (role.getLeaf().isEmpty()) {
            AddNodeRequest.AddNodeRequestBody node = new AddNodeRequest.AddNodeRequestBody(
                    new IdmName(shortName),
                    EMPTY_NAME,
                    shortName,
                    system,
                    pathToParent,
                    role.getUniqueId()
            );
            batchRequests.add(new AddNodeRequest(node));

            if (!role.isRoot()) {
                String slug = role.getLevelsJoinedWithDelimiter();
                LOG.info("Add key node: slug = {}, parent = {}", slug, node.getPath());
                AddNodeRequest.AddNodeRequestBody keyNode = new AddNodeRequest.AddNodeRequestBody(
                        new IdmName(role.getIDMKeyNodeName()),
                        EMPTY_NAME,
                        slug,
                        system,
                        node.getPath(),
                        ""
                );
                batchRequests.add(new AddNodeRequest(keyNode));
            }
        } else {
            // add leaf node
            String leaf = role.getLeaf();
            Optional<IdmLeaf> leafOpt = rolesInfo.getRoleDescription(leaf);
            IdmLeaf idmLeaf = leafOpt.orElse(new IdmLeaf(new IdmName(leaf), new IdmName(leaf)));
            AddNodeRequest.AddNodeRequestBody leafNode = new AddNodeRequest.AddNodeRequestBody(
                    idmLeaf.getName(),
                    idmLeaf.getHelp(),
                    shortName,
                    system,
                    pathToParent,
                    role.getUniqueId()
            );
            batchRequests.add(new AddNodeRequest(leafNode));
        }

        return batchRequests;
    }

    private AddSubjectRequest createAddRoleSubjectRequest(RoleSubject subject) {
        AddSubjectRequest addSubjectRequest;
        LOG.info("Add subject [{}: {}]",
                subject.isPersonal() ? subject.getLogin() : subject.getGroupId(),
                subject.getRole().getExtendedDescription()
        );

        IdmRole idmRole = IdmRole.createFromRole(subject.getRole());
        if (subject.isPersonal()) {
            addSubjectRequest = new AddUserSubjectRequest(system,
                    idmRole.getValuePath(),
                    subject.getLogin());
        } else {
            addSubjectRequest = new AddGroupSubjectRequest(system,
                    idmRole.getValuePath(),
                    subject.getGroupId());
        }
        return addSubjectRequest;
    }

    private RoleSubject createRoleSubject(Role role) {
        RoleSubject roleSubject = null;
        String owner = role.getCreator();

        if (owner.contains(GROUP_ROLE_DELIMITER)) {
            String[] chunks = owner.split(GROUP_ROLE_DELIMITER);
            if (staffApi != null && chunks.length >= GroupRoleParts.MIN_SIZE.ordinal()) {
                Long groupId = null;
                if (chunks[GroupRoleParts.SOURCE.ordinal()].equals(ABC_GROUP_PREFIX)) {
                    // we have to convert abc service to staff group
                    try {
                        groupId = staffApi.getStaffGroupId(Long.parseLong(chunks[GroupRoleParts.ID.ordinal()], 10))
                                .get(apiRequestsTimeoutMilliseconds, TimeUnit.MILLISECONDS);
                    } catch (Exception ex) {
                        LOG.error("Failed to resolve owner group '{}' to staff group id", owner);
                    }
                } else {
                    groupId = Long.parseLong(chunks[GroupRoleParts.ID.ordinal()], 10);
                }
                if (groupId != null) {
                    roleSubject = new RoleSubject("", groupId, role);
                }
            }
        } else {
            // owner is person
            roleSubject = new RoleSubject(role.getCreator(), 0L, role);
        }
        return roleSubject;
    }

    private enum RequestResult {
        OK,
        BAD_REQUEST,
        NEED_RETRY,
        NEED_RESEND
    }

    private enum GroupRoleParts {
        SOURCE,
        TYPE,
        ID,
        MIN_SIZE
    }
}
