package ru.yandex.infra.auth.idm.service;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import ru.yandex.infra.auth.Role;

import static java.util.Collections.emptyMap;

public class IdmRole {
    public static final String IDM_ROOT_ROLE_NAME = "ROOT";
    public static final String IDM_ROLE_PATH_DELIMITER = "/";

    private final Map<String, String> roles;
    private final String idmPath;
    private final String name;
    private final String path;
    private final String idmFullPath;
    private final String valuePath;

    public IdmRole(Map<String, String> roles) {
        this.roles = roles;
        if (roles.isEmpty()) {
            name = IDM_ROOT_ROLE_NAME;
            path = "";
            idmFullPath = IDM_ROLE_PATH_DELIMITER + IDM_ROOT_ROLE_NAME + IDM_ROLE_PATH_DELIMITER;
            idmPath = IDM_ROLE_PATH_DELIMITER;
            valuePath = IDM_ROLE_PATH_DELIMITER;
        } else {
            StringBuilder fullPathBuilder = new StringBuilder();
            fullPathBuilder.append(IDM_ROLE_PATH_DELIMITER);

            String lastKey = IDM_ROOT_ROLE_NAME;
            String lastValue = roles.getOrDefault(IDM_ROOT_ROLE_NAME, "");
            while (true) {
                fullPathBuilder.append(lastKey);
                fullPathBuilder.append(IDM_ROLE_PATH_DELIMITER);
                fullPathBuilder.append(lastValue);
                fullPathBuilder.append(IDM_ROLE_PATH_DELIMITER);
                String newKey = lastKey.equals(IDM_ROOT_ROLE_NAME) ? lastValue
                        : String.format("%s.%s", lastKey, lastValue);
                if (!roles.containsKey(newKey)) {
                    break;
                }
                lastKey = newKey;
                lastValue = roles.get(lastKey);
            }

            path = (roles.size() > 1) ? lastKey : "";
            name = lastValue;
            idmFullPath = fullPathBuilder.toString();
            idmPath = idmFullPath.substring(0, idmFullPath.length() - name.length() - 1);
            valuePath = (path.isEmpty() ? IDM_ROLE_PATH_DELIMITER : IDM_ROLE_PATH_DELIMITER
                    + path.replace(Role.ROLE_NAME_DELIMITER, IDM_ROLE_PATH_DELIMITER)
                    + IDM_ROLE_PATH_DELIMITER)
                    + name + IDM_ROLE_PATH_DELIMITER;
        }
    }

    public static IdmRole createFromRole(Role role) {
        return new IdmRole(createMap(role));
    }

    private static Map<String, String> createMap(Role role) {
        final Map<String, String> map = new HashMap<>();

        if (role.isRoot()) {
            return emptyMap();
        }

        Optional<String> keyOpt = role.getLevelName(0);
        if (keyOpt.isEmpty()) {
            throw new IllegalArgumentException();
        }
        String key = keyOpt.get();
        if (key.isEmpty()) {
            // there is root role
            key = IDM_ROOT_ROLE_NAME;
        } else {
            map.put(IDM_ROOT_ROLE_NAME, key);
        }
        for (int level = 0; level < role.size() - 1; level++) {
            if (level != 0) {
                Optional<String> valueOpt = role.getLevelName(level);
                if (valueOpt.isEmpty()) {
                    throw new IllegalArgumentException();
                }
                key += Role.ROLE_NAME_DELIMITER + valueOpt.get();
            }
            Optional<String> valueOpt = role.getLevelName(level + 1);
            if (valueOpt.isEmpty()) {
                throw new IllegalArgumentException();
            }
            map.put(key, valueOpt.get());
        }
        return map;
    }

    public String getFullIdmPath() {
        return idmFullPath;
    }

    public String getIdmPath() {
        return idmPath;
    }

    public String getPath() {
        return path;
    }

    public String getName() {
        return name;
    }

    public String getValuePath() {
        return valuePath;
    }

    public Map<String, String> getRoles() {
        return roles;
    }
}
