package ru.yandex.infra.auth.idm.service;

import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubject;
import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.auth.nanny.NannyService;
import ru.yandex.infra.auth.yp.YpGroupsHelper;
import ru.yandex.infra.auth.yp.YpObjectsTreeGetterError;
import ru.yandex.infra.auth.yp.YpService;
import ru.yandex.infra.auth.yp.YpServiceException;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static ru.yandex.infra.auth.idm.service.IdmRole.IDM_ROOT_ROLE_NAME;
import static ru.yandex.infra.auth.idm.service.IdmServiceResponse.ResponseCode.RESPONSE_CODE_ERROR;
import static ru.yandex.infra.auth.idm.service.IdmServiceResponse.ResponseCode.RESPONSE_CODE_SUCCESS;
import static ru.yandex.infra.auth.idm.service.IdmServiceResponse.ResponseCode.RESPONSE_CODE_WARNING;

public final class IdmService {
    private static final Logger LOG = LoggerFactory.getLogger(IdmService.class);
    private static final TypeReference<Map<String, String>> STRING_MAP_TYPE = new TypeReference<>() {};
    private static final Map<String, String> IDM_GROUP_LABELS = Map.of(YpGroupsHelper.SYSTEM_LABEL_KEY, YpGroupsHelper.SYSTEM_IDM_LABEL_VALUE);

    private final YpService ypService;
    private final NannyService nannyService;
    private final RolesInfo rolesInfo;

    private final Map<RequestType, AtomicLong> metricRequests = new HashMap<>();
    private final Map<RequestType, AtomicLong> metricFailedRequests = new HashMap<>();

    public enum RequestType {
        INFO,
        ADD_ROLE,
        REMOVE_ROLE,
        GET_ALL_ROLES,
        ADD_BATCH_MEMBERSHIPS,
        REMOVE_BATCH_MEMBERSHIPS,
        GET_MEMBERSHIPS,
        GET_PROJECT_ACL;
    }

    public IdmService(YpService ypService, RolesInfo rolesInfo, NannyService nannyService, GaugeRegistry gaugeRegistry) {
        this.ypService = ypService;
        this.rolesInfo = rolesInfo;
        this.nannyService = nannyService;

        GaugeRegistry registry = new NamespacedGaugeRegistry(gaugeRegistry, "idm");

        Arrays.stream(RequestType.values()).forEach(requestType -> {
            final String name = requestType.name().toLowerCase();

            AtomicLong requests = new AtomicLong();
            metricRequests.put(requestType, requests);
            registry.add(name + "_requests_count", new GolovanableGauge<>(requests::get, "dmmm"));

            AtomicLong failedRequests = new AtomicLong();
            metricFailedRequests.put(requestType, failedRequests);
            registry.add(name + "_failed_requests_count", new GolovanableGauge<>(failedRequests::get, "dmmm"));
        });
    }

    public AtomicLong getRequestsMetric(RequestType requestType) {
        return metricRequests.get(requestType);
    }

    public AtomicLong getFailedRequestsMetric(RequestType requestType) {
        return metricFailedRequests.get(requestType);
    }

    private void incrementFailedRequestsMetric(RequestType requestType) {
        metricFailedRequests.get(requestType).incrementAndGet();
    }

    public IdmInfoResponse getInfo() {
        try {
            SortedSet<Role> roles = new TreeSet<>(ypService.getRoles());

            if (roles.isEmpty()) {
                return new IdmInfoResponse(
                        RESPONSE_CODE_WARNING,
                        "There is no role nodes in the system yet",
                        Optional.empty()
                );
            }

            final StringNode treeRoot = collectToTree(roles);
            final IdmRoleNode idmRoot = convertTree(treeRoot);
            return new IdmInfoResponse(
                    RESPONSE_CODE_SUCCESS,
                    "",
                    Optional.of(idmRoot)
            );
        } catch (YpObjectsTreeGetterError ex) {
            incrementFailedRequestsMetric(RequestType.INFO);
            return new IdmInfoResponse(
                    RESPONSE_CODE_ERROR,
                    "Failed to get role nodes from YP",
                    Optional.empty()
            );
        }
    }

    public IdmRolesResponse getRoles() {
        try {
            Set<RoleSubject> roleSubjects = ypService.getRoleSubjects();
            if (roleSubjects == null || roleSubjects.isEmpty()) {
                return new IdmRolesResponse(
                        RESPONSE_CODE_WARNING,
                        "Cannot find any roles in database",
                        emptySet(), emptySet());
            }

            Map<String, IdmPersonalRoles> loginToUserMap = new HashMap<>();
            Map<Long, IdmGroupRoles> idToGroupMap = new HashMap<>();
            for (RoleSubject roleSubject : roleSubjects) {
                IdmRole idmRole = IdmRole.createFromRole(roleSubject.getRole());

                if (roleSubject.isPersonal()) {
                    String login = roleSubject.getLogin();
                    loginToUserMap.computeIfAbsent(login, ignored -> new IdmPersonalRoles(login, new ArrayList<>()))
                            .addRole(idmRole);
                } else {
                    Long groupId = roleSubject.getGroupId();
                    idToGroupMap.computeIfAbsent(groupId, ignored -> new IdmGroupRoles(groupId, new ArrayList<>()))
                            .addRole(idmRole);
                }
            }
            return new IdmRolesResponse(
                    RESPONSE_CODE_SUCCESS,
                    "",
                    loginToUserMap.values(),
                    idToGroupMap.values()
            );
        } catch (Throwable exception) {
            incrementFailedRequestsMetric(RequestType.GET_ALL_ROLES);
            throw new RuntimeException("Failed to get roles from YP", exception);
        }
    }

    public IdmMembershipsResponse getAllMemberships() {
        try {
            Map<Long, Set<String>> groups = ypService.getGroupsWithLabels(IDM_GROUP_LABELS).entrySet().stream()
                    .collect(Collectors.toMap(
                            entry -> Long.parseLong(entry.getKey().replace(YpGroupsHelper.IDM_GROUP_PREFIX, "")),
                            Map.Entry::getValue));
            List<IdmMembership> memberships = groups.entrySet().stream()
                    .flatMap(entry -> entry.getValue().stream()
                            .map(login -> new IdmMembership(login, entry.getKey())))
                    .collect(Collectors.toList());
            return new IdmMembershipsResponse(RESPONSE_CODE_SUCCESS, "", memberships);
        } catch (Throwable exception) {
            incrementFailedRequestsMetric(RequestType.GET_MEMBERSHIPS);
            throw new RuntimeException("Failed to get staff groups membership from YP", exception);
        }
    }

    public IdmServiceResponse addOrRemoveRole(
            Optional<String> login,
            Optional<Long> groupId,
            String jsonEncodedRole,
            String uniqueId,
            IdmService.RequestType requestType) {
        String errorMessage;
        try {
            Map<String, String> roleInfoMap = new ObjectMapper().readValue(
                    URLDecoder.decode(jsonEncodedRole, StandardCharsets.UTF_8), STRING_MAP_TYPE
            );

            IdmRole idmRole = new IdmRole(roleInfoMap);
            Role role = new Role(idmRole.getPath(), idmRole.getName(), "", uniqueId);
            RoleSubject roleSubject = new RoleSubject(login.orElse(""), groupId.orElse(0L), role);

            //Creation of groups like "idm:234234"
            //i.e. role was granted to staff group ("idm:234234"), but IDM haven't pushed to AuthCtl yet
            //Todo: DEPLOY-5095
            if (requestType == RequestType.ADD_ROLE && groupId.isPresent()) {
                String idmGroupId = YpGroupsHelper.IDM_GROUP_PREFIX + groupId.get();
                try {
                    ypService.addMembersToGroup(idmGroupId, emptySet());
                } catch (YpServiceException ex) {
                    incrementFailedRequestsMetric(requestType);
                    return new IdmServiceResponse(RESPONSE_CODE_ERROR,
                            format("Failed to %s: %s. Group %s hasn`t been created. Details: %s",
                                    requestType, roleSubject, idmGroupId, ex.getMessage()));
                }
            }

            try {
                if (role.isNannyRole() && nannyService == NannyService.DISABLED) {
                    throw new RuntimeException("Nanny role updates are disabled by authctl config", null);
                }

                if (requestType == RequestType.ADD_ROLE) {
                    ypService.addRoleSubject(roleSubject);
                } else {
                    ypService.removeRoleSubject(roleSubject);
                }

                nannyService.updateRoleSubject(roleSubject, requestType);

                LOG.info("Succeeded to {}: {}", requestType, roleSubject);
                return new IdmServiceResponse(RESPONSE_CODE_SUCCESS, "");
            } catch (RuntimeException ex) {
                errorMessage = format("Failed to %s: %s. Details: %s", requestType, roleSubject, ex.getMessage());
                LOG.error(errorMessage, ex);
            }

        } catch (IOException e) {
            errorMessage = "Cannot parse role: " + jsonEncodedRole;
            LOG.error(errorMessage, e);
        }

        incrementFailedRequestsMetric(requestType);
        return new IdmServiceResponse(RESPONSE_CODE_ERROR, errorMessage);
    }

    public IdmServiceResponse processMemberships(String jsonEncodedMemberships, RequestType requestType) {
        String memberships = URLDecoder.decode(jsonEncodedMemberships, StandardCharsets.UTF_8);
        ObjectMapper mapper = new ObjectMapper();
        TypeFactory typeFactory = mapper.getTypeFactory();

        List<IdmMembership> idmMembershipList;
        try {
            idmMembershipList = mapper.readValue(memberships,
                    typeFactory.constructCollectionType(List.class, IdmMembership.class));
        } catch (IOException e) {
            incrementFailedRequestsMetric(requestType);
            return new IdmServiceResponse(RESPONSE_CODE_ERROR, "Cannot parse role: " + jsonEncodedMemberships);
        }

        Long groupId = idmMembershipList.size() > 0 ? idmMembershipList.get(0).getGroupId() : 0L;
        Set<String> logins = idmMembershipList.stream()
                .map(IdmMembership::getLogin)
                .collect(Collectors.toSet());

        String errorMessage;
        try {
            if (groupId != 0L) {
                final String groupIdWithPrefix = YpGroupsHelper.IDM_GROUP_PREFIX + groupId;
                if (requestType == RequestType.ADD_BATCH_MEMBERSHIPS) {
                    ypService.addMembersToGroup(groupIdWithPrefix, logins);
                } else {
                    ypService.removeMembersFromGroup(groupIdWithPrefix, logins);
                }
                LOG.info("Succeeded to {}: {} {}", requestType, groupId, logins);
                return new IdmServiceResponse(RESPONSE_CODE_SUCCESS, "");
            }

            errorMessage = format("Failed to %s: %s %s", requestType, groupId, logins);
        } catch (YpServiceException ex){
            errorMessage = format("Failed to %s: %s %s. Details: %s",
                    requestType, groupId, logins, ex.getMessage());
        }
        LOG.error(errorMessage);
        incrementFailedRequestsMetric(requestType);
        return new IdmServiceResponse(RESPONSE_CODE_ERROR, errorMessage);
    }

    private StringNode collectToTree(Collection<Role> roleList) {
        StringNode root = new StringNode(Role.getIDMKeyNodeName(false, false, 0, ""), IDM_ROOT_ROLE_NAME);
        for (Role role : roleList) {
            if (role.isRoot()) {
                // skip root node because it's added by default
                continue;
            }
            final StringBuilder currentLevelName = new StringBuilder();
            StringNode currentNode = root;
            StringRoles stringRoles = null;

            for (int level = 0; level < role.size(); level++) {
                String levelName = role.getLevelName(level).orElse("");
                if (levelName.isEmpty()) {
                    continue;
                }
                if (level != 0) {
                    currentLevelName.append(".");
                }
                currentLevelName.append(levelName);
                int levelPlus1 = level + 1;
                stringRoles = currentNode.values.computeIfAbsent(
                        levelName,
                        ignored -> new StringRoles(levelName,
                                new StringNode(Role.getIDMKeyNodeName(
                                                    role.isNannyRole(),
                                                    role.getLevelName(0).orElse("").equals(Role.NANNY_ROLES_PARENT_NODE),
                                                    levelPlus1,
                                                    levelName),
                                        currentLevelName.toString())));
                currentNode = stringRoles.root;
            }

            if (stringRoles != null) {
                stringRoles.uniqueId = role.getUniqueId();
            }
        }
        return root;
    }

    private IdmRoleNode convertTree(StringNode roleTree) {
        Map<String, Object> values = new TreeMap<>();

        for (Map.Entry<String, StringRoles> entry : roleTree.values.entrySet()) {
            Optional<IdmLeaf> description = rolesInfo.getRoleDescription(entry.getKey());

            final StringRoles stringRoles = entry.getValue();
            if (description.isEmpty()) {
                values.put(entry.getKey(), new IdmSubRoles(stringRoles.name, convertTree(stringRoles.root), stringRoles.uniqueId));
            } else {
                values.put(entry.getKey(), description.get().withUniqueId(stringRoles.uniqueId));
            }
        }
        return new IdmRoleNode(roleTree.slug, roleTree.name, values);
    }

    static final class StringNode {
        final String slug;
        final String name;
        Map<String, StringRoles> values;

        StringNode(String name, String slug) {
            this.name = name;
            this.slug = slug;
            values = new HashMap<>();
        }
    }

    static final class StringRoles {
        final String name;
        final StringNode root;
        String uniqueId;

        StringRoles(String name, StringNode root) {
            this.name = name;
            this.root = root;
        }
    }
}

