package ru.yandex.infra.auth.nanny;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.typesafe.config.Config;
import org.apache.http.HttpStatus;
import org.asynchttpclient.AsyncHttpClient;

import ru.yandex.infra.auth.utils.ApiClientWithAuthorization;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;

import static ru.yandex.infra.controller.util.ConfigUtils.token;

public class NannyApiImpl extends ApiClientWithAuthorization implements NannyApi {

    static final String METRIC_GET_SERVICE = "get_service";
    static final String METRIC_GET_SERVICES_RANGE = "get_services_range";
    static final String METRIC_GET_ALL_SERVICES = "get_all_services";
    static final String METRIC_UPDATE_SERVICE = "update_service";

    private final AtomicLong metricGetService = new AtomicLong();
    private final AtomicLong metricGetServicesRange = new AtomicLong();
    private final AtomicLong metricGetAllServices = new AtomicLong();
    private final AtomicLong metricUpdateService = new AtomicLong();

    public NannyApiImpl(Config config, AsyncHttpClient httpClient, GaugeRegistry gaugeRegistry) {
        super(httpClient, config.getString("host"), token(config.getString("token_file")), gaugeRegistry);

        gaugeRegistry.add("api." + METRIC_GET_SERVICE, new GolovanableGauge<>(metricGetService::get, "dmmm"));
        gaugeRegistry.add("api." + METRIC_GET_SERVICES_RANGE, new GolovanableGauge<>(metricGetServicesRange::get, "dmmm"));
        gaugeRegistry.add("api." + METRIC_GET_ALL_SERVICES, new GolovanableGauge<>(metricGetAllServices::get, "dmmm"));
        gaugeRegistry.add("api." + METRIC_UPDATE_SERVICE, new GolovanableGauge<>(metricUpdateService::get, "dmmm"));
    }

    @Override
    public CompletableFuture<NannyServiceInfo> getService(String serviceId) {
        String url = String.format("%s/v2/services/%s/auth_attrs/", host, serviceId);
        metricGetService.incrementAndGet();
        return executeRequest(url, root -> NannyServiceInfo.fromAuthAttrsNode(serviceId, root));
    }

    @Override
    public CompletableFuture<NannyServiceInfo> getService(String serviceId, int maxAttemptsCount, Duration retryDelay) {
        return executeRequestWithRetries(() -> getService(serviceId), maxAttemptsCount, retryDelay, Set.of(HttpStatus.SC_NOT_FOUND));
    }

    @Override
    public CompletableFuture<List<NannyServiceInfo>> getAllServices(int offset, int count) {
        String url = String.format("%s/v2/services/?skip=%d&limit=%d", host, offset, count);

        metricGetServicesRange.incrementAndGet();
        return executeRequest(url, root -> {
            List<NannyServiceInfo> allServices = new ArrayList<>();
            root.get("result").forEach(serviceNode -> allServices.add(NannyServiceInfo.fromJsonNode(serviceNode)));
            return allServices;
        });
    }

    @Override
    public CompletableFuture<List<NannyServiceInfo>> getAllServices(int offset, int count, int maxAttemptsCount, Duration retryDelay) {
        return executeRequestWithRetries(() -> getAllServices(offset, count), maxAttemptsCount, retryDelay, Collections.emptySet());
    }

    @Override
    public CompletableFuture<Set<String>> getAllServices() {
        String url = String.format("%s/api/repo/ListSummaries/", host);

        metricGetAllServices.incrementAndGet();
        return executeRequest(url, root -> {
            Set<String> allServices = new HashSet<>();
            root.get("value").forEach(serviceNode -> allServices.add(serviceNode.get("serviceId").textValue()));
            return allServices;
        });
    }

    @Override
    public CompletableFuture<Set<String>> getAllServices(int maxAttemptsCount, Duration retryDelay) {
        return executeRequestWithRetries(this::getAllServices, maxAttemptsCount, retryDelay, Collections.emptySet());
    }

    @Override
    public CompletableFuture<NannyServiceInfo> updateService(NannyServiceInfo service) {
        String url = String.format("%s/v2/services/%s/auth_attrs/", host, service.getName());

        String json;
        try {
            json = mapper.writeValueAsString(service);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error while convert requests to json", e);
        }

        metricUpdateService.incrementAndGet();
        return runWithAuthorization(httpClient.preparePut(url).setBody(json), response -> {
            try {
                if (response.getStatusCode() != HttpStatus.SC_OK) {
                    throw new RuntimeException(String.format("Error response from Nanny API request: %s\n[%d] %s\n%s",
                            url,
                            response.getStatusCode(),
                            response.getStatusText(),
                            response.getResponseBody()));
                }

                JsonNode root = mapper.readTree(response.getResponseBodyAsBytes());
                return NannyServiceInfo.fromAuthAttrsNode(service.getName(), root);
            } catch (IOException e) {
                throw new RuntimeException("Error while parsing response from nanny", e);
            }
        });

    }

}
