package ru.yandex.infra.auth.nanny;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.infra.auth.yp.YpClients;
import ru.yandex.infra.auth.yp.YpGroup;
import ru.yandex.infra.auth.yp.YpGroupsClient;

import static org.slf4j.LoggerFactory.getLogger;

public class NannyGroupsUpdaterForSingleIteration {
    private static final Logger LOG = getLogger(NannyGroupsUpdaterForSingleIteration.class);

    static final String NANNY_SERVICE_LABEL = "service";
    static final String NANNY_ROLE_LABEL = "role";

    private final String cluster;
    private final YpGroupsClient ypGroupClient;
    private final Map<String,String> nannyGroupLabels;
    private final Map<String, String> staffToYpGroupMap;
    private final Set<String> ypUsers;

    private final Map<Tuple2<String, String>, YpGroup> nannyGroupByServiceAndRole = new HashMap<>();

    public NannyGroupsUpdaterForSingleIteration(String cluster,
                                                Map<String,String> nannyGroupLabels,
                                                YpClients ypClients,
                                                Map<String, String> staffToYpGroup,
                                                Set<String> ypUsers) {
        this.cluster = cluster;
        this.nannyGroupLabels = nannyGroupLabels;
        this.ypGroupClient = ypClients.getGroupsClient();
        this.staffToYpGroupMap = staffToYpGroup;
        this.ypUsers = ypUsers;
    }

    public String getCluster() {
        return cluster;
    }

    public void syncNannyService(NannyServiceInfo service) {
        service.getAuthGroups().forEach((roleName, nannyRole) -> {
            //nannyRole is "owner", "configuration_managers" etc
            //all of them have 1:1 matching with some yp.group in each cluster (where pod_set is deployed)
            YpGroup group = nannyGroupByServiceAndRole.get(Tuple2.tuple(service.getName(), roleName));
            if (group != null) {
                syncGroup(group, nannyRole);
            }
        });
    }

    private void syncGroup(YpGroup group, NannyAuthGroup nannyRole) {
        LOG.debug("[{}] Syncing nanny auth group {}", cluster, group.getId());

        Stream<String> validUsers = nannyRole.getUsers()
                .stream()
                .filter(ypUsers::contains);

        Stream<String> validGroups = nannyRole.getStaffGroups()
                .stream()
                .map(staffToYpGroupMap::get)
                .filter(Objects::nonNull);

        Set<String> membersInNannyAuthAttributes = Stream.concat(validUsers, validGroups).collect(Collectors.toSet());
        Set<String> ypGroupMembers = group.getMembers();
        if (!membersInNannyAuthAttributes.equals(ypGroupMembers)) {
            try {
                NannyRolesSynchronizer.metricGroupUpdates.incrementAndGet();
                LOG.info("[{}] Updating nanny auth group {}", cluster, group.getId());
                ypGroupClient.updateMembers(group.getId(), membersInNannyAuthAttributes).get();
                NannyRolesSynchronizer.metricSucceededGroupUpdates.incrementAndGet();
                LOG.info("[{}] Updated nanny auth group {} with members: {}", cluster, group.getId(), membersInNannyAuthAttributes);
            } catch (Exception error) {
                NannyRolesSynchronizer.metricFailedGroupUpdates.incrementAndGet();
                LOG.error(String.format("[%s] Failed to update nanny auth group %s", cluster, group.getId()), error);
            }
        }
    }

    public CompletableFuture<Map<Tuple2<String, String>, YpGroup>> loadNannyGroups() {
        return loadAllGroups(nannyGroupLabels)
                .thenAccept(groups -> groups.values().forEach(group -> {
                    String service = group.getLabels().get(NANNY_SERVICE_LABEL);
                    String role = group.getLabels().get(NANNY_ROLE_LABEL);
                    if (service == null || role == null) {
                        LOG.warn("[{}] Skipping group {} due to wrong labels: {}", cluster, group.getId(), group.getLabels());
                        return;
                    }
                    nannyGroupByServiceAndRole.put(Tuple2.tuple(service, role), group);
                }))
                .thenApply(x -> nannyGroupByServiceAndRole);
    }

    private CompletableFuture<Map<String, YpGroup>> loadAllGroups(Map<String,String> filterLabels) {
        long startTimeMillis = System.currentTimeMillis();
        return ypGroupClient.getGroupsByLabels(filterLabels, Optional.empty())
                .whenComplete((groups, error) -> {
                    if (error != null) {
                        LOG.error("[{}}] Failed to load YP groups with labels {}}: {}", cluster, filterLabels, error.getMessage());
                    } else {
                        LOG.debug("[{}] Loaded {} YP groups with labels {} in {} ms", cluster, groups.size(), filterLabels, System.currentTimeMillis() - startTimeMillis);
                    }
                });
    }
}
