package ru.yandex.infra.auth.nanny;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import com.typesafe.config.Config;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.infra.auth.GroupsAndUsersCache;
import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubject;
import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.auth.idm.service.IdmService;
import ru.yandex.infra.auth.yp.MissedYpObjectException;
import ru.yandex.infra.auth.yp.YpClients;
import ru.yandex.infra.auth.yp.YpGroupsHelper;
import ru.yandex.infra.auth.yp.YpService;
import ru.yandex.infra.auth.yp.YpServiceReadOnlyImpl;
import ru.yandex.infra.controller.RepeatedTask;
import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.util.ExceptionUtils;

import static java.util.Collections.emptySet;
import static org.slf4j.LoggerFactory.getLogger;

public class NannyRolesSynchronizer implements NannyService {
    private static final Logger LOG = getLogger(NannyRolesSynchronizer.class);

    static final String METRIC_YP_OBJECTS_LOAD_TIME_MS = "yp_objects_load_time_ms";
    static final String METRIC_GROUP_UPDATES = "group_updates";
    static final String METRIC_SUCCEEDED_GROUP_UPDATES = "succeeded_group_updates";
    static final String METRIC_FAILED_GROUP_UPDATES = "failed_group_updates";
    static final String METRIC_NANNY_SERVICES = "nanny_services";
    static final String METRIC_CURRENT_NANNY_SERVICE_OFFSET = "current_nanny_service_offset";
    static final String METRIC_NANNY_GROUPS_COUNT = "nanny_groups_count";
    static final String METRIC_SYNC_QUEUE_SIZE = "sync_queue_size";
    static final String METRIC_SYNC_IN_PROGRESS_QUEUE_SIZE = "sync_in_progress_queue_size";
    static final String METRIC_SYNC_QUEUE_EXCEPTIONS = "sync_queue_exceptions";
    static final String METRIC_CURRENT_NANNY_SERVICES_SYNC_ITERATION_DURATION_TIME = "current_nanny_services_sync_iteration_duration_ms";
    static final String METRIC_FINISHED_NANNY_SERVICES_SYNC_ITERATIONS_COUNT = "finished_nanny_services_sync_iterations";

    static final String METRIC_AUTH_ATTRS_UPDATES = "auth_attrs_updates";
    static final String METRIC_FAILED_AUTH_ATTRS_UPDATES = "failed_auth_attrs_updates";

    private volatile Integer metricCurrentNannyServiceOffset;
    private volatile Long metricLastYpObjectsLoadTimeMilliseconds;
    private volatile Integer metricNannyServices;
    static final AtomicLong metricGroupUpdates = new AtomicLong();
    static final AtomicLong metricSucceededGroupUpdates = new AtomicLong();
    static final AtomicLong metricFailedGroupUpdates = new AtomicLong();
    private final AtomicInteger metricSyncInProgressQueueSize = new AtomicInteger();
    private final AtomicLong metricSyncQueueExceptions = new AtomicLong();
    private final AtomicLong metricAuthAttrsUpdates = new AtomicLong();
    private final AtomicLong metricFailedAuthAttrsUpdates = new AtomicLong();
    private Map<Tuple2<String,String>, Integer> metricsPerCluster = new HashMap<>();
    private long lastNannyServicesSyncIterationStartTimeMillis;
    private final AtomicLong metricFinishedNannyServicesSyncIterationsCount = new AtomicLong();

    private final LeaderService leaderService;
    private final NannyApi nannyApi;
    private final YpService ypService;
    private final Map<String, YpClients> ypClients;
    private final GroupsAndUsersCache groupsAndUsersCache;
    private final int batchSize;
    private final int httpMaxRetries;
    private final Duration httpRetryDelay;
    private final Duration ypObjectsLoadTimeout;
    private final boolean syncAuthAttrsToPodsetYpGroups;
    private final boolean syncAuthAttrs;
    private final boolean syncIdmNodes;
    private final boolean processNannyServicesWithoutProject;
    private final Map<String,String> nannyGroupLabels;
    private final ScheduledExecutorService executor;
    private final ScheduledExecutorService syncQueueExecutor;
    private final Duration cycleInterval;
    private final RepeatedTask syncTask;
    private final RolesInfo rolesInfo;
    private final Set<String> nannyServiceRoleNames;

    private final BlockingQueue<Tuple2<String, String>> nannyServicesSyncQueue = new LinkedBlockingQueue<>();

    public NannyRolesSynchronizer(Config config,
                                  NannyApi nannyApi,
                                  YpService ypService,
                                  GroupsAndUsersCache groupsAndUsersCache,
                                  LeaderService leaderService,
                                  GaugeRegistry gaugeRegistry,
                                  RolesInfo rolesInfo) {
        this.nannyApi = nannyApi;
        this.leaderService = leaderService;
        this.ypService = ypService;
        this.ypClients = ypService.getSlaveClusterClients();
        this.groupsAndUsersCache = groupsAndUsersCache;
        this.rolesInfo = rolesInfo;
        this.nannyGroupLabels = ImmutableMap.of("system", config.getString("yp_group_system_label"));
        this.nannyServiceRoleNames = rolesInfo.getRolesPerLevel(RolesInfo.LevelName.NANNY_SERVICE);

        this.batchSize = config.getInt("batch_size");
        this.cycleInterval = config.getDuration("update_interval");
        this.ypObjectsLoadTimeout = config.getDuration("yp_objects_load_timeout");
        this.syncAuthAttrsToPodsetYpGroups = config.getBoolean("sync_auth_attrs_to_podset_yp_groups");
        this.syncAuthAttrs = config.getBoolean("sync_auth_attrs");
        this.syncIdmNodes = config.getBoolean("sync_idm_nodes");
        this.processNannyServicesWithoutProject = config.getBoolean("process_services_without_project");

        if (!syncAuthAttrsToPodsetYpGroups) {
            LOG.warn("Synchronization between Nanny service auth_attrs and pod_set YP groups is disabled by config option nanny.sync_auth_attrs_to_podset_yp_groups = false");
        }

        Config httpConfig = config.getConfig("http");
        this.httpMaxRetries = httpConfig.getInt("max_retries");
        this.httpRetryDelay = httpConfig.getDuration("retry_delay");

        this.executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "nanny"));
        this.syncQueueExecutor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "nanny_sync"));

        GaugeRegistry nannyRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "nanny");
        nannyRegistry.add(METRIC_YP_OBJECTS_LOAD_TIME_MS, new GolovanableGauge<>(() -> metricLastYpObjectsLoadTimeMilliseconds, "axxx"));
        nannyRegistry.add(METRIC_GROUP_UPDATES, new GolovanableGauge<>(metricGroupUpdates::get, "dmmm"));
        nannyRegistry.add(METRIC_SUCCEEDED_GROUP_UPDATES, new GolovanableGauge<>(metricSucceededGroupUpdates::get, "dmmm"));
        nannyRegistry.add(METRIC_FAILED_GROUP_UPDATES, new GolovanableGauge<>(metricFailedGroupUpdates::get, "dmmm"));
        nannyRegistry.add(METRIC_NANNY_SERVICES, new GolovanableGauge<>(() -> metricNannyServices, "axxx"));
        nannyRegistry.add(METRIC_CURRENT_NANNY_SERVICE_OFFSET, new GolovanableGauge<>(() -> metricCurrentNannyServiceOffset, "axxx"));
        nannyRegistry.add(METRIC_SYNC_QUEUE_SIZE, new GolovanableGauge<>(nannyServicesSyncQueue::size, "axxx"));
        nannyRegistry.add(METRIC_SYNC_IN_PROGRESS_QUEUE_SIZE, new GolovanableGauge<>(metricSyncInProgressQueueSize::get, "axxx"));
        nannyRegistry.add(METRIC_SYNC_QUEUE_EXCEPTIONS, new GolovanableGauge<>(metricSyncQueueExceptions::get, "dmmm"));
        nannyRegistry.add(METRIC_AUTH_ATTRS_UPDATES, new GolovanableGauge<>(metricAuthAttrsUpdates::get, "dmmm"));
        nannyRegistry.add(METRIC_FAILED_AUTH_ATTRS_UPDATES, new GolovanableGauge<>(metricFailedAuthAttrsUpdates::get, "dmmm"));
        nannyRegistry.add(METRIC_CURRENT_NANNY_SERVICES_SYNC_ITERATION_DURATION_TIME,
                new GolovanableGauge<>(() -> lastNannyServicesSyncIterationStartTimeMillis != 0 ?
                        System.currentTimeMillis() - lastNannyServicesSyncIterationStartTimeMillis : null, "axxx"));
        nannyRegistry.add(METRIC_FINISHED_NANNY_SERVICES_SYNC_ITERATIONS_COUNT,
                new GolovanableGauge<>(metricFinishedNannyServicesSyncIterationsCount::get, "dmmm"));

        GaugeRegistry ypRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "yp");

        ypClients.keySet().forEach(cluster -> {
            addMetricPerCluster(ypRegistry, cluster, METRIC_NANNY_GROUPS_COUNT);
        });

        syncTask = new RepeatedTask(this::mainLoop,
                cycleInterval,
                config.getDuration("main_loop_timeout"),
                executor,
                Optional.of(nannyRegistry),
                LOG,
                false);
    }

    private void addMetricPerCluster(GaugeRegistry registry, String cluster, String metric) {
        registry.add(cluster + "." + metric, new GolovanableGauge<>(() -> getLastCycleMetrics().get(Tuple2.tuple(cluster, metric)), "axxx"));
    }

    private Map<Tuple2<String,String>, Integer> getLastCycleMetrics() {
        return metricsPerCluster;
    }

    public void start() {
        if (cycleInterval.isZero()) {
            LOG.warn("Nanny sync is disabled by config option 'nanny.update_interval' = 0");
        } else {
            syncTask.start();
        }

        syncQueueExecutor.submit(this::nannyServiceAuthAttrsSyncMain);
    }

    public void shutdown() {
        syncTask.stop();
        executor.shutdown();
        syncQueueExecutor.shutdown();
    }

    @VisibleForTesting
    CompletableFuture<?> mainLoop() {

        if (groupsAndUsersCache.getLastSnapshotTimestampMillis() == null) {
            LOG.warn("Staff groups was not loaded yet");
            return CompletableFuture.completedFuture(null);
        }

        if (!leaderService.isLeader()) {
            //Will update nanny groups only from leader instance
            return CompletableFuture.completedFuture(null);
        }

        return getNannyGroupUpdatersPerCluster()
                .thenApplyAsync(this::reloadAllNannyServices, executor);
    }

    private CompletableFuture<List<NannyGroupsUpdaterForSingleIteration>> getNannyGroupUpdatersPerCluster() {

        if (!syncAuthAttrsToPodsetYpGroups) {
            return CompletableFuture.completedFuture(Collections.emptyList());
        }

        long startTimeMillis = System.currentTimeMillis();
        Map<Tuple2<String,String>, Integer> metricsForCurrentCycle = new HashMap<>();

        //Loading all clusters objects in parallel
        Map<NannyGroupsUpdaterForSingleIteration, CompletableFuture<Void>> allUpdaters = ypClients.entrySet().stream()
                .map(entry -> new NannyGroupsUpdaterForSingleIteration(
                        entry.getKey(),
                        nannyGroupLabels,
                        entry.getValue(),
                        groupsAndUsersCache.getStaffToYpGroupMap(),
                        groupsAndUsersCache.getYpUsers()))
                .collect(Collectors.toMap(updater -> updater, updater -> updater.loadNannyGroups()
                        .orTimeout(ypObjectsLoadTimeout.toMillis(), TimeUnit.MILLISECONDS)
                        .thenAcceptAsync(groups -> metricsForCurrentCycle.put(Tuple2.tuple(updater.getCluster(), METRIC_NANNY_GROUPS_COUNT), groups.size()), executor)
                ));

        return CompletableFuture.allOf(allUpdaters.values().toArray(CompletableFuture<?>[]::new))
                .exceptionally(error -> null) //Some clusters could fail, will ignore them and continue with the rest
                .thenApplyAsync(x -> {
                    metricLastYpObjectsLoadTimeMilliseconds = System.currentTimeMillis() - startTimeMillis;
                    LOG.debug("Loaded all YP/Staff objects in {} ms", metricLastYpObjectsLoadTimeMilliseconds);
                    metricsPerCluster = metricsForCurrentCycle;

                    List<NannyGroupsUpdaterForSingleIteration> result = allUpdaters
                            .entrySet()
                            .stream()
                            .filter(entry -> !entry.getValue().isCompletedExceptionally())
                            .map(Map.Entry::getKey)
                            .collect(Collectors.toList());
                    if (result.isEmpty()) {
                        LOG.warn("Available clusters list for nanny groups update is empty");
                    }
                    return result;
                }, executor);
    }

    private Map<String, NannyServiceInfo> reloadAllNannyServices(List<NannyGroupsUpdaterForSingleIteration> updatersPerCluster) {
        int offset = 0;
        metricCurrentNannyServiceOffset = 0;
        int lastBatchCount;
        Map<String, NannyServiceInfo> allNannyServices = new HashMap<>();

        do {
            LOG.debug("Loading nanny services from offset {}", offset);

            List<NannyServiceInfo> services;
            try {
                services = nannyApi.getAllServices(offset, batchSize, httpMaxRetries, httpRetryDelay).get();
            } catch (ExecutionException|InterruptedException e) {
                throw new RuntimeException(String.format("Failed to load nanny services after %d attempts", httpMaxRetries));
            }

            lastBatchCount = services.size();
            LOG.debug("Loaded {} services from offset {}", lastBatchCount, offset);
            offset += lastBatchCount;
            metricCurrentNannyServiceOffset = offset;

            services.forEach(service -> {
                allNannyServices.put(service.getName(), service);
                updatersPerCluster.forEach(updater -> updater.syncNannyService(service));
            });

        } while (lastBatchCount != 0);

        metricNannyServices = allNannyServices.size();
        return allNannyServices;
    }

    @Override
    public Set<NannyRole> getRolesWithSubjects(String projectId, String serviceName, String serviceUuid) {

        if (groupsAndUsersCache.getLastSnapshotTimestampMillis() == null) {
            throw new RuntimeException("Can't load initial nanny service roles. Staff groups was not loaded yet.", null);
        }

        NannyServiceInfo serviceInfo;
        try {
            serviceInfo = nannyApi.getService(serviceName, httpMaxRetries, httpRetryDelay).get();
            LOG.info("Downloaded initial nanny service auth_attrs snapshot: {}", serviceInfo);
        } catch (ExecutionException|InterruptedException e) {
            throw new RuntimeException(e);
        }

        final String rolePath = Role.getRolePathForNannyService(projectId, serviceName);

        Set<NannyRole> roles = new HashSet<>();
        Map<String, NannyAuthGroup> authGroups = serviceInfo.getAuthGroups();

        final Map<String, Long> departmentUrlToStaffGroup = groupsAndUsersCache.getDepartmentUrlToStaffGroupIdMap();
        final Map<String, String> staffToYpGroupMap = groupsAndUsersCache.getStaffToYpGroupMap();
        authGroups.forEach((nannyAuthRoleName, nannyAuthGroup) -> {
            Set<Long> groups = nannyAuthGroup.getStaffGroups()
                    .stream()
                    .map(idString -> {
                        try {
                            return Long.parseLong(idString);
                        } catch (NumberFormatException e) {
                            return departmentUrlToStaffGroup.get(idString);
                        }
                    })
                    .filter(groupId -> groupId != null && staffToYpGroupMap.containsKey(groupId.toString()))
                    .collect(Collectors.toSet());
            Set<String> users = nannyAuthGroup.getUsers()
                    .stream()
                    .filter(user -> groupsAndUsersCache.getYpUsers().contains(user))
                    .collect(Collectors.toSet());
            NannyRole role = new NannyRole(rolePath,
                    nannyAuthRoleName,
                    Role.getLeafUniqueId(serviceUuid, nannyAuthRoleName),
                    users,
                    groups);
            roles.add(role);
        });

        return roles;
    }

    @Override
    public void syncNannyServiceRoleIntoYP(NannyRole nannyRole) {

        nannyRole.getGroups().forEach(staffGroupId -> {
            //Creation of groups like "idm:234234"
            //i.e. role was granted to staff group ("idm:234234"), but IDM haven't pushed to AuthCtl yet
            //Todo: DEPLOY-5095
            ypService.addMembersToGroup(YpGroupsHelper.IDM_GROUP_PREFIX + staffGroupId, emptySet());
        });
        String groupId = YpServiceReadOnlyImpl.getYpGroupName(nannyRole, ypService.getSystemName());

        Set<String> usersAndGroups = Stream.concat(
                nannyRole.getLogins()
                        .stream(),
                nannyRole.getGroups()
                        .stream()
                        .map(staffGroupId -> YpGroupsHelper.IDM_GROUP_PREFIX + staffGroupId)
        ).collect(Collectors.toSet());
        //if we have any user or group for that role
        usersAndGroups.stream().findAny().ifPresent(subject -> {
            RoleSubject roleSubject;
            if (subject.startsWith(YpGroupsHelper.IDM_GROUP_PREFIX)) {
                roleSubject = new RoleSubject("", Long.parseLong(subject.substring(YpGroupsHelper.IDM_GROUP_PREFIX.length())), nannyRole);
            } else {
                roleSubject = new RoleSubject(subject, 0L, nannyRole);
            }

            //updating yp.nanny_service.meta.acl
            ypService.addRoleSubject(roleSubject);
            if (usersAndGroups.size() > 1) {
                //updating group with all other users/groups. No reason to call 'ypService.addRoleSubject' many times
                ypService.addMembersToGroup(groupId, usersAndGroups);
            }
        });

    }

    @Override
    public void syncNannyServiceAuthAttrsAsync(String serviceId, String commitMessage) {
        nannyServicesSyncQueue.add(Tuple2.tuple(serviceId, commitMessage));
    }

    @Override
    public boolean isNodesSyncEnabled() {
        return syncIdmNodes;
    }

    @Override
    public void updateRoleSubject(RoleSubject roleSubject, IdmService.RequestType requestType) {

        final Role role = roleSubject.getRole();
        final RolesInfo.LevelName leafLevel = role.getLeafLevel();

        String commitComment = String.format("IDM %s %s", requestType, roleSubject);

        if (leafLevel == RolesInfo.LevelName.NANNY_SERVICE) {
            try {
                String projectIdFromRole = role.getProjectId().orElse(null);
                String serviceId = role.getNannyServiceId().orElseThrow();
                String projectIdFromNannyService = ypService.getProjectIdFor(serviceId);
                if (!Objects.equals(projectIdFromRole, projectIdFromNannyService)) {
                    LOG.warn("Project mismatch: {}.project_id = {} in YP, but requested {} {}", serviceId, projectIdFromNannyService, requestType, role);
                    return;
                }
                //sync update. IDM will retry in case of crash.
                syncNannyServiceAuthAttrs(serviceId, commitComment);
            } catch (MissedYpObjectException exception) {
                if (requestType == IdmService.RequestType.ADD_ROLE) {
                    throw exception;
                }
                LOG.warn("Nanny service was removed from YP, skipping {} for {}", requestType, roleSubject);
            }
        } else if (leafLevel == RolesInfo.LevelName.PROJECT) {
            String projectId = role.getProjectId().orElseThrow();
            List<String> services = ypService.getNannyServices(projectId);
            //async update. Could be too many services in the single project.
            services.forEach(service -> nannyServicesSyncQueue.add(Tuple2.tuple(service, commitComment)));
        }
    }

    private void nannyServiceAuthAttrsSyncMain() {
        while (!syncQueueExecutor.isShutdown()) {
            try {
                Tuple2<String, String> firstItem = nannyServicesSyncQueue.take();
                lastNannyServicesSyncIterationStartTimeMillis = System.currentTimeMillis();
                List<Tuple2<String, String>> allQueueItems = new ArrayList<>();
                nannyServicesSyncQueue.drainTo(allQueueItems);
                allQueueItems.add(firstItem);

                //Join all commit messages per each nanny service
                Map<String, String> servicesWithComment = allQueueItems.stream()
                        .collect(Collectors.groupingBy(t -> t._1,
                                Collectors.collectingAndThen(Collectors.toList(),
                                        list -> list.stream().map(t -> t._2).collect(Collectors.joining("\n")))));

                metricSyncInProgressQueueSize.set(servicesWithComment.size());
                servicesWithComment.forEach((serviceId, commitComment) -> {
                    try {
                        syncNannyServiceAuthAttrs(serviceId, commitComment);
                    } catch (Exception exception) {
                        LOG.error("Failed to sync nanny service '{}' auth_attrs", serviceId, exception);
                    }
                    metricSyncInProgressQueueSize.decrementAndGet();
                });
            } catch (Throwable throwable) {
                metricSyncQueueExceptions.incrementAndGet();
                LOG.error("Exception in nanny service auth_attrs sync thread", throwable);
            }
            metricFinishedNannyServicesSyncIterationsCount.incrementAndGet();
            lastNannyServicesSyncIterationStartTimeMillis = 0;
        }
    }

    private void syncNannyServiceAuthAttrs(String serviceId, String commitComment) {

        String projectId = ypService.getProjectIdFor(serviceId);

        if (projectId == null && !processNannyServicesWithoutProject) {
            LOG.warn("Nanny service '{}' has empty /meta/project_id. Skipping sync for: {}", serviceId, commitComment);
            return;
        }

        Map<String, Set<String>> projectRoles = projectId != null ?
                ypService.getRoleMembers(RolesInfo.LevelName.PROJECT, projectId) :
                Collections.emptyMap();
        Map<String, Set<String>> serviceRoles = ypService.getRoleMembers(RolesInfo.LevelName.NANNY_SERVICE, serviceId);
        Map<String, Set<String>> roleMembers = mergeRoles(projectRoles, serviceRoles);

        try {
            NannyServiceInfo serviceInfo = nannyApi.getService(serviceId).get();
            LOG.info("Received nanny service info: {}", serviceInfo);

            Map<String, NannyAuthGroup> newAuthGroups = getAuthGroups(roleMembers);
            if (serviceInfo.getAuthGroups().equals(newAuthGroups)) {
                LOG.info("'{}' auth_attrs was already synced. Skipping update: {}", serviceId, commitComment);
                return;
            }

            LOG.info("Updating nanny service '{}' auth_attrs (current snapshot = {}): {}", serviceId, serviceInfo.getSnapshotId(), commitComment);
            metricAuthAttrsUpdates.incrementAndGet();
            if (syncAuthAttrs) {
                NannyServiceInfo updatedServiceInfo = nannyApi.updateService(serviceInfo.withAuthGroups(newAuthGroups, commitComment)).get();
                LOG.info("Successfully updated nanny service '{}' (snapshot: {} -> {}) with auth_attrs: {}",
                        serviceId, serviceInfo.getSnapshotId(), updatedServiceInfo.getSnapshotId(), roleMembers);
            } else {
                LOG.warn("Writing into Nanny auth_attrs is disabled by config option 'nanny.sync_auth_attrs' = false; Skipping update of '{}'", serviceId);
            }
        } catch (InterruptedException|ExecutionException exception) {
            metricFailedAuthAttrsUpdates.incrementAndGet();
            if (ExceptionUtils.tryExtractHttpErrorCode(exception).orElse(-1) == HttpStatus.SC_NOT_FOUND) {
                LOG.warn("Nanny service '{}' was not found in Nanny. Skipping sync for: {}", serviceId, commitComment);
                return;
            }
            throw new RuntimeException("Failed to update nanny service '" + serviceId + "' auth_attrs for: " + commitComment, exception);
        }
    }

    @VisibleForTesting
    Map<String, Set<String>> mergeRoles(Map<String, Set<String>> projectRoles,
                                        Map<String, Set<String>> serviceRoles) {
        projectRoles.forEach((projectRole, projectRoleMembers) -> {
            rolesInfo.getMappedRoles(projectRole).stream()
                    .filter(nannyServiceRoleNames::contains)
                    .forEach(nannyRole -> {
                        Set<String> members = serviceRoles.getOrDefault(nannyRole, new HashSet<>());
                        members.addAll(projectRoleMembers);
                        serviceRoles.put(nannyRole, members);
                    });
        });
        return serviceRoles;
    }

    private Map<String, NannyAuthGroup> getAuthGroups(Map<String, Set<String>> roleMembers) {
        //insert missed nanny service roles with empty list of members
        Sets.difference(nannyServiceRoleNames, roleMembers.keySet())
                .forEach(missedRole -> roleMembers.put(missedRole, Collections.emptySet()));

        return roleMembers.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> {
                    Set<String> users = new HashSet<>();
                    Set<String> groups = new HashSet<>();
                    e.getValue().forEach(subject -> {
                        if (subject.startsWith(YpGroupsHelper.IDM_GROUP_PREFIX)) {
                            groups.add(subject.substring(YpGroupsHelper.IDM_GROUP_PREFIX.length()));
                        } else {
                            users.add(subject);
                        }
                    });
                    return new NannyAuthGroup(users, groups);
                }));
    }

}
